------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                               BODY_MODELS                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;

--  Model of bodies currently selected for some operations (eg display,
--  primary significators or promissors).
package body Body_Models is

   --  Create and initialize a body iterator
   function Initialize (M : Body_Model) return Body_Iterator is
      Result : Body_Iterator;
   begin
      --  Copy model fields so that multiple iterators act independently
      --  of each other. Note relatively insignificant memory leak for
      --  later correction
      Result.Model := new Body_Model;

      Result.Model.Extended := M.Extended;
      Result.Model.Standard := M.Standard;

      return Result;
   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   --  Compiler uses 6 bits to represent the basic bodies enumeration
   --  type, so we define such an integer type for conversions between
   --  Sweph codes and enumeration literals
   type Integer_6 is range - (2 ** 5) .. 2 ** 5 - 1;
   for Integer_6'Size use 6;

   -- Get the Sweph numeric code representation of a body:
   function Code (B : Basic_Bodies) return Integer is
      function Conv is new
        Ada.Unchecked_Conversion (Basic_Bodies, Integer_6);
   begin
      return Integer (Conv (B));
   end Code;

   -- Get the enumeration literal for a basic body, given its Sweph code
   function To_Body (C : Integer) return Basic_Bodies is
      pragma Assert
        (C in Code (Basic_Bodies'First) .. Code (Basic_Bodies'Last),
         "Attempt to convert invalid code to basic body");
      function Conv is new
        Ada.Unchecked_Conversion (Integer_6, Basic_Bodies);
   begin
      return Conv (Integer_6 (C));
   end To_Body;

   -- Get coordinates of last calculation
   function Longitude return Long_Float is
   begin
      return Ephemeris.Last_Results (1);
   end Longitude;

   function Latitude return Long_Float is
   begin
      return Ephemeris.Last_Results (2);
   end Latitude;

   function Velocity return Long_Float is
   begin
      return Ephemeris.Last_Results (4);
   end Velocity;

   --  Snapshot of selected basic bodies, for use by bodies selection
   --  dialog controller
   function Basic (M : Body_Model) return Basic_Bodies_Array is
   begin
      return M.Standard;
   end Basic;

   --  Code corresponds to a basic body?
   function Is_Basic (C : Integer) return Boolean is
   begin
      return C in Code (Basic_Bodies'First) .. Code (Basic_Bodies'Last);
   end Is_Basic;

   --  Number of selected bodies
   function Count (M : Body_Model) return Natural is
      Iter : Body_Iterator := Initialize (M);
      Result : Natural := 0;
   begin
      Start (Iter);
      while not Off (Iter) loop
         Result := Result + 1;
         Forth (Iter);
      end loop;
      return Result;
   end Count;

   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Clear extended bodies model
   procedure Clear_Extended_Model (M : Body_Model_Ptr) is
   begin
      Wipe_Out (M.Extended);
   end Clear_Extended_Model;


   --  Add an extended body
   procedure Extend (M : Body_Model_Ptr;  Code : Integer) is
   begin
      Extend (M.Extended, new Integer'(Code));
   end Extend;



   --  Commit changes from controller to model
   procedure Update_All
     (B : Body_Model_Ptr;
      S : Basic_Bodies_Array) is

      Evt : Selection_Event_Ptr := new Selection_Event;
   begin
      --  Update state of dialog
      B.Standard := S;
      Notify_Observers (B, Evt);
   end Update_All;


   --  Position calculations

   -- Call ephemeris calculation routine using body name and ET
   procedure Calculate_ET (JD_ET : Long_Float; B : Basic_Bodies) is
   begin
      Calculate_ET (JD_ET, Code (B));
   end Calculate_ET;

   --  Call ephemeris using body code and ET
   procedure Calculate_ET (JD_ET : Long_Float; Body_Code : Integer)
     renames Ephemeris.Calculate_ET;


   -- Versions taking Universal Time and converting to ET internally.
   -- Call ephemeris calculation routine using body name and UT
   procedure Calculate_UT (JD_UT : Long_Float; B : Basic_Bodies) is
   begin
      Calculate_UT (JD_UT, Code (B));
   end Calculate_UT;

   procedure Calculate_UT (JD_UT : Long_Float; Body_Code : Integer) renames
     Ephemeris.Calculate_UT;


   --  Iterator operations

   --  Oh yes, this is an _ugly_ implementation!

   --  Selected bodies state is represented by an array of booleans
   --  indicating selection status for the sweph basic bodies, indexed
   --  by the position values of the enumeration type "Basic_Bodies".
   --  The selected extended bodies have their codes in a list that
   --  extends the array. The "Item" operation returns the sweph body
   --  code in both cases

   --  Move to start of selected bodies
   procedure Start (I : in out Body_Iterator) is
   begin
      I.Cursor := Basic_Bodies'Pos (Basic_Bodies'First);
      if not I.Model.Standard (Basic_Bodies'First) then
         Forth (I);
      end if;
   end Start;


   --  Advance to next selected body
   procedure Forth (I : in out Body_Iterator) is
      pragma Assert (not Off (I),
                     "Attempted ""forth"" on ""off"" body iterator");
   begin
      if I.Cursor /= Extended then

         --  Iterator is still pointing into array of basic bodies
         I.Cursor := I.Cursor + 1;

         --  Advance to next selected body or off the array
         while I.Cursor < Extended
           and then not I.Model.Standard (Basic_Bodies'Val (I.Cursor)) loop
            I.Cursor := I.Cursor + 1;
         end loop;

         --  Now off the array.  Point to the first body in the
         --  "extended bodies" list
         if I.Cursor = Extended then
            Start (I.Model.Extended);
         end if;
      else
         --  Iterator is in the extended bodies list.  Advance one
         --  position
         Forth (I.Model.Extended);
      end if;

      pragma Assert
        (not (I.Cursor = Extended and Empty (I.Model.Extended)) or Off (I),
         "Forth beyond end of list fails to indicate Off");
   end Forth;

   --  Return the Sweph body code for current cursor position
   function Item (I : Body_Iterator) return Natural is
      pragma Assert (not Off (I));
   begin
      if I.Cursor
        in Basic_Bodies'Pos (Basic_Bodies'First)
        .. Basic_Bodies'Pos (Basic_Bodies'Last) then
         --  Convert cursor value (array index) to sweph body code
         return Code (Basic_Bodies'Val (I.Cursor));
      else
         --  Just return the code as stored in the extended bodies list
         return Item (I.Model.Extended).all;
      end if;
   end Item;

   --  "Off" the range of selected bodies?
   function Off (I : Body_Iterator) return Boolean is
   begin
      return I.Cursor = Before or else
        (I.Cursor = Extended and then Off (I.Model.Extended));
   end Off;

end Body_Models;
