------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                               BODY_MODELS                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Publish_Subscribe; use Publish_Subscribe;
with MVC; use MVC;
with Ephemeris; use Ephemeris;
with Lists;

--  Model of currently selected bodies for some operation (eg display,
--  primary direction significators or promissors).
package Body_Models is

   type Body_Model is new Model_Obj with private;
   type Body_Model_Ptr is access all Body_Model'Class;

   --  Iterator over all selected bodies.  Views representing positions
   --  use this access mechanism to obtain sweph body codes
   type Body_Iterator is private;

   --  The bodies provided with fixed codes by the Swiss Ephemeris.
   --  Others have codes determined by calculation based on the object's
   --  minor planet catalog number
   type Basic_Bodies is
     (Obliquity_and_Nutation,  -- See swephprg document
      Sun,
      Moon,
      Mercury,
      Venus,
      Mars,
      Jupiter,
      Saturn,
      Uranus,
      Neptune,
      Pluto,
      Mean_Node,
      True_Node,
      Mean_Apogee,
      Osculating_Apogee,
      Earth,
      Chiron,
      Pholus,
      Ceres,
      Pallas,
      Juno,
      Vesta);

   --  Ephemeris codes for the basic bodies
   for Basic_Bodies use
      (Obliquity_and_Nutation => -1,
       Sun => 0,
       Moon => 1,
       Mercury => 2,
       Venus => 3,
       Mars => 4,
       Jupiter => 5,
       Saturn => 6,
       Uranus => 7,
       Neptune => 8,
       Pluto => 9,
       Mean_Node => 10,
       True_Node => 11,
       Mean_Apogee => 12,
       Osculating_Apogee => 13,
       Earth => 14,
       Chiron => 15,
       Pholus => 16,
       Ceres => 17,
       Pallas => 18,
       Juno => 19,
       Vesta => 20);

   --  Public type representing the selection state of the basic bodies for
   --  use by the bodies selection controller
   type Basic_Bodies_Array is array (Basic_Bodies) Of Boolean;


   ----------------------
   --  Initialization  --
   ----------------------

   --  Create and initialize a body iterator
   function Initialize (M : Body_Model) return Body_Iterator;


   ---------------
   --  Queries  --
   ---------------

   -- Get the numeric code representation of a basic body, as used by sweph
   function Code (B : Basic_Bodies) return Integer;

   -- Get the enumeration literal for a basic body, given its code
   function To_Body (C : Integer) return Basic_Bodies;

   -- Get coordinates of last calculation performed by the ephemeris
   function Longitude return Long_Float;
   function Latitude return Long_Float;
   function Velocity return Long_Float;

   --  Snapshot of basic bodies selection, for use by selection dialog
   --  controller
   function Basic (M : Body_Model) return Basic_Bodies_Array;

   --  Code corresponds to a basic body?
   function Is_Basic (C : Integer) return Boolean;

   --  Number of selected bodies
   function Count (M : Body_Model) return Natural;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Body model operations

   --  Clear extended bodies model
   procedure Clear_Extended_Model (M : Body_Model_Ptr);

   --  Add an extended body.  "Code" is the sweph code for the body
   procedure Extend (M : Body_Model_Ptr;  Code : Integer);

   --  Commit changes made in the selection dialog
   procedure Update_All
     (B : Body_Model_Ptr;
      S : Basic_Bodies_Array);


   --  Call ephemeris calculation routines:

   --  Raise when calculation fails
   Calculation_Failed : exception renames Ephemeris.Calculation_Failed;

   -- Versions using Ephemeris Time:
   procedure Calculate_ET (JD_ET : Long_Float; B : Basic_Bodies);
   procedure Calculate_ET (JD_ET : Long_Float; Body_Code : Integer);

   -- Versions taking Universal Time and converting to ET internally:
   procedure Calculate_UT (JD_UT : Long_Float; B : Basic_Bodies);
   procedure Calculate_UT (JD_UT : Long_Float; Body_Code : Integer);


   --  Iterator operations.  These iterators are independent of each other

   --  Move iterator to first selected body
   procedure Start (I : in out Body_Iterator);

   --  Advance to next selected body
   procedure Forth (I : in out Body_Iterator);

   --  Return sweph body code for current cursor position of the iterator
   function Item (I : Body_Iterator) return Natural;

   --  Off the list of selected bodies?
   function Off (I : Body_Iterator) return Boolean;

private

   --  Range of codes for bodies that aren't in the "basic" subset
   subtype Extended_Bodies_Range is Positive range  10_000 .. Positive'Last;
   type Extended_Bodies_Ptr is access all Extended_Bodies_Range;

   --  Extension to the array of selected basic bodies, for representing
   --  extended bodies
   package Extended_Bodies_List is
      new Lists (Extended_Bodies_Range, Extended_Bodies_Ptr);
   use Extended_Bodies_List;

   --  Selected bodies model
   type Body_Model is new Model_Obj with record
      Standard : Basic_Bodies_Array :=
         (Obliquity_And_Nutation | True_Node .. Earth | Pholus .. Vesta
          => False,
          others => True);
      Extended : List;
   end record;

   --  Iterator cursor positions within basic bodies subset. We use
   --  the position values of the "Basic_Bodies" enumeration type
   Lower : constant := Basic_Bodies'Pos (Basic_Bodies'First);
   Upper : constant := Basic_Bodies'Pos (Basic_Bodies'Last);

   --  Full cursor range.  Lower - 1 indicates an iterator is "before"
   --  the conceptual list.  Upper + 1 indicates that the iterator is
   --  in the extended bodies list or "after" the conceptual list
   Before : constant := Lower - 1;
   Extended : constant := Upper + 1;

   subtype Cursor_Range is Integer range Before .. Extended;


   --  Iterator over selected bodies
   type Body_Iterator is record
      Model : Body_Model_Ptr;
      Cursor : Cursor_Range := Before;
   end record;

   --  Event to trigger update of layout of body selection dialog
   type Selection_Event is new Event with null record;
   type Selection_Event_Ptr is access all Selection_Event'Class;

   pragma Inline (Longitude, Latitude, Velocity);
end Body_Models;
