------------------------------------------------------------------------------
--                                                                          --
--                              AstroFrames                                 --
--                                                                          --
--                            EVENT_CONTROLLERS                             --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Exceptions; use Ada.Exceptions;

with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Field_Validation; use Field_Validation;
with Times.Strings; use Times, Times.Strings;

with Ephemeris; use Ephemeris;

with Gtk.Gentry; use Gtk.Gentry;

with Event_Models.Persistence;
use Event_Models, Event_Models.Persistence;
with Event_Views; use Event_Views;

with Message_Boxes; use Message_Boxes;

--  Controller for an event definition dialog
package body Event_Controllers is

   ----------------------
   --  Implementation  --
   ----------------------

   --  Extract date strings and update model
   procedure Set_Date (This : access Event_Controller);

   --  Extract event description and update model
   procedure Set_Description (This : access Event_Controller);

   --  Extract location strings and update model
   procedure Set_Location (This : access Event_Controller);

   --  Extract time strings and update model
   procedure Set_Time (This : access Event_Controller);

   pragma Inline (Set_Date, Set_Time);


    ----------------------
   --  Initialization  --
   ----------------------

   --  Set up controller
   procedure Initialize
     (C : access Event_Controller;
      M : access Model_Obj'Class;
      V : access View_Obj'Class) is
   begin
      --  Call super initialization operation
      Initialize (Controller_Obj (C.all)'Access, M, V);

      C.Dialog := Dialog (Event_View (V.all)'Access);
   end Initialize;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Close an event specification dialog
   procedure Close_Dialog (This : access Event_Controller) is
   begin
      Hide_All (This.Dialog);
   end Close_Dialog;

   --  Display an event specification dialog
   procedure Open_Dialog (This : access Event_Controller) is
   begin
      Show_All (This.Dialog);
   end Open_Dialog;

   --  Open notes textbox
   procedure Open_Notes (This : access Event_Controller) is
   begin
      Show_Notes (Event_View (View (This).all)'Access);
   end Open_Notes;

   --  Set modified bit. This indicates that the in-memory
   --  description of an event no longer matches its disk version
   procedure Set_Modified (This : access Event_Controller) is
   begin
      Set_Modified (Event_Model (Model (This).all)'access);
   end Set_Modified;

   -- Reset modified bit
   procedure Reset_Modified (This : access Event_Controller) is
   begin
      Reset_Modified (Event_Model (Model (This).all)'access);
   end Reset_Modified;

   --  Update event model and all active views
   procedure Update_All  (This : access Event_Controller) is
   begin
      Set_Description (This);
      Set_Date (This);
      Set_Location (This);
      Set_Time (This);

      Update_All (Event_Model (Model (This).all)'Access,
                  new Update_Event);
   exception
      --  Intercept bad input or calculation errors, and pop
      --  up a message box

      when E : Calculation_Failed =>
         Calculation_Error_Box (Exception_Message (E));

      when E :
        Invalid_Date | Invalid_Time | Invalid_Latitude
        |Invalid_Longitude | Invalid_Altitude  =>
         Invalid_Input_Box (Exception_Message (E));

   end Update_All;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Extract and validate date strings, and update model
   procedure Set_Date (This : access Event_Controller) is
   begin
      Set_Date (Event_Model (Model (This).all),
                Day_Value (Get_Text (This.Dialog.Date_Day)),
                Month_Value (Get_Text (This.Dialog.Date_Month)),
                Year_Value (Get_Text (This.Dialog.Date_Year)));
   exception
      when E : Invalid_Date =>
         Raise_Exception (Exception_Identity (E), "Bad Date Entered");
   end Set_Date;


   --  Extract event description and update model
   procedure Set_Description (This : access Event_Controller) is
   begin
      Set_Description
        (Event_Model (Model (This).all),
         Get_Text (This.Dialog.Description_Entry));
   end Set_Description;


   --  Extract and validate location strings, and update model
   procedure Set_Location (This : access Event_Controller) is
      Long_I, Long_M, Lat_I, Lat_M : Integer;
      Long_S, Lat_S : Long_Float;
   begin
      --  Validate and convert longitude strings
      Long_I := Integer'Value
        (Validate_Longitude (Get_Text (This.Dialog.Long_Degree)));
      Long_M := Minutes_Part'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Long_Minute)));
      Long_S := Long_Float'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Long_Second)));

      if Validate_EW (Get_Text (This.Dialog.Long_Direction)) = "West" then
         Long_I := - Long_I;
         Long_M := - Long_M;
         Long_S := - Long_S;
      end if;

      --  Validate and convert latitude strings
      Lat_I := Integer'Value
        (Validate_Latitude (Get_Text (This.Dialog.Lat_Degree)));
      Lat_M := Minutes_Part'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Lat_Minute)));
      Lat_S := Long_Float'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Lat_Second)));

      if Validate_NS (Get_Text (This.Dialog.Lat_Direction)) = "South" then
         Lat_I := - Lat_I;
         Lat_M := - Lat_M;
         Lat_S := - Lat_S;
      end if;

      --  Set location for event
      Set_Location (Model => Event_Model (Model (This).all),

                    Long  => To_Decimal (
                                         Make_Sexagesimal
                                         (Long_I, Long_M, Long_S)),
                    Lat => To_Decimal (
                                       Make_Sexagesimal
                                       (Lat_I, Lat_M, Lat_S)),
                    Alt => Long_Float'Value
                    (Validate_Altitude
                     (Get_Text (This.Dialog.Alt_Entry))));

   exception
      when E : Invalid_Longitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Longitude Entered");

      when E : Invalid_Latitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Latitude Entered");

      when E : Invalid_Altitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Altitude Entered");
   end Set_Location;


   --  Extract and validate time strings, and update model
   procedure Set_Time (This : access Event_Controller) is
   begin
      Set_Time (Event_Model (Model (This).all),

                Time_Value (Get_Text (This.Dialog.UT_Hour),
                            Get_Text (This.Dialog.UT_Minute),
                            Get_Text (This.Dialog.UT_Second)),

                Zone_Value (Get_Text (This.Dialog.Zone_Hour),
                            Get_Text (This.Dialog.Zone_Minute),
                            Get_Text (This.Dialog.Zone_Direction)));
   end Set_Time;

end Event_Controllers;
