------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                               EVENT_MODELS                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ephemeris; use Ephemeris;

with Main_Models; use Main_Models;

--  Model of a point in spacetime.  This is an abstraction of
--  anything for which you might want to draw a "chart"
package body Event_Models is

   ----------------------------
   --  Model Initialization  --
   ----------------------------

   procedure Initialize
     (This : access Event_Model;
      Main : access Model_Obj'Class;
      Use_Default_Location : Boolean := False) is
   begin
      This.Main_Model := Model_Ptr'(Main.all'Access);

      if Use_Default_Location then
         This.Current_Location
           := Default_Location (Main_Model (This.Main_Model.all)'Access);
         This.Date_Time.Zone :=
           Default_Time_Zone (Main_Model (This.Main_Model.all)'Access);
      end if;

      This.Current_Location.Dynamic_Angle
        := Dynamic_Angle (This.Current_Location.Lat);
   end Initialize;

   --  Make a new event that's a copy of "Prototype"
   procedure Clone
     (This : access Event_Model;
      Main : access Model_Obj'Class;
      Prototype : access Event_Model'Class) is
   begin
      This.Description := Prototype.Description;
      This.Date_Time := Prototype.Date_Time;
      This.Current_Location := Prototype.Current_Location;
      This.Notes := Prototype.Notes;
      This.Main_Model := Model_Ptr'(Main.all'Access);
      This.Modified := True;

      --  Recompute dynamic angle because the default location
      --  could have been the source of the prototype, and it
      --  doesn't have "Dynamic_Angle" set
      This.Current_Location.Dynamic_Angle
        := Dynamic_Angle (This.Current_Location.Lat);
   end Clone;


   ---------------------
   --  Model Queries  --
   ---------------------

   --  Value of ascendant.  Calculate_Angles must be called first
   function ASC (This : access Event_Model) return Long_Float is
   begin
      return Last_Angles (1);
   end ASC;

   --  Access model of bodies selected for display in event views
   function Body_Model (This : access Event_Model) return Body_Model_Ptr is
   begin
      return Body_Model (Main_Model (This.Main_Model.all)'Access);
   end Body_Model;

   --  Value of a cusp.  Calculate_Angles must be called first
   function Cusp (This : access Event_Model; Index : Cusps_Range)
                 return Long_Float is
   begin
      return Last_Cusps (Index);
   end Cusp;

   --  Event date & time
   function Date_Time (This : access Event_Model)
                      return Date_Time_Components is
   begin
      return This.Date_Time;
   end Date_Time;


    --  Event description
   function Description (This : access Event_Model) return String is
   begin
      return To_String (This.Description);
   end Description;


   --  File spec for event
   function File_Spec (This : access Event_Model) return Unbounded_String is
   begin
      return This.File_Spec;
   end File_Spec;


   --  Greenwich sidereal time in hours decimal
   function GST (JD_UT : Julian_Day) return Decimal_Hours is
   begin
      return Ephemeris.GST (JD_UT);
   end GST;

   --  Event location
   function Location (This : access Event_Model) return Location_Record is
   begin
      return This.Current_Location;
   end Location;


   --  Reference to main model
   function Main (This : access Event_Model) return Model_Ptr is
   begin
      return This.Main_Model;
   end Main;

   --  Value of MC.  Calculate_Angles must be called first
   function MC (This : access Event_Model) return Long_Float is
   begin
      return Last_Angles (2);
   end MC;

   --  Event notes
   function Notes (This : access Event_Model) return String is
   begin
      return To_String (This.Notes);
   end Notes;

   --  Value of RAMC.  Calculate_Angles must be called first
   function RAMC (This : access Event_Model) return Long_Float is
   begin
      return Last_Angles (3);
   end RAMC;


   ------------------------------
   --  Model Basic Operations  --
   ------------------------------

   --  Determine angles
   procedure Calculate_Angles
     (Model : access Event_Model;
      Latitude : latitudes) is
      JD_UT : Julian_Day :=
        Julian_Day_From_Gregorian
        (Model.Date_Time.D, Model.Date_Time.M,
         Model.Date_Time.Y, Model.Date_Time.Local_Time)
        - Model.Date_Time.Zone / 24.0;
   begin
      Calculate_Angles
        (JD_UT,
         Long => Model.Current_Location.Long,
         Lat => Latitude);
   end Calculate_Angles;

   --  Calculate house cusps
   procedure Calculate_Cusps
     (Model : access Event_Model;
      Latitude : Latitudes;
      System : House_Position_System) is
      JD_UT : Julian_Day :=
        Julian_Day_From_Gregorian
        (Model.Date_Time.D, Model.Date_Time.M,
         Model.Date_Time.Y, Model.Date_Time.Local_Time)
        - Model.Date_Time.Zone / 24.0;
   begin
      Calculate_Houses
        (JD_UT,
         Model.Current_Location.Long,
         Latitude,
         System);
   end Calculate_Cusps;


   -- Set date.  Gregorian only for now
   procedure Set_Date (Model : in out Event_Model;
                       D : Days; M : Months; Y : Years) is
   begin
      Model.Date_Time.D := D;
      Model.Date_Time.M := M;
      Model.Date_Time.Y := Y;
   end Set_Date;


   --  Set event description
   procedure Set_Description
     (Model : in out Event_Model; Descr : String) is
   begin
      Model.Description := To_Unbounded_String (Descr);
   end Set_Description;


   --  Set location
   procedure Set_Location  (Model : in out Event_Model;
                            Long : Longitudes;
                            Lat : Latitudes;
                            Alt : Altitudes) is
   begin
      Model.Current_Location := (Long, Lat, Dynamic_Angle (Lat), Alt);
   end Set_Location;


   -- Set Notes
   procedure Set_Notes (This : access Event_Model; S : String) is
   begin
      This.Notes := To_Unbounded_String (S);
   end Set_Notes;


   --  Set local time and zone
   procedure Set_Time (Model : in out Event_Model;
                       Local_Time : Decimal_Hours;
                       Zone : Time_Zone) is
   begin
      Model.Date_Time.Local_Time := Local_Time;
      Model.Date_Time.Zone := Zone;
   end Set_Time;


   --  Set enabled views displayable
   procedure Set_Views_Displayable (This : access Event_Model) is
      Evt : Update_Event_Ptr := new Update_Event;
   begin
      --  Force views to update
      Evt.Display_Status := Force_On;
      Update_All
        (Event_Model_Ptr'(This.all'Access),
         Evt,
         Suppress_Main_Update => True);
   end Set_Views_Displayable;


   --  Set enabled views undisplayable
   procedure Set_Views_Undisplayable (This : access Event_Model) is
      Evt : Update_Event_Ptr := new Update_Event;
   begin
      --  Force views to hide
      Evt.Display_Status := Force_Off;
      Update_All (Event_Model_Ptr'(This.all'Access), Evt);
   end Set_Views_Undisplayable;


   --  Send update notification to active views
   procedure Update_All
     (M : Event_Model_Ptr;
      E : access Event'Class;
      Suppress_Main_Update : Boolean := False) is

   begin
      if E.all in Update_Event'Class then
         --  Event has been changed.  Reflect changes in active views
         declare
            Evt : Update_Event_Ptr
              := Update_Event_Ptr'(Update_Event (E.all)'Access);
         begin
            --  Prepare update event information
            Evt.JD_UT := Julian_Day_From_Gregorian
              (M.Date_Time.D, M.Date_Time.M,
               M.Date_Time.Y, M.Date_Time.Local_Time);

            --  Adjust to UT.  Zone convention is West = negative
            Evt.JD_UT := Evt.JD_UT - M.Date_Time.Zone / 24.0;

            Evt.Dynamic_Angle := M.Current_Location.Dynamic_Angle;
            Evt.Longitude := M.Current_Location.Long;
            Evt.Latitude := M.Current_Location.Lat;
            Evt.Altitude := M.Current_Location.Alt;
         end;
      end if;

      Notify_Observers (M, E);

      --  For certain operations, we defer update of the main window,
      --  because the update affects the list of selected events under
      --  Windows - probably a GTK+ porting bug
      if not Suppress_Main_Update then
         --  Update event line in main window
         Update_All (Main_Model (M.Main_Model.all)'Access);
      end if;

   end Update_All;


   -----------------------------
   --  Export Event Queries  --
   -----------------------------

   --  Constructor
   function New_Export_Event (File_Spec : String) return Export_Event_Ptr is
      Result : Export_Event_Ptr := new Export_Event;
   begin
      Result.File_Spec := To_Unbounded_String (File_Spec);
      return Result;
   end New_Export_Event;

   --  File spec associated with export command
   function File_Spec (This : access Export_Event) return String is
   begin
      return To_String (This.File_Spec);
   end File_Spec;


   ----------------------------
   --  Update Event Queries  --
   ----------------------------

   --  Altitude
   function Altitude (E : Update_Event) return Altitudes is
   begin
      return E.Altitude;
   end Altitude;

   --  Get command for view updating
   function Display_Status (E : Update_Event) return Display_State is
   begin
      return E.Display_Status;
   end Display_Status;

   --  Svarogich's dynamic angle
   function Dynamic_Angle (E : Update_Event) return Latitudes is
   begin
      return E.Dynamic_Angle;
   end Dynamic_Angle;

   --  Julian Day Number and fractional Universal Time
   function JD_UT (E : Update_Event) return Julian_Day is
   begin
      return E.JD_UT;
   end JD_UT;

   --  Geographic latitude
   function Latitude (E : Update_Event) return Latitudes is
   begin
      return E.Latitude;
   end Latitude;

   --  Geographic longitude
   function Longitude (E : Update_Event) return Longitudes is
   begin
      return E.Longitude;
   end Longitude;


end Event_Models;
