------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                               EVENT_MODELS                               --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;

with Times; use Times;
with Locations; use Locations;
with Ephemeris; use Ephemeris;

with MVC; use MVC;
with Publish_Subscribe; use Publish_Subscribe;
with Body_Models; use Body_Models;

--  Model of a point in spacetime.  This is anything for which you might
--  wish to draw a "chart"
package Event_Models is

   -------------
   --  Model  --
   -------------

   type Event_Model is new Model_Obj with private;
   type Event_Model_Ptr is access all Event_Model'Class;


   ----------------------------
   --  Model Initialization  --
   ----------------------------

   --  Initialize an event
   procedure Initialize
     (This : access Event_Model;
      Main : access Model_Obj'Class;
      Use_Default_Location : Boolean := False);

   --  Create a new event, copying "Prototype"
   procedure Clone
     (This : access Event_Model;
      Main : access Model_Obj'Class;
      Prototype : access Event_Model'Class);


   ---------------------
   --  Model Queries  --
   ---------------------

   --  Value of ascendant.  Calculate_Angles must be called first
   function ASC (This : access Event_Model) return Long_Float;

   --  Access the model of bodies selected for display in event views
   function Body_Model (This : access Event_Model) return Body_Model_Ptr;

   --  Value of a cusp.  Calculate_Angles must be called first
   function Cusp (This : access Event_Model; Index : Cusps_Range)
                return Long_Float;

   --  Event date & time
   function Date_Time (This : access Event_Model) return Date_Time_Components;

   --  Event description
   function Description (This : access Event_Model) return String;

   --  Event file specification
   function File_Spec (This : access Event_Model) return Unbounded_String;

   --  Greenwich sidereal time in hours decimal
   function GST (JD_UT : Julian_Day) return Decimal_Hours;

   --  Event location
   function Location (This : access Event_Model) return Location_Record;

   --  Reference to the main model
   function Main (This : access Event_Model) return Model_Ptr;

   --  Value of MC.  Calculate_Angles must be called first
   function MC (This : access Event_Model) return Long_Float;

   --  Event notes
   function Notes (This : access Event_Model) return String;

   --  Value of RAMC.  Calculate_Angles must be called first
   function RAMC (This : access Event_Model) return Long_Float;


   -----------------------------
   --  Model Basic Operations --
   -----------------------------

   --  Determine angles
   procedure Calculate_Angles
     (Model : access Event_Model;
      Latitude : Latitudes);

   --  Calculate house cusps
   procedure Calculate_Cusps
     (Model : access Event_Model;
      Latitude : Latitudes;
      System : House_Position_System);

   --  Set date.  Gregorian only for now
   procedure Set_Date
     (Model : in out Event_Model; D : Days; M : Months; Y : Years);

   --  Set event description
   procedure Set_Description
     (Model : in out Event_Model; Descr : String);

   --  Set location
   procedure Set_Location  (Model : in out Event_Model;
                            Long : Longitudes;
                            Lat : Latitudes;
                            Alt : Altitudes);

   -- Set Notes
   procedure Set_Notes (This : access Event_Model; S : String);

   --  Set local time
   procedure Set_Time (Model : in out Event_Model;
                       Local_Time : Decimal_Hours;
                       Zone : Time_Zone);

   --  Set enabled views displayable
   procedure Set_Views_Displayable (This : access Event_Model);

   --  Set enabled views undisplayable
   procedure Set_Views_Undisplayable (This : access Event_Model);

   --  Update all active views of the event.  By default, the main list
   --  view is also updated.  "Suppress_Main_Update" skips this part of
   --  the operation
   procedure Update_All
     (M : Event_Model_Ptr;
      E : access Event'Class;
      Suppress_Main_Update : Boolean := False);


   --------------------
   --  Modify Event  --
   --------------------

   --  Command to initiate event data modification.  Causes the event
   --  specification dialog to be displayed.
   type Modification_Event is new Event with private;
   type Modification_Event_Ptr is access all Modification_Event'Class;


   --------------------
   --  Export Event  --
   --------------------

   --  Command to export current positions table to a file
   type Export_Event is new Event with private;
   type Export_Event_Ptr is access all Export_Event'Class;

   --  Constructor
   function New_Export_Event (File_Spec : String) return Export_Event_Ptr;

   --  File spec associated with export command
   function File_Spec (This : access Export_Event) return String;


   ----------------------------
   --  Primaries Open Event  --
   ----------------------------

   --  Command to open primary directions list dialog
   type Primaries_Open_Event is new Event with private;
   type Primaries_Open_Event_Ptr is access all Export_Event'Class;


   --------------------
   --  Update Event  --
   --------------------

   --  Command to update any active views representing the event.  Created
   --  when event data has changed
   type Update_Event is new Event with private;
   type Update_Event_Ptr is access all Update_Event'Class;

   --  Altitude
   function Altitude (E : Update_Event) return Altitudes;

   --  Action on display.  Affects the state of views on the event
   type Display_State is (Force_On, Force_Off, Freeze, Use_Current);

   --  Get status to enforce in a view
   function Display_Status (E : Update_Event) Return Display_State;

   --  Svarogich's dynamic angle.  Perpendicular to the horizon of gravitation
   function Dynamic_Angle (E : Update_Event) return Latitudes;

   --  Julian Day and fraction representing Universal Time
   function JD_UT (E : Update_Event) return Julian_Day;

   --  Geographic latitude
   function Latitude (E : Update_Event) return Latitudes;

   --  Geographic longitude
   function Longitude (E : Update_Event) return Longitudes;


private

   ------------------------
   --  Model Definition  --
   ------------------------

   --  Definition of a point in spacetime
   type Event_Model is new Model_Obj with record
      Modified : Boolean := False;
      Description : Unbounded_String := To_Unbounded_String ("");
      Date_Time : Date_Time_Components;
      Current_Location : Location_Record;
      Notes : Unbounded_String := To_Unbounded_String ("");
      File_Spec : Unbounded_String := To_Unbounded_String ("");
      Main_Model : Model_Ptr;
   end record;


   --------------------
   --  Modify Event  --
   --------------------

   --  Command to show event specification dialog
   type Modification_Event is new Event with null record;


   --------------------
   --  Export Event  --
   --------------------

   --  Command to export current positions to a file
   type Export_Event is new Event with record
      File_Spec : Unbounded_string;
   end record;


   ----------------------------
   --  Primaries Open Event  --
   ----------------------------

   --  Command to open primary directions list dialog
   type Primaries_Open_Event is new Event with null record;


   -------------------------------
   --  Update Event Definition  --
   -------------------------------

   --  Command to update views
   type Update_Event is new Event with record
      Display_Status : Display_State := Use_Current;
      JD_UT : Julian_Day;
      Dynamic_Angle : Latitudes;
      Longitude : Longitudes;
      Latitude : Latitudes;
      Altitude : Altitudes;
   end record;


   pragma Inline (GST, JD_UT, Longitude, Dynamic_Angle, Set_Time, Altitude);
   pragma Inline (Description, Date_Time, Location, Notes);
   pragma Inline (Display_Status);
   pragma Inline (Main, File_Spec);
   pragma Inline (ASC, MC, RAMC, Cusp);
   pragma Inline (File_Spec);
end Event_Models;
