------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                          MAIN_MODELS.PERSISTENCE                         --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.4 $                              --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Body_Models; use Body_Models;
with Event_Views; use Event_Views;
with Positions_Views; use Positions_Views;
with Primaries_Views; use Primaries_Views;

with Gnat.Os_Lib; use Gnat.Os_Lib;

with Ada.Streams.Stream_IO; use Ada.Streams.Stream_IO;

with Event_Models.Persistence; use Event_Models.Persistence;

--  Save and restore AstroFrames preferences
package body Main_Models.Persistence is
      type Events_Array is array (Natural range <>) of  Unbounded_String;

   --  Save the preferences to disk location SE_EPHE_PATH/astroframes.cfg
   procedure Persist (This : access Main_Model'Class) is
      F : File_Type;
      S : Stream_Access;
      Dir : String := Getenv ("SE_EPHE_PATH").all;
      Iter : Body_Iterator := Initialize (This.Body_Model.all);
      Event_Specs : Events_Array (1 ..  Event_Lists.Count (This.Events_List));
      Num_Events : Natural := 0;
   begin
      if Dir /= "" then
         Create (F, Name => Dir & Directory_Separator & "astroframes.cfg");
         S := Stream (F);

         --  Persist list of open events that have been saved to disk
         Start (This.Events_List);
         while not Off (This.Events_List) loop
            if not Modified (Item (This.Events_List)) and then
            To_String (File_Spec (Item (This.Events_List))) /= "" then
               Num_Events := Num_Events + 1;
               Event_Specs (Num_Events) :=
                 File_Spec (Item (This.Events_List));
            end if;
            Forth (This.Events_List);
         end loop;

         Natural'Write (S, Num_Events);
         for I in 1 .. Num_Events loop
            Unbounded_String'Write (S, Event_Specs (I));
         end loop;

         --  Write general preferences fields
         Unbounded_String'Write (S, This.Root_Event_Directory);
         Location_Record'Write (S, This.Default_Location);
         Time_Zone'Write (S, This.Default_Time_Zone);
         Boolean'Write (S, This.Startup_Event);
         Display_Array'Write (S, This.Selected_Displays);
         Coordinate_Systems'Write (S, This.Coordinates);

         --  Write body model
         Basic_Bodies_Array'Write (S, Basic (This.Body_Model.all));
         Start (Iter);
         while not Off (Iter) loop
            if not Is_Basic (Item (Iter)) then
               Integer'Write (S, Item (Iter));
            end if;
            Forth (Iter);
         end loop;

         Close (F);
      end if;
   end Persist;


   --  Restore preferences from disk location $SE_EPHE_PATH/astroframes.cfg
   function Restore return Main_Model_Ptr is
      Dir : String := Getenv ("SE_EPHE_PATH").all;
      F : File_Type;
      S : Stream_Access;
      Result : Main_Model_Ptr := new Main_Model;
      Basic_Bodies_Arr : Basic_Bodies_Array;
      Extended_Body_Code : Integer;
      Num_Events : Natural;
      Config_Path : constant String :=
        Dir & Directory_Separator & "astroframes.cfg";
   begin
      Result.Body_Model := new Body_Models.Body_Model;

      if Is_Regular_File (Config_Path) then
         Open (F, In_File, Config_Path);
         S := Stream (F);

         --  Reclaim file specs for events that were open last invocation
         Natural'Read (S, Num_Events);
         declare
            File_Spec : Unbounded_String;
            Event : Event_Model_Ptr;
         begin
            for I in  1 .. Num_Events Loop
               Unbounded_String'Read (S, File_Spec);
               Event := Restore (To_String (File_Spec));
               --  Put a reference to the main window model into the event
               --  model
               Event_Models.Initialize (Event, Result);

               --  Create its data entry view and controller
               Event_Views.Initialize (new Event_View, Event);

               --  Create its view and controller for a table of positions
               Positions_Views.Initialize (new Positions_View, Event);

               --  Create view and controller for primary directiosn
               Primaries_Views.Initialize (new Primaries_View, Event);

               --  Add the new event to main's list and update main's display
               Extend (Result, Event);
            end loop;
         end;

         --  Initialize general preferences fields from configuration file
         Unbounded_String'Read (S, Result.Root_Event_Directory);
         Location_Record'Read (S, Result.Default_Location);
         Time_Zone'Read (S, Result.Default_Time_Zone);
         Boolean'Read (S, Result.Startup_Event);
         Display_Array'Read (S, Result.Selected_Displays);
         Coordinate_Systems'Read (S, Result.Coordinates);

         --  Restore the selected bodies model
         Basic_Bodies_Array'Read (S, Basic_Bodies_Arr);
         while not End_Of_File (F) loop
            Integer'Read (S, Extended_Body_Code);
            Extend (Result.Body_Model, Extended_Body_Code);
         end loop;

         Close (F);
         Update_All (Result.Body_Model, Basic_Bodies_Arr);

         --  Set ephemeris to proper coordinate system
         Set_Coordinates (Result, Result.Coordinates);
      end if;

      return Result;
   end Restore;

end Main_Models.Persistence;
