------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                                MAIN_VIEWS                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.5 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Interfaces.C; use Interfaces.C;
with Gnat.Os_Lib; use Gnat.Os_Lib;

with Main_Models; use Main_Models;
with Main_Controllers; use Main_Controllers;
with Body_Controllers; use Body_Controllers;
with Filter_Primaries_Controllers; use Filter_Primaries_Controllers;
with Location_Controllers; use Location_Controllers;

with Event_Models.Persistence;
use Event_Models, Event_Models.Persistence;

with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Times.Strings; use Times, Times.Strings;
with Locations.Strings; use Locations.Strings;

with Glib; use Glib;
with Gtkada.Types; use Gtkada.Types;
with Gtk.Check_Menu_Item; use Gtk.Check_Menu_Item;
with Gtk.Enums; use Gtk.Enums;
with Gtk.Clist; use Gtk.Clist;
with Gtk.Radio_Menu_Item; use Gtk.Radio_Menu_Item;

--  View for main window.  A list of events
package body Main_Views is

   use Main_Controllers.Identification;

   --  Generate an event for program startup time at default location
   procedure Generate_Startup_Event (This : access Main_View'Class);

   ----------------------
   --  Initialization  --
   ----------------------

   --  Initialize event list view and controller
   procedure Initialize
     (This : access Main_View;
      M : access Model_Obj'Class) is
      Model_Cast : Main_Model_Ptr := Main_Model (M.all)'Access;
   begin
      --  Super initialization operation.  Handles MVC setup
      MVC.Initialize (View_Obj (This.all)'Access, M);
      Add_Observer (M, Observer (This.all)'Access);

      --  Create  main window and controller
      Gtk_New (This.Window);
      This.Controller := new Main_Controller;

      --  Initialize main window.  Controller reference allows
      --  GTK+ callbacks to dispatch to the controller for GUI event
      --  handling.
      Main_Window_Pkg.Initialize
        (This.Window, Controller_Ptr'(This.Controller.all'Access));

      --  Set column justifications for Event_List
      Set_Column_Justification (This.Window.Event_Table, 0, Justify_Center);
      Set_Column_Justification (This.Window.Event_Table, 1, Justify_Left);

      for I in 3 .. Get_Columns (This.Window.Event_Table) loop
         Set_Column_Justification
           (This.Window.Event_Table, I - 1, Justify_Right);
      end loop;

      --  Enable user drag & drop row-reordering
      Set_Reorderable (This.Window.Event_Table, True);
      Set_Use_Drag_Icons (This.Window.Event_Table, True);

      --  Initialize controller for window Event_List
      Main_Controllers.Initialize (This.Controller, M, This);

      --  Create view for body selection dialog
      This.Body_View := new Body_View;

      --  Initialize view for body selection dialog
      Body_Views.Initialize (This.Body_View, Model_Cast);

      --  Create and initialize default location dialog
      This.Location_View := new Location_View;
      Location_Views.Initialize (This.Location_View, M);

      --  Create and initialize primaries filtering dialog
      This.Filter_Primaries_View := new Filter_Primaries_View;
      Filter_Primaries_Views.Initialize (This.Filter_Primaries_View, M);

      --  Generate startup event (current transit time)
      if Startup_Event (Model_Cast) then
         Generate_Startup_Event (This);
      end if;

      --  Initialize display selection menu, setting the menu items
      --  to the values from the last session
      Set_Active (This.Window.Table_Of_Positions,
                  Displayable (Model_Cast, Table_Of_Positions));

      Set_Active (This.Window.Conventional_Chart,
                  Displayable (Model_Cast, Conventional_Chart));

      Set_Active (This.Window.Solar_Zodiac,
                  Displayable (Model_Cast, Solar_Zodiac));

      Set_Active (This.Window.Lunar_Zodiac,
                  Displayable (Model_Cast, Lunar_Zodiac));

      Set_Active (This.Window.Terrestrial_Zodiac,
                 Displayable (Model_Cast, Terrestrial_Zodiac));

      --  Set selection of coordinate system in preferences menu
      case Coordinate_System (Model_Cast) is
         when Topocentric_True =>
            Set_Active (This.Window.Topocentric_True, True);
         when Topocentric_Apparent =>
            Set_Active (This.Window.Topocentric_Apparent, True);
         when Conventional =>
            Set_Active (This.Window.Conventional, True);
      end case;

      --  Display main window
      Show_All (This.Window);
      Update_All (Model_Cast);
   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   --  Access to main window
   function Window (This : access Main_View) return Main_Window_Access is
   begin
      return This.Window;
   end Window;



   -------------------------
   --  Basic Operations   --
   -------------------------

   --  Pop up dialog for body selection
   procedure Display_Bodies_Dialog (This : access Main_View) is
   begin
      --  Refresh dialog shadow data from selected bodies model
      Reset_Dialog_State (Controller (This.Body_View));
      Update_All (Controller (This.Body_View), Opening_Dialog => True);
      Display (Controller (This.Body_View));
   end Display_Bodies_Dialog;


   --  Trigger popup of primaries filtering dialog
   procedure Display_Filter_Primaries_Dialog (This : access Main_View) is
   begin
      Open_Dialog (Controller (This.Filter_Primaries_View));
   end Display_Filter_Primaries_Dialog;

   --  Trigger popup of default location dialog
   procedure Display_Location_Dialog (This : access Main_View) is
   begin
      Open_Dialog (Controller (This.Location_View));
   end Display_Location_Dialog;


   --  Update main window
   procedure Update
     (This : access Main_View;
      Source : access Observable'Class;
      Evt : access Event'Class) is

      --  Downcast "Source" to "Main_Model_Ptr"
      M : Main_Model_Ptr := Main_Model_Ptr'(Main_Model (Source.all)'Access);
      L : Gtk_Clist := This.Window.Event_Table;
      Event : Event_Model_Ptr;
   begin
      --  Suspend visual updates, then clear the main window
      Freeze (L);
      Clear (L);

      --  Display each active event
      Start (M);
      while not Off (M) loop
         Event := Item (M);

         declare
            --  Format table entries for an event
            Status : String (1..5) := "     ";
            Description : String :=
              Event_Models.Description (Event);

            Time : String :=
              Time_String (Date_Time (Event).Local_Time);

            Zone : String :=
              Zone_String (Date_Time (Event).Zone);

            Date : String :=
              Date_String
              (Date_Time (Event).D,
              Date_Time (Event).M,
              Date_Time (Event).Y);

            Latitude : String :=
              Latitude_String (Location (Event).Lat);
            Longitude : String :=
              Longitude_String (Location (Event).Long);
            Altitude : String :=
              Altitude_String (Location (Event).Alt);
            Row : Gint;
         begin
            if Modified (Event) then
               Status := "  *  ";
            end if;

            Row := Append
              (L,
               Status + Description + Time + Zone + Date +
               Latitude + Longitude + Altitude);

            --  Attach a reference to the model of the event to the row
            --  so it can still be referenced if the row is reordered in
            --  the table by the user
            Set (This.Window.Event_Table, Row, Event);
         end;

         Forth (M);
      end loop;

      --  Allow screen update of main window
      Thaw (L);
   end Update;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Generate event for program startup.  In order to get UT, we generate
   --  a temp file and grab its timestamp.  This may not work on windows
   --  95/98/ME, in which case, we'll need to get the timezone info.  Will
   --  probably want to add it anyway for user convenience, and the code
   --  will be cleaner
   procedure Generate_Startup_Event (This : access Main_View'Class) is
      Event : Event_Model_Ptr := new Event_Model;
      Temp_Name : Temp_File_Name;

      --  Reserve space for file name
      C_Name : Char_Array := To_C (Temp_File_Name'(others => ' '));
      F : File_Descriptor;
      Now : Os_Time;
      Dummy : Boolean;
      M : Main_Model_Ptr :=
        Main_Model (Model (This).all)'Access;
      Temp_JD : Long_Float;
   begin
      --  Create temp file in order to grab timestamp
      Create_Temp_File (F, Temp_Name);
      Now := File_Time_Stamp (F);
      C_Name := To_C (Temp_Name);
      Delete_File (C_Name'Address, Dummy);

      --  Convert OS GMT to Julian day, then adjust for default time
      --  zone.  Conversion back to local time takes care of
      --  any movement into the previous or next day due to zone
      --  adjustment
      Temp_JD :=
        Gregorian_To_Julian_Day
        (Gm_Day (Now), Months'Val (Gm_Month (Now) - 1), Gm_Year (Now),
         To_Decimal
         (Make_Sexagesimal
          (Gm_Hour (Now),
           Gm_Minute (Now),
           Long_Float (Gm_Second (Now)))));

      --  Convert to local time. Convention is west zones are negative
      Temp_JD := Temp_JD + Default_Time_Zone (M) / 24.0;

      Set_Date (Event.all,
                Day (Temp_JD),
                Month (Temp_JD),
                Year (Temp_JD));

      Set_Time (Event.all,
                JD_Hours (Temp_JD),   -- Actually local time
                Default_Time_Zone (M));

      Set_Description (Event.all, "Transit Event");

      Set_Location (Event.all,
                    Default_Location (M).Long,
                    Default_Location (M).Lat,
                    Default_Location (M).Alt);

      Main_Controllers.Clone (This.Controller, Event);
      Update_All (M);
   end Generate_Startup_Event;

end Main_Views;
