------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                             PRIMARIES_VIEWS                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.4 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Event_Models; use Event_Models;
with Primaries_Views.Calculations;
with Primaries_Controllers.Events; use Primaries_Controllers.Events;

with Times.Strings; use Times.Strings;
with Locations.Strings; use Locations.Strings;

with Glib; use Glib;
with Gtk.Button; use Gtk.Button;
with Gtk.Enums; use Gtk.Enums;
with Gtk.Label; use Gtk.Label;
with Gtk.Gentry; use Gtk.Gentry;
with Gtk.Clist; use Gtk.Clist;
with Gtkada.Types; use Gtkada.Types;

with Interfaces.C.Strings;

with Ada.Text_IO; use Ada.Text_IO;
with Ada.Strings.Fixed; use Ada.Strings, Ada.Strings.Fixed;

--  View for primary directions list dialog. Model is an event.
package body Primaries_Views is

   --  Current Key.  Used in exporting directions to file
   Current_Key : Keys;

   ----------------------
   --  Implementation  --
   ----------------------

   --  Dump the positions into a designated file
   procedure Export_Primaries (
                    This: access Primaries_View;
                    Model : access Event_Model'Class;
                    Evt: access Primaries_Export_Event'Class);

   --  Return array of strings representing directions
   function Primaries
     (Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class;
      Display_Columns : Positive) return Directions_Table;

   --  Write event description
   procedure Write_Event_Info
     (F : File_Type; Event : access Event_Model'Class);


   ---------------------
   -- Initialization  --
   ---------------------

   procedure Initialize
     (This : access Primaries_View;
      M : access Model_Obj'Class) is
   begin
      --  Super initialization operation
      MVC.Initialize (View_Obj (This.all)'Access, M);
      Add_Observer (M, View_Obj (This.all)'Access);

      --  Create primaries list dialog and controller
      Gtk_New (This.Dialog);
      This.Controller := new Primaries_Controller;

      --  Initialize primaries list dialog
      Primaries_Dialog_Pkg.Initialize
        (This.Dialog, Controller_Ptr'(This.Controller.all'Access));

      for I in 0 .. Get_Columns (This.Dialog.Direction_List) loop
         if I mod 2 = 0 then
            Set_Column_Justification
              (This.Dialog.Direction_List, I, Justify_Right);
         else
            Set_Column_Justification
              (This.Dialog.Direction_List, I, Justify_Left);
         end if;
      end loop;

      Filter_Primaries_Views.Initialize (This.Filter'Access, M);

      Primaries_Controllers.Initialize
        (This.Controller, M, This);

   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   function Dialog
     (This : access Primaries_View)
      return Primaries_Dialog_Access is
   begin
      return This.Dialog;
   end Dialog;

   --  Reference to the filtering dialog
   function Filter (This : access Primaries_View)
                   return Filter_Primaries_View_Ptr is
   begin
      return This.Filter'Access;
   end Filter;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Close the primary directions list dialog
   procedure Close (This : access Primaries_View) is
   begin
      This.Displaying := False;
      Hide_All (This.Dialog);
   end Close;


   --  Response to model notifications
   procedure Update
     (This : access Primaries_View;
      Source : access Observable'Class;
      Evt : access Event'Class) is

      Radix : Event_Model_Ptr := Event_Model (Source.all)'Access;
   begin
      if Evt.all in Primaries_Open_Event'Class then
         --  Opening the primaries dialog

         This.Displaying := True;

         --  Initialize dialog
         declare
            Time : String :=
              Time_String (Date_Time (Radix).Local_Time);

            Zone : String :=
              " Zone: " & Zone_String (Date_Time (Radix).Zone);

            Date : String :=
              Date_String
              (Date_Time (Radix).D,
               Date_Time (Radix).M,
               Date_Time (Radix).Y);

            --  Default range is radix to radix + 30 years
            End_Month : Months := Date_Time (Radix).M;
            End_Year : Years := Date_Time (Radix).Y + 30;

            Latitude : String :=
              "Lat: " & Latitude_String (Location (Radix).Lat);
            Longitude : String :=
              " Long: " & Longitude_String (Location (Radix).Long);
            Altitude : String :=
              " Alt: " & Altitude_String (Location (Radix).Alt);
         begin

            --  Update dialog title for the radix
            Set_Title
              (This.Dialog,
               "AstroFrames: Primary Directions List for " &
               Description (Radix));
            Show_All (This.Dialog);

            --  Set radix data
            Set_Text
              (This.Dialog.Radix_Label,
               "Radix: " & Date  & Time & Zone &
               Latitude & Longitude & Altitude);

            --  Set default primaries range to first 30 years
            Set_Text
              (This.Dialog.Start_Month,
               Month_String (Date_Time (Radix).M));

            Set_Text
              (This.Dialog.Start_Year,
               Year_String (Date_Time (Radix).Y));

            --  Decrement End_Month
            if End_Month = Jan then
               End_Month := Dec;
               End_Year := End_Year - 1;
            else
               End_Month := Months'Pred (End_Month);
            end if;

            Set_Text
              (This.Dialog.End_Month, Month_String (End_Month));

            Set_Text
              (This.Dialog.End_Year, Year_String (End_Year));

            Clear (This.Dialog.Direction_List);

            Set_Sensitive (This.Dialog.Export_Button, False);
         end;

      elsif This.Displaying then
         if Evt.all in Primaries_Update_Event'Class then
            --  List needs to be updated
            declare
               Display_Columns : constant Positive :=
                 Natural (Get_Columns (This.Dialog.Direction_List));

               pragma Assert (Display_Columns mod 2 = 0
                              and Display_Columns >= 2,
                              "Odd number of columns in directions table");
               Row : Gint;

               --  Get table of formatted strings representing directions
               Directions : Directions_Table :=
                 Primaries
                 (Radix,
                  Primaries_Update_Event (Evt.all)'Access,
                  Display_Columns);

            begin
               Current_Key := Key (Primaries_Update_Event (Evt.all)'Access);

               Freeze (This.Dialog.Direction_List);
               Clear (This.Dialog.Direction_List);

               --  Update the dialog directions list
               for I in Directions'Range (1) loop
                  declare
                     use Interfaces.C, Interfaces.C.Strings;
                     Row_Contents : Gtkada.Types.Chars_Ptr_Array
                       (0 .. Size_T (Display_Columns - 1));
                  begin
                     for J in Directions'Range (2) loop
                        Row_Contents (Size_T (J - 1)) :=
                          New_String (To_String (Directions (I, J)));
                     end loop;
                     Row := Append
                       (This.Dialog.Direction_List, Row_Contents);
                  end;
               end loop;

               Thaw (This.Dialog.Direction_List);
            end;
         elsif Evt.all in Update_Event'Class then
            --  Data or a relevant setting has been changed.
            --  Regenerate the primaries list
            Calculate (This.Controller);
         elsif Evt.all in Primaries_Export_Event'Class then
            --  Export list of directions to file
            declare
               Model : Event_Model_Ptr := Event_Model (Source.all)'Access;
            begin
               Export_Primaries
                 (This, Model, Primaries_Export_Event (Evt.all)'Access);
            end;
         end if;
      end if;
   end Update;

   ----------------------
   --  Implementation  --
   ----------------------

   --  Dump the positions into a designated file.  Really simple; no
   --  attempt to do it prettily
   procedure Export_Primaries (
                    This: access Primaries_View;
                    Model : access Event_Model'Class;
                    Evt: access Primaries_Export_Event'Class) is

      F : File_Type;
      Date_Width : constant := 12;
      Date_String : String (1 .. Date_Width);

      --  Used to control line breaks between years
      Last_Direction_Year : Integer := Integer'Last;
      This_Direction_Year : Integer;

   begin
      Create (F, Name => File_Spec (Evt));

      Write_Event_Info (F, Model);

      New_Line (F);

      --  Write column headers
      Put_Line (F, "    Date            Promissor      ->   Significator");
      New_Line (F);

      for Column in 0 .. 1 loop
         for Row in 0 .. Get_Rows (This.Dialog.Direction_List) - 1 loop
               --  Write a direction
            Move (Get_Text (This.Dialog.Direction_List, Row,
                            Gint (2 * Column)),
                  Date_String,
                  Justify => Right);

            if Date_String /= String'(Date_Width * ' ') then
               --  Some entries at the end of the table can be empty.
               --  Don't bother writing them

               --  Insert line breaks between years for better readability
               This_Direction_Year :=
                 Integer'Value
                 (Date_String (Date_String'Last - 3 .. Date_String'Last));

               if This_Direction_Year > Last_Direction_Year then
                  New_Line (F);
               end if;
               Last_Direction_Year := This_Direction_Year;

               Put (F, Date_String);
               Set_Col (F, Ada.Text_IO.Count (Date_Width + 5));
               Put_Line (F,
                         Get_Text (This.Dialog.Direction_List, Row,
                                   Gint (2 * Column + 1)));
            end if;
         end loop;
      end loop;

      Close (F);
   end Export_Primaries;

   --  Return table of formatted strings representing directions.
   function Primaries
     (Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class;
      Display_Columns : Positive) return Directions_Table is

      pragma Assert (Display_Columns mod 2 = 0 and Display_Columns >= 2);
      use Primaries_Views.Calculations;

      --  Generate directions for the specified range of dates
      Directions : Directions_Array := Generate (Radix, Evt);

      Directions_Per_Row : Positive := Display_Columns / 2;

      --  Table of strings to be loaded into dialog directions table
      Result : Directions_Table
        (1 .. (Directions'Length - 1) / Directions_Per_Row + 1,
         1 .. Display_columns)
        := (others => (others => Null_Unbounded_String));

      subtype Empty_Table is
        Directions_Table (1 .. 1, 1 .. Display_Columns);
      No_Directions : constant Empty_Table :=
        (others => (To_Unbounded_String ("No directions"),
                    others => Null_Unbounded_String));

   begin
      if Directions'Length = 1 then
         --  All Directions_Array contain at least an element for
         --  index 0.  Required by GNAT.Hsort_A used to sort directions
         --  by date
         return No_Directions;
      else
         --  Move sorted directions into table and format them
         for I in 1 .. Directions'Last loop
            declare
               --  Determine table position
               Row : Positive := (I - 1) mod Result'Last (1) + 1;
               Col : Positive := 2 * ((I - 1) / Result'Last (1)) + 1;
            begin
               --  Write date string
               Result (Row, Col) :=
                 To_Unbounded_String
                 (Date_String
                  (Day (Directions (I).Date),
                   Month (Directions (I).Date),
                   Year (Directions (I).Date)));

               --  Write description string
               Result (Row, Col + 1) := Directions (I).Description;
            end;
         end loop;

         return Result;
      end if;
   end Primaries;

   --  Write event description
   procedure Write_Event_Info
     (F : File_Type; Event : access Event_Model'Class) is
      -- Format various strings
      Description : String :=
        "Primary Directions for " & Event_Models.Description (Event);

      Time : String :=
        Time_String (Date_Time (Event).Local_Time);

      Zone : String :=
        " Zone: " & Zone_String (Date_Time (Event).Zone);

      Date : String :=
        Date_String
        (Date_Time (Event).D,
         Date_Time (Event).M,
         Date_Time (Event).Y);

      Latitude : String :=
        "Lat: " & Latitude_String (Location (Event).Lat);
      Longitude : String :=
        " Long: " & Longitude_String (Location (Event).Long);
      Altitude : String :=
        " Alt: " & Altitude_String (Location (Event).Alt);
   begin
      Put_Line (F, Description);
      New_Line (F);

      Put_Line (F, Date  & Time & Zone);
      Put_Line (F, Latitude & Longitude & Altitude);

      New_Line (F);
      Put_Line (F, "Key = " & Keys'Image (Current_Key));
      New_Line (F);
   end Write_Event_Info;

end Primaries_Views;
