------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                           ASPECT_SELECTIONS                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

--  Abstraction for a selection of aspects.  The implementation is an
--  arrayed list of fixed length.
package body Aspect_Selections is

   --  Selectable aspects that have 360 degree complements
   subtype Complementary_Aspect is Selector_Aspect
     range Semi_Sextile .. Quincunx;

   --  Return an aspect's 360-degree complement
   function Complement (A : Complementary_Aspect) return Aspect is
   begin
      return Aspect'Val
        (Aspect'Pos (Aspect'Last) -
         (Aspect'Pos (Aspect (A)) -
          Aspect'Pos (Aspect (Selector_Aspect'First))) + 1);
   end Complement;

   ---------
   -- Add --
   ---------

   --  Add a selectable aspect and its 360-degree complement
   procedure Add (A : in out Aspect_Selection; Item : Selector_Aspect) is
   begin
      A.Selected (Aspect (Item)) := True;
      if Item in Complementary_Aspect then
         A.Selected (Complement (Item)) := True;
      end if;
   end Add;

   ---------
   -- Add --
   ---------

   --  Add a common aspect (one not associated with a promissor or
   --  significator in primary directions)
   procedure Add (A : in out Aspect_Selection; Item : Common_Aspect) is
   begin
      A.Selected (Aspect (Item)) := True;
   end Add;

   -----------
   -- Forth --
   -----------

   --  Advance cursor to next selected aspect, or Off
   procedure Forth (A : in out Aspect_Selection) is
      pragma Assert (not A.Off);
   begin
      while not A.Off loop
         if A.Cursor = Aspect'Last then
            A.Off := True;
         else
            A.Cursor := Aspect'Succ (A.Cursor);
            exit when A.Selected (A.Cursor);
         end if;
      end loop;
   end Forth;

   ----------
   -- Item --
   ----------

   --  Return aspect at current cursor location
   function Item (A : Aspect_Selection) return Aspect is
      pragma Assert (not A.Off);
   begin
      return A.Cursor;
   end Item;

   ---------
   -- Off --
   ---------

   --  Cursor off the list?
   function Off (A : Aspect_Selection) return Boolean is
   begin
      return A.Off;
   end Off;

   ------------
   -- Remove --
   ------------

   --  Remove designated aspect and its 360 degree complement
   procedure Remove (A : in out Aspect_Selection; Item: Selector_Aspect) is
   begin
      A.Selected (Aspect (Item)) := False;
      if Item in Complementary_Aspect then
         A.Selected (Complement (Item)) := False;
      end if;
   end Remove;

   ------------
   -- Remove --
   ------------

   --  Remove designated common aspect
   procedure Remove (A : in out Aspect_Selection; Item: Common_Aspect) is
   begin
      A.Selected (Aspect (Item)) := False;
   end Remove;

   -----------
   -- Start --
   -----------

   --  Initialize cursor for iteration
   procedure Start (A : in out Aspect_Selection) is
   begin
      A.Off := False;
      A.Cursor := Aspect'First;
      if not A.Selected (A.Cursor) then
         Forth (A);
      end if;
   end Start;

   --------------
   -- Wipe_Out --
   --------------

   --  Clear the selection
   procedure Wipe_Out (A : in out Aspect_Selection) is
   begin
      A.Off := True;
      A.Selected := (others => False);
   end Wipe_Out;

end Aspect_Selections;

