------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                                  TIMES                                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Decimal_Sexagesimal; use Decimal_Sexagesimal;

--  Time and Date types and operations
package body Times is

   --  Convert month numeric code to "Months" value
   function Month is
      new Ada.Unchecked_Conversion (Month_Range, Months);

   --  Convert "Months" value to its numeric code
   function Code is
      new Ada.Unchecked_Conversion (Months, Month_Range);


   --  Forward declarations of implementation routines

   type Date_Intermediate is record
      B, C, D, E, F : Long_Float;
   end record;

   --  Intermediate parameters for Julian Day to Gregorian Calendar conversion
   function Date_Parameters (JD : Julian_Day) return Date_Intermediate;

   --  Common part of conversion to Julian day
   function Common_JD_Conversion
     (Year_Number, Month_Number : Integer;
      Day : Long_Float;
      B : Integer) return Julian_Day;

   ---------------
   --  Queries  --
   ---------------

   -- Check validity of date.  Gregorian only.
   function Valid_Date (D : Days; M : Months; Y : Years) return Boolean is
   begin
      case M is
         when Apr | Jun | Sep | Nov =>
            return D in 1 .. 30;

         when Feb =>
            if Y mod 4 = 0 and then
              (Y mod 400 = 0 or else Y mod 100 /= 0) then
               return D in 1..29;
            else
               return D in 1..28;
            end if;

         when others =>
            return True;
      end case;
   end Valid_Date;


   -------------------
   --  Conversions  --
   -------------------

   --  Convert date in Gregorian calendar to Julian day
   function Gregorian_To_Julian_Day
     (D : Days; M : Months; Y : Years; H : Decimal_Hours) return Julian_Day is

      pragma Assert (Valid_Date (D, M, Y), "Invalid Date");

      Mn : Natural := Code (M);
      A, B : Integer;
      Yn : Integer := Y;
      Day : Long_Float := Long_Float (D) + H / 24.0;

   begin
      if Mn <= 2 then
         Yn := Yn - 1;
         Mn := Mn + 12;
      end if;

      A := Yn / 100;
      B := 2 - A + A / 4;

      return Common_JD_Conversion (Yn, Mn, Day, B);
   end Gregorian_To_Julian_Day;

   --  Convert date in Julian calendar to Julian Day
   function Julian_To_Julian_Day
     (D : Days; M : Months; Y : Years; H : Decimal_Hours) return Julian_Day is

      Mn : Natural := Code (M);
      Yn : Integer := Y;
      Day : Long_Float := Long_Float (D) + H / 24.0;
   begin
      if Mn <= 2 then
         Yn := Yn - 1;
         Mn := Mn + 12;
      end if;

      return Common_JD_Conversion (Yn, Mn, Day, 0);
   end Julian_To_Julian_Day;


   -------------------
   --  Measurement  --
   -------------------

   --  Hours part of JD in H.dddd format
   function JD_Hours (JD : Julian_Day) return Decimal_Hours is
      Integral : Long_Float := Long_Float'Truncation (JD + 0.5);
   begin
      return 24.0 * (JD + 0.5 - Integral);
   end JD_Hours;

   --  Extract Gregorian day component from Julian day
   function Day (JD : Julian_Day) return Days is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
   begin
      return Integer (Long_Float'Truncation (Intermediate.B - Intermediate.D -
        Long_Float'Truncation (30.6001 * Intermediate.E) + Intermediate.F));
   end Day;

   --  Extract Gregorian month component from Julian day
   function Month (JD : Julian_Day) return Months is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
      Result : Months;
   begin
      if Intermediate.E < 14.0 then
         Result := Month (Integer (Intermediate.E - 1.0));
      else
         Result := Month (Integer (Intermediate.E - 13.0));
      end if;
      return Result;
   end Month;

   --  Extract Gregorian year component from Julian day
   function Year (JD : Julian_Day) return Years is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
      Month_Result : Month_Range;
      Result : Years;
      pragma Assert (Intermediate.E in 4.0 .. 15.0, "E out of range");
   begin
      if Intermediate.E < 14.0 then
         Month_Result :=Integer (Intermediate.E - 1.0);
      else
         Month_Result := Integer (Intermediate.E - 13.0);
      end if;

      if Month_Result > 2  then
         Result := Years (Intermediate.C - 4716.0);
      else
         Result := Years (Intermediate.C - 4715.0);
      end if;
      return Result;
   end Year;

   --  Extract Gregorian hour component from Julian day
   function Hour (JD : Julian_Day) return Hours is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
   begin
      return Integral
        (To_Sexagesimal (24.0 * Intermediate.F));
   end Hour;

   --  Extract Gregorian minute component from Julian day
   function Minute (JD : Julian_Day) return Minutes is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
   begin
      return Decimal_Sexagesimal.Minutes
        (To_Sexagesimal (24.0 * Intermediate.F));
   end Minute;

   --  Extract Gregorian second component from Julian day
   function Second (JD : Julian_Day) return Seconds is
      Intermediate : Date_Intermediate := Date_Parameters (JD);
   begin
      return Decimal_Sexagesimal.Seconds
        (To_Sexagesimal (24.0 * Intermediate.F));
   end Second;



   ----------------------
   --  Implementation  --
   ----------------------

   --  From J. Meeus "Astronomical Algorithms", Chapt 7

   --  Common part of conversion to Julian Day
   function Common_JD_Conversion
     (Year_Number, Month_Number : Integer;
      Day : Long_Float;
      B : Integer) return Julian_Day is
   begin
      return
        Long_Float'Truncation
        (365.25 * Long_Float (Year_Number + 4716)) +
        Long_Float'Truncation
        (30.6001 * Long_Float (Month_Number + 1)) +
        Long_Float (B) + Day - 1524.5;
   end Common_JD_Conversion;

   --  Intermediate results of conversion from Julian Day to Gregorian
   function Date_Parameters (JD : Julian_Day) return Date_Intermediate is
      Result : Date_Intermediate;
      Z : Long_Float := Long_Float'Truncation (JD + 0.5);
      A : Long_Float;

   begin
      --  Portion of day since noon
      Result.F := JD + 0.5 - Z;

      if Z < 2299_161.0 then
         A := Z;
      else
         declare
            Alpha : Long_Float := Long_Float'Truncation
              ((Z - 1867_216.25)/36524.25);
         begin
            A := Z + 1.0 + Alpha - Long_Float'Truncation (Alpha / 4.0);
         end;
      end if;

      Result.B := A + 1524.0;
      Result.C := Long_Float'Truncation
        ((Result.B - 122.1) / 365.25);
      Result.D := Long_Float'Truncation
        (365.25 * Result.C);
      Result.E := Long_Float'Truncation
        ((Result.B - Result.D) / 30.6001);
      return Result;
   end Date_Parameters;

end Times;
