------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                                EPHEMERIS                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

-- Ada Wrapper for Swiss Ephemeris. (Partial)

with Ephemeris.Flags;
use Ephemeris.Flags;

with Interfaces.C;
use Interfaces, Interfaces.C;

with Ada.Unchecked_Conversion;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;
with Ada.Exceptions; use Ada.Exceptions;

package body Ephemeris is

   ------------
   -- Locals --
   ------------

   -- Message buffer size:
   Message_Size : constant := 256;

   -- Results of last angles calculation:
   Angles : aliased Angles_Array := (others => 0.0);

   -- Error code from last angles calculation:
   Angles_Code : Unsigned_32;

   --  Results of cusp calculations:
   type Cusps_Array is array (0 .. Cusps_Range'Last) of Long_Float;
   Cusps : aliased Cusps_Array := (others => 0.0);

   -- Results of last calculation:
   Results : aliased Result_Array := (others => 0.0);

   -- Flags from last calculation:
   Result_Flags : Unsigned_32;

   -- Last Error message:
   Result_String : Unbounded_String;


   procedure Set_Ephemeris_Path (Path : Char_Array);

#IF GNAT_WIN then
   pragma Import (DLL, Set_Ephemeris_Path,
                    External_Name => "swe_set_ephe_path");

#ELSIF OA_WIN then
   pragma Import (DLL_stdcall, Set_Ephemeris_Path,
                    External_Name => "swe_set_ephe_path");
#ELSE
   pragma Import (C, Set_Ephemeris_Path, "swe_set_ephe_path");
#END IF;
   type Calendar is (Julian, Gregorian);
   for Calendar use (Julian => 0, Gregorian => 1);

   -- Convert from a civil calendar date to a Julian day number and fraction:
   function Julian_Day (Year, Month, Day : Integer;
                        UT : Long_Float; Cal : Calendar)
                       return Long_Float;
#IF GNAT_WIN then
   pragma Import (DLL, Julian_Day, External_Name => "swe_julday");
#ELSIF OA_WIN then
   pragma Import (DLL_stdcall, Julian_Day, External_Name => "swe_julday");
#ELSE
   pragma Import (C, Julian_Day, "swe_julday");
#END IF;

   --  Calculate houses and cusps
   function Calc_Houses
     (JD_UT : Long_Float;
      Lat, Long : Long_Float;
      House_System : Integer;
      Cusps : access Cusps_Array;
      Angles : access Angles_Array) return Unsigned_32;
#IF GNAT_WIN then
   pragma Import (DLL, Calc_Houses, External_Name => "swe_houses");
#ELSIF OA_WIN then
   pragma Import (DLL_stdcall, Calc_Houses, External_Name => "swe_houses");
#ELSE
   pragma Import (C, Calc_Houses, "swe_houses");
#END IF;


   -- Get the numeric code representation for a house position system:
   function Code (S : House_Position_System) return Integer is
      type Unsigned_7 is mod (2 ** 7);
      for Unsigned_7'Size use 7;
      function Conv is new Ada.Unchecked_Conversion
        (House_Position_System, Unsigned_7);
   begin
      return Integer (Conv (S));
   end Code;

   -- Get the numeric code for a value of "Times.Months"
   function Code is new Ada.Unchecked_Conversion (Months, Month_Range);


   --------------------
   -- Initialization --
   --------------------

   -- Topocentric apparent coordinates:
   procedure Make_Apparent_Positions (Path : String := "") is
   begin
      -- Ephemeris and location:
      Use_Swiss;
      Set_Ephemeris_Path (To_C (Path));

      -- Position calculations:
      Use_Apparent_Positions;
      Generate_Velocities;
      Use_Nutation;

      -- Position format:
      Use_Polar;
      Use_Degrees;

      -- Coordinate Center:
      Use_Topocentric;

      -- Coordinate Plane and Reference Point:
      Use_Equatorial;
      Use_Tropical;

      -- Equinox:
      Use_Equinox_Of_Date;
   end Make_Apparent_Positions;

     -- Geocentric apparent coordinates:
     procedure Make_Conventional_Positions (Path: String := "") is
   begin
      -- Ephemeris and location:
      Use_Swiss;
      Set_Ephemeris_Path (To_C (Path));

      -- Position calculations:
      Use_Apparent_Positions;
      Generate_Velocities;
      Use_Nutation;

      -- Position format:
      Use_Polar;
      Use_Degrees;

      -- Coordinate Center:
      Use_Geocentric;

      -- Coordinate Plane and Reference Point:
      Use_Equatorial;
      Use_Tropical;

      -- Equinox:
      Use_Equinox_Of_Date;
   end Make_Conventional_Positions;


   -- Topocentric "True" equatorial coordinates.
   procedure Make_True_Positions (Path : String := "") is
   begin
      -- Ephemeris and location:
      Use_Swiss;
      Set_Ephemeris_Path (To_C (Path));

      -- Position calculations:
      Use_True_Positions;
      Generate_Velocities;
      Use_Nutation;

      -- Position format:
      Use_Polar;
      Use_Degrees;

      -- Coordinate Center:
      Use_Topocentric;

      -- Coordinate Plane and Reference Point:
      Use_Equatorial;
      Use_Tropical;

      -- Equinox:
      Use_Equinox_Of_Date;
   end Make_True_Positions;


   -------------
   -- Queries --
   -------------

   function Last_Angles return Angles_Array is
   begin
      return Angles;
   end Last_Angles;

   function Last_Angles_Code return Unsigned_32 is
   begin
      return Angles_Code;
   end Last_Angles_Code;

   function Last_Cusps (Cusp : Cusps_Range) return Long_Float is
   begin
      return Cusps (Cusp);
   end Last_Cusps;

   function Last_Results return Result_Array is
   begin
      return Results;
   end Last_Results;

   function Last_Result_Flags return Unsigned_32 is
   begin
      return Result_Flags;
   end Last_Result_Flags;

   function Last_Result_String return String is
   begin
      return To_String (Result_String);
   end Last_Result_String;


   -----------------
   -- Measurement --
   -----------------

   type Long_Lat_Array is array (1..2) of Long_Float;

   function House_Position
     (RAMC : Long_Float;
      Lat : Long_Float;
      Obliquity : Long_Float;
      System : Integer;
      Ecliptic_Coords : access Long_Lat_Array;
      Err_Msg : access Char_Array) return Long_Float;

#IF GNAT_WIN then
      pragma Import (DLL, House_Position, External_Name => "swe_house_pos");
#ELSIF OA_WIN then
      pragma Import (DLL_stdcall, House_Position, External_Name => "swe_house_pos");
#ELSE
      pragma Import (C, House_Position, "swe_house_pos");
#END IF;

   -- House position:
   function House_Position
     (RAMC : Long_Float;
      Geo_Latitude : Long_Float;
      Obliquity : Long_Float;
      System : House_Position_System;
      Ecliptic_Longitude : Long_Float;
      Ecliptic_Latitude : Long_Float) return Long_Float is

      Cstr : aliased Char_Array := (1..Message_Size => Nul);
      Ecliptic : aliased Long_Lat_Array :=
        (Ecliptic_Longitude, Ecliptic_Latitude);
      Result : Long_Float;

   begin
      Result :=
        House_Position
        (RAMC, Geo_Latitude, Obliquity, Code (System),
         Ecliptic'Access, Cstr'Access);

      if Cstr (1) /= Nul then
         Result_String := To_Unbounded_String (To_Ada (Cstr));
         return -1.0;
      else
         return 30.0 * (Result - 1.0);
      end if;
   end House_Position;


   --------------------
   -- Status Setting --
   --------------------

--    procedure Set_Ayanamsha (Mode : Ayanamshas);
--    procedure Set_Custom_Ayanamsha (Epoch, Ayanamsha : Long_Float);


   -----------------
   -- Conversions --
   -----------------

   function Julian_Day_From_Gregorian
     (D : Days;
      M : Months;
      Y : Years;
      UT : Long_Float) return Long_Float is
   begin
      pragma Assert (Valid_Date (D, M, Y));
      return Julian_Day (Y, Code (M), D, UT, Gregorian);
   end Julian_Day_From_Gregorian;


   function Julian_Day_From_Julian
     (D : Days;
      M : Months;
      Y : Years;
      UT : Long_Float) return Long_Float is
   begin
      return Julian_Day (Y, Code (M), D, UT, Julian);
   end Julian_Day_From_Julian;


   --  Calculate angles
   procedure Calculate_Angles (JD_UT : Long_Float; Long, Lat : Long_Float) is
   begin
      --  House system doesn't matter for angles
        Calculate_Houses (JD_UT, Long, Lat, Placidus);
   end Calculate_Angles;


   --  Calculate house cusps
   procedure Calculate_Houses
     (JD_UT : Long_Float;
      Long, Lat : Long_Float; System : House_Position_System) is
   begin
      --  House system doesn't matter for angles
      Angles_Code :=
        Calc_Houses (JD_UT, Lat, Long, Code (System),
                     Cusps'Access, Angles'Access);
      if Angles_Code /= 0 then
         raise Calculation_Failed;
      end if;
   exception
      when E : Calculation_Failed =>
         Raise_Exception (Exception_Identity (E),
                          "Cusps Calculation Failed");
   end Calculate_Houses;

   -- Call ephemeris calculation routine using body code and ET:
   procedure Calculate_ET (JD_ET : Long_Float; Body_Code : Integer) is

      function Calc_ET
        (JD_ET : Long_Float;
         Body_Code : Integer;
         Flags : Unsigned_32;
         Result : access Result_Array;
         Result_String : access Char_Array) return Unsigned_32;
#IF GNAT_WIN then
      pragma Import (DLL, Calc_ET, External_Name => "swe_calc");
#ELSIF OA_WIN then
      pragma Import (DLL_stdcall, Calc_ET, External_Name => "swe_calc");
#ELSE
      pragma Import (C, Calc_ET, "swe_calc");
#END IF;
      -- Message_Size is the designated buffer size for the error message:
      Cstr : aliased Char_Array := (1..Message_Size => nul);
      Flags : Unsigned_32 := Calculation_Flags;
   begin
      if Body_Code = -1 then
         --  Obliquity & Nutation
         Flags := 0;
      end if;

      Result_Flags :=
        Calc_ET (JD_ET, Body_Code, Flags, Results'Access,
              Cstr'access);
      Result_String := To_Unbounded_String (To_Ada (Cstr));

      if Result_Flags = 16#FFFF_FFFF# then
         raise Calculation_Failed;
      end if;
   exception
      when E : Calculation_Failed =>
         Raise_Exception (Exception_Identity (E),
                          To_String (Result_String));
   end Calculate_ET;

   -- Call ephemeris calculation routine using body code and UT:
   procedure Calculate_UT (JD_UT : Long_Float; Body_Code : Integer) is

      function Calc_UT
        (JD_UT : Long_Float;
         Body_Code : Integer;
         Flags : Unsigned_32;
         Result : access Result_Array;
         Result_String : access Char_Array) return Unsigned_32;
#IF GNAT_WIN then
      pragma Import (DLL, Calc_UT, External_Name => "swe_calc_ut");
#ELSIF OA_WIN then
      pragma Import (DLL_stdcall, Calc_UT, External_Name => "swe_calc_ut");
#ELSE
      pragma Import (C, Calc_UT, "swe_calc_ut");
#END IF;

      -- Message_Size is the designated buffer size for the error message:
      Cstr : aliased Char_Array := (1..Message_Size => nul);
   begin
      if Body_Code = -1 then
         --  Obliquity & Nutation.  Appears that swe_calc_ut doesn't
         --  work for this.  *** Need to convert UT to ET
         Calculate_ET (JD_UT, Body_Code);
      else
         begin
            Result_Flags :=
              Calc_UT (JD_UT, Body_Code, Calculation_Flags, Results'Access,
                       Cstr'access);

            Result_String := To_Unbounded_String (To_Ada (Cstr));

            if Result_Flags = 16#FFFF_FFFF# then
               raise Calculation_Failed;
            end if;
         exception
            when E : Calculation_Failed =>
               Raise_Exception (Exception_Identity (E),
                                To_String (Result_String));
         end;
      end if;
   end Calculate_UT;

end Ephemeris;
