------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                                EPHEMERIS                                 --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

-- Ada Wrapper for Swiss Ephemeris (partial)

with Interfaces; use Interfaces;
with Times; use Times;

package Ephemeris is

   --------------------
   -- Initialization --
   --------------------

   -- Topocentric apparent coordinates:
   procedure Make_Apparent_Positions (Path : String := "");

   -- Geocentric apparent coordinates:
   procedure Make_Conventional_Positions (Path: String := "");

   -- Topocentric "true" equatorial coordinates:
   procedure Make_True_Positions (Path : String := "");


   -------------
   -- Queries --
   -------------

   -- Sidereal Ayanamshas:
   type Ayanamshas is
     (Fagan_Bradley,
      Lahiri,
      DeLuce,
      Raman,
      Ushashashi,
      Krishnamurti,
      Djwal_Khul,
      Yukteshwar,
      JN_Bhasin,

      Babylonian_Kugler_1,
      Babylonian_Kugler_2,
      Babylonian_Kugler_3,
      Babylonian_Huber,
      Babylonian_ETPSC,

      Aldebaran_15_Tau,
      Hipparchos,
      Sassanian,
      Galactic_Center_0_Sag,
      Ayanamsha_J2000,
      Ayanamsha_J1900,
      Ayanamsha_B1950);


   -----------------
   -- Measurement --
   -----------------

   -- Greenwich Sidereal Time in Hours decimal:
   function GST (JD_UT : Long_Float) return Long_Float;

   function Delta_T (JD_UT : Long_Float) return Long_Float;

   --  Results of last angles calculation:
   type Angles_Array is array (1 .. 10) of Long_Float;
   function Last_Angles return Angles_Array;

   -- Error code from last angles calculation:
   function Last_Angles_Code return Unsigned_32;

   --  Results of last houses calculation
   subtype Cusps_Range is Positive range 1 ..12;
   function Last_Cusps (Cusp : Cusps_Range) return Long_Float;

   -- Result of last ephemeris calculation:
   type Result_Array is array (1..6) of Long_Float;
   function Last_Results return Result_Array;

   -- Ephemeris flags after last calculation:
   function Last_Result_Flags return Unsigned_32;

   -- Error message from last calculation:
   function Last_Result_String return String;

   -- House position systems:
   type House_Position_System is
     (Equal,
      Campanus,
      Koch,
      Placidus,
      Regiomontanus,
      Topocentric,
      Vehlow);

   -- House position:
   function House_Position
     (RAMC : Long_Float;
      Geo_Latitude : Long_Float;
      Obliquity : Long_Float;
      System : House_Position_System;
      Ecliptic_Longitude : Long_Float;
      Ecliptic_Latitude : Long_Float) return Long_Float;

   --------------------
   -- Status Setting --
   --------------------

   --    procedure Set_Ayanamsha (Mode : Ayanamshas);
   --    procedure Set_Custom_Ayanamsha (Epoch, Ayanamsha : Long_Float);

   -- Set the location for determining parallax-corrected positions.
   -- Alt (-itude) is in meters.  European convention (West < 0) is
   -- used for the geographic longitude.  You still need to call
   -- Ephemeris.Flags.Use_Topocentric:
   procedure Set_Topocentric_Parameters (Long, Lat, Alt : Long_Float);


   -----------------
   -- Conversions --
   -----------------

   -- Calendar to Julian Day Number Conversions:
   function Julian_Day_From_Gregorian
     (D : Days;
      M : Months;
      Y : Years;
      UT : Long_Float) return Long_Float;

   function Julian_Day_From_Julian
     (D : Days;
      M : Months;
      Y : Years;
      UT : Long_Float) return Long_Float;


   ----------------------
   -- Basic Operations --
   ----------------------

   --  Call ephemeris calculation routine:

   --  Raise when calculation fails
   Calculation_Failed : exception;

   --  Calculate angles
   procedure Calculate_Angles (JD_UT : Long_Float; Long, Lat : Long_Float);

   --  Calculate house cusps
   procedure Calculate_Houses
     (JD_UT : Long_Float;
      Long, Lat : Long_Float; System : House_Position_System);

   -- Versions using Ephemeris Time:
   procedure Calculate_ET (JD_ET : Long_Float; Body_Code : Integer);

   -- Versions taking Universal Time and converting to ET internally:
   procedure Calculate_UT (JD_UT : Long_Float; Body_Code : Integer);


   -- Close ephemeris and files:
   procedure Close;

private
   -- Ephemeris codes for Ayanamshas:
   for Ayanamshas use
      (Fagan_Bradley => 0,
       Lahiri => 1,
       DeLuce => 2,
       Raman => 3,
       Ushashashi => 4,
       Krishnamurti => 5,
       Djwal_Khul => 6,
       Yukteshwar => 7,
       JN_Bhasin => 8,

       Babylonian_Kugler_1 => 9,
       Babylonian_Kugler_2 => 10,
       Babylonian_Kugler_3 => 11,
       Babylonian_Huber => 12,
       Babylonian_ETPSC => 13,

       Aldebaran_15_Tau => 14,
       Hipparchos => 15,
       Sassanian => 16,
       Galactic_Center_0_Sag => 17,
       Ayanamsha_J2000 => 18,
       Ayanamsha_J1900 => 19,
       Ayanamsha_B1950 => 20);

   -- Ephemeris codes for house position systems:
   for House_Position_System use
      (Equal => Character'Pos ('A'),
       Campanus => Character'Pos ('C'),
       Koch => Character'Pos ('K'),
       Placidus => Character'Pos ('P'),
       Regiomontanus => Character'Pos ('R'),
       Topocentric => Character'Pos ('T'),
       Vehlow => Character'Pos ('V'));

#IF GNAT_WIN then
   pragma Import (Dll, GST, External_Name => "swe_sidtime");
   pragma Import (DLL, Set_Topocentric_Parameters,
                    External_Name => "swe_set_topo");
   pragma Import (Dll, Close, External_Name => "swe_close");
   pragma Import (Dll, Delta_T, External_Name => "swe_deltat");

#ELSIF OA_WIN then
   pragma Import (Dll_Stdcall, GST, External_Name => "swe_sidtime");
   pragma Import (dll_stdcall, Set_Topocentric_Parameters,
                    External_Name => "swe_set_topo");
   pragma Import (dll_stdcall, Close, External_Name => "swe_close");
   pragma Import (Dll_stdcall, Delta_T, External_Name => "swe_deltat");

#ELSE
   pragma Import (C, GST, External_Name => "swe_sidtime");
   pragma Import (C, Set_Topocentric_Parameters, "swe_set_topo");
   pragma Import (C, Close, "swe_close");
   pragma Import (C, Delta_T, "swe_deltat");
#END IF;

   pragma Inline (Last_Cusps);
end Ephemeris;
