------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                        BODY_SELECTOR_CONTROLLERS                         --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Strings; use Ada.Strings;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;
with Ada.Exceptions; use Ada.Exceptions;

with Glib; use Glib;
with Gtk.Label; use Gtk.Label;
with Gtk.Gentry; use Gtk.Gentry;
with Gtk.Clist; use Gtk.Clist;

with Body_Selector; use Body_Selector;
with Body_Selector_Views; use Body_Selector_Views;

--  Controller for body selector widget
package body Body_Selector_Controllers is

   ----------------------
   --  Initialization  --
   ----------------------

   --  Set up controller
   procedure Initialize
     (C : access Body_Selector_Controller;
      M : access Model_Obj'Class;
      V : access View_Obj'Class) is

   begin
      --  Call super initialization
      Initialize (Controller_Obj (C.all)'Access, M, V);

      C.Selector := Selector (Body_Selector_View (V.all)'Access);
   end Initialize;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Set selector state to model values
   procedure Reset_Selector_State (This : access Body_Selector_Controller) is
   begin
      This.Button_List_State := Basic (Body_Model (Model (This).all));
   end Reset_Selector_State;


   --  Update the selector state to reflect a change in one body
   procedure Update (This : access Body_Selector_Controller;
                     Button : access Gtk_Check_Button_Record'Class) is
      use Gtk;
      Body_String : Unbounded_String;
      Body_Id : Basic_Bodies;
      Label : Gtk_Label;
      Blank_Index : Integer;
   begin
      --  Do not execute before selector is fully built.  GTK+ unfortunately
      --  signals "toggled" when first associating signals to callbacks
      if This.Selector /= null then

         --  Extract string identifying the check button
         Label  := Gtk_Label (Get_Child (Button));
         Body_String :=
           To_Unbounded_String (Get (Label));

         --  Replace blank with underscore for labels with two words.
         --  Current assumption is that the body labels contain no more
         --  than one blank
         Blank_Index := Index (Body_String, " ");
         if Blank_Index /= 0 then
            Overwrite (Body_String, Blank_Index, "_");
         end if;

         --  Update temporary state for dialog
         Body_Id := Basic_Bodies'Value (To_String (Body_String));
         This.Button_List_State (Body_Id) := Is_Active (Button);
      end if;
   end Update;

   --  Add an extended body to the dialog
   procedure Add_Extended_Body (This : access Body_Selector_Controller) is
      S : String := Get_Text (This.Selector.Extended_Entry);
   begin
      --  Validate
      begin
         --  Assume catalog numbers are in the range 1 .. 40_000
         if Integer'Value (S) < 1 or else Integer'Value (S) > 40_000 then
            raise Invalid_Input;
         end if;
      exception
         when others =>
            raise Invalid_Input;
      end;

      --  Update dialog
      Add_Item (Body_Selector_View (View (This).all)'access, S);
   exception
      when E : Invalid_Input =>
         Raise_Exception (Exception_Identity (E), "Bad catalog number");
   end Add_Extended_Body;


   --  Delete selected extended bodies from the list
   procedure Delete_Extended_Bodies (This : access Body_Selector_Controller) is
   begin
      Delete_Selection (Body_Selector_View (View (This).all)'Access);
   end Delete_Extended_Bodies;


   --  Clear the extended bodies list in the dialog
   procedure Clear_Extended_Bodies (This : access Body_Selector_Controller) is
   begin
      Clear_Extended (Body_Selector_View (View (This).all)'Access);
   end Clear_Extended_Bodies;


   --  Commit extended list changes made in the dialog to the model
   procedure Update_All
     (This : access Body_Selector_Controller;
      Opening_Dialog : Boolean) is
      Num_Rows : Gint := Get_Rows (This.Selector.Extended_List);
   begin
      if not Opening_Dialog then
         Clear_Extended_Model (Body_Model (Model (This).all)'Access);
         for I in 1 .. Num_Rows loop
            --  Sweph codes for asteroids are 10_000 + catalog number.
            --  We store them using the sweph code
            Extend
              (Body_Model (Model (This).all)'Access,
               Integer'Value
               (Get_Text (This.Selector.Extended_List, I - 1, 0))
               + 10_000);
         end loop;
      end if;
      Update_All
        (Body_Model (Model (This).all)'Access, This.Button_List_State);
   end Update_All;

end Body_Selector_Controllers;
