------------------------------------------------------------------------------
--                                                                          --
--                              AstroFrames                                 --
--                                                                          --
--                          EVENT_MODELS.PERSISTENCE                        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Streams.Stream_IO; use Ada.Streams.Stream_IO;

--  Facility for saving and restoring events
package body Event_Models.Persistence is

   ---------------
   --  Queries  --
   ---------------

   --  Last filespec for the event
   function File_Spec (This : access Event_Model'Class) return String is
   begin
      return To_String (This.File_Spec);
   end File_Spec;

   --  Event status (true if modified and not saved to disk)
   function Modified (This : access Event_Model'Class) return Boolean is
   begin
      return This.Modified;
   end Modified;

   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Reset modified bit.  Called when an event is saved to disk
   procedure Reset_Modified (This : access Event_Model'Class) is
   begin
      This.Modified := False;
   end Reset_Modified;


   --  Set file spec for persisting event
   procedure Set_File_Spec
     (Model : access Event_Model'Class; Spec : String) is
   begin
      Model.File_Spec := To_Unbounded_String (Spec);
   end Set_File_Spec;

   --  Set modified bit.  Called when event has changed relative to last
   --  saved version
   procedure Set_Modified (This : access Event_Model'Class) is
   begin
      This.Modified := True;
   end Set_Modified;


   --  Persist event
   procedure Persist (This : access Event_Model'Class) is
      F : File_Type;
      S : Stream_Access;
   begin
      Create (F, Name => To_String (This.File_Spec));
      S := Stream (F);

      --  Write selected fields of event
      Unbounded_String'Write (S, This.Description);
      Date_Time_Components'Write (S, This.Date_Time);
      Location_Record'Write (S, This.Current_Location);
      Unbounded_String'Write (S, This.Notes);

      Close (F);
      This.Modified := False;
   end Persist;

   --  Restore an event from disk
   function Restore (Spec : String) return Event_Model_Ptr is
      F : File_Type;
      S : Stream_Access;
      Temp : Event_Model_Ptr := new Event_Model;
   begin
      Open (F, In_File, Spec);
      S := Stream (F);

      --  Restore event
      Unbounded_String'Read (S, Temp.Description);
      Date_Time_Components'Read (S, Temp.Date_Time);
      Location_Record'Read (S, Temp.Current_Location);
      Unbounded_String'Read (S, Temp.Notes);

      Temp.File_Spec := To_Unbounded_String (Spec);
      Temp.Modified := False;

      Close (F);

      return Temp;
   end Restore;

end Event_Models.Persistence;
