------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                            LOCATION_CONTROLLERS                          --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Exceptions; use Ada.Exceptions;

with Times.Strings; use Times.Strings;
with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Field_Validation; use Field_Validation;

with Gtk.Gentry; use Gtk.Gentry;
with Gtk.Check_Button; use Gtk.Check_Button;

with Main_Models; use Main_Models;
with Location_Views; use Location_Views;

with Message_Boxes; use Message_Boxes;

--  Controller for default location dialog
package body Location_Controllers is

   ----------------------
   --  Implementation  --
   ----------------------

   --  Extract location strings and update model
   procedure Set_Location (This : access Location_Controller);

   --  Extract whether an event should be generated when starting up
   procedure Set_Startup_Event (This : access Location_Controller);

   --  Extract time zone
   procedure Set_Zone (This : access Location_Controller);

   pragma Inline (Set_Zone);


   ----------------------
   --  Initialization  --
   ----------------------

   --  Set up controller
   procedure Initialize
     (C : access Location_Controller;
      M : access Model_Obj'Class;
      V : access View_Obj'Class) is
   begin
      --  Call super initialization operation
      Initialize (Controller_Obj (C.all)'Access, M, V);
      C.Dialog := Dialog (Location_View (V.all)'Access);
      Update_All (Main_Model (M.all)'Access);
   end Initialize;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Close the location specification dialog
   procedure Close_Dialog (This : access Location_Controller) is
   begin
      Hide_All (This.Dialog);
   end Close_Dialog;

   --  Display an event specification dialog
   procedure Open_Dialog (This : access Location_Controller) is
   begin
      Show_All (This.Dialog);
   end Open_Dialog;

   --  Update model and all active views
   procedure Update_All  (This : access Location_Controller) is
   begin
      Set_Location (This);
      Set_Zone (This);
      Set_Startup_Event (This);

      Update_All (Main_Model (Model (This).all)'Access);

      Close_Dialog (This);

   exception
      --  Intercept bad input and pop up a message box
      when E : Invalid_Longitude | Invalid_Latitude |
        Invalid_Altitude | Invalid_Time  =>
         Invalid_Input_Box (Exception_Message (E));

   end Update_All;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Extract and validate location strings, and update model
   procedure Set_Location (This : access Location_Controller) is
      Long_I, Long_M, Lat_I, Lat_M : Integer;
      Long_S, Lat_S : Long_Float;
   begin
      --  Validate and convert longitude strings
      Long_I := Integer'Value
        (Validate_Longitude (Get_Text (This.Dialog.Long_Degree)));
      Long_M := Minutes_Part'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Long_Minute)));
      Long_S := Long_Float'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Long_Second)));

      if Validate_EW (Get_Text (This.Dialog.Long_Direction)) = "West" then
         Long_I := - Long_I;
         Long_M := - Long_M;
         Long_S := - Long_S;
      end if;

      --  Validate and convert latitude strings
      Lat_I := Integer'Value
        (Validate_Latitude (Get_Text (This.Dialog.Lat_Degree)));
      Lat_M := Minutes_Part'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Lat_Minute)));
      Lat_S := Long_Float'Value
        (Validate_Sexagesimal (Get_Text (This.Dialog.Lat_Second)));

      if Validate_NS (Get_Text (This.Dialog.Lat_Direction)) = "South" then
         Lat_I := - Lat_I;
         Lat_M := - Lat_M;
         Lat_S := - Lat_S;
      end if;

      --  Set location for event
      Set_Default_Location (This => Main_Model (Model (This).all)'Access,

                            Location =>
                              (Long  => To_Decimal (
                                         Make_Sexagesimal
                                         (Long_I, Long_M, Long_S)),
                               Lat => To_Decimal (
                                       Make_Sexagesimal
                                       (Lat_I, Lat_M, Lat_S)),
                               Alt => Long_Float'Value
                                 (Validate_Altitude
                                  (Get_Text (This.Dialog.Alt_Entry))),
                               Dynamic_Angle => 0.0));

   exception
      when E : Invalid_Longitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Longitude Entered");

      when E : Invalid_Latitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Latitude Entered");

      when E : Invalid_Altitude  =>
         Raise_Exception (Exception_Identity (E), "Bad Altitude Entered");
   end Set_Location;

   --  Extract whether an event should be generated when starting up
   procedure Set_Startup_Event (This : access Location_Controller) is
   begin
      Set_Startup_Event
        (Main_Model (Model (This).all)'Access,
         Is_Active (This.Dialog.Startup_Event_Button));
   end Set_Startup_Event;

   --  Extract and set time zone
   procedure Set_Zone (This : access Location_Controller) is
   begin
      Set_Default_Zone (Main_Model (Model (This).all)'Access,
                        Zone_Value (Get_Text (This.Dialog.Zone_Hour),
                                    Get_Text (This.Dialog.Zone_Minute),
                                    Get_Text (This.Dialog.Zone_Direction)));
   end Set_Zone;

end Location_Controllers;
