------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                             MAIN_CONTROLLERS                             --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                             $Revision: 1.4 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Main_Views; use Main_Views;
with Main_Models.Persistence;
use Main_Models, Main_Models.Persistence;
with Main_Window_Pkg; use Main_Window_Pkg;

with Event_Views; use Event_Views;
with Event_Models.Persistence; use Event_Models.Persistence;

with Positions_Views; use Positions_Views;
with Primaries_Views; use Primaries_Views;

with Ephemeris; use Ephemeris;

with Message_Boxes; use Message_Boxes;

with Gtk.Main;
with Glib; use Glib;
with Gtk.Enums; use Gtk.Enums;
with Gtk.Check_Menu_Item; use Gtk.Check_Menu_Item;
with Gtkada.File_Selection; use Gtkada.File_Selection;
with Gtkada.Dialogs; use Gtkada.Dialogs;

with Ada.Exceptions; use Ada.Exceptions;
with GNAT.Os_Lib; use GNAT.OS_Lib;

--  Main window controller.  The main window is a list of "events"
--  being worked with
package body Main_Controllers is

   ----------------------
   --  Implementation  --
   ----------------------

   --  Apply an operation to all selected events.  "Do_All" returns "True" if
   --  successful
   type Action is access
     procedure (This : access Main_Controller; Event : Event_Model_Ptr);

   function Do_All (This : access Main_Controller; Operation : Action)
                   return Boolean;

   --  Delete an event
   procedure Delete (This : access Main_Controller; Event : Event_Model_Ptr);

   --  Display all user-enabled views for an event
   procedure Display (This : access Main_Controller; Event : Event_Model_Ptr);

   --  Open the specification dialog for an event
   procedure Modify (This : access Main_Controller; Event : Event_Model_Ptr);

   --  Open the primary directions list dialog for an event
   procedure Primary_Directions
     (This : access Main_Controller; Event : Event_Model_Ptr);

   --  Save an event
   procedure Save (This : access Main_Controller; Event : Event_Model_Ptr);

   --  Display an information dialog
   procedure Show_Information;

   --  Update any displays for all events
   procedure Update_Event_Views (This : access Main_Controller);


   ----------------------
   --  Initialization  --
   ----------------------

   procedure Initialize (C : access Main_Controller;
                         M : access Model_Obj'Class;
                         V : access View_Obj'Class) is
   begin
      --  Super initialization operation
      Initialize (Controller_Obj (C.all)'Access, M, V);
      C.Window := Window (Main_View (V.all)'Access);
   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   --  Access the view that created this controller
   function Get_View (Controller : access Controller_Obj'Class) return
     Main_View_Ptr is
   begin
      return Main_View (View (Controller).all)'Access;
   end Get_View;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Clone an event
   procedure Clone
     (This : access Main_Controller;
      Event : Event_Model_Ptr) is
      Temp : Event_Model_Ptr := new Event_Model;
   begin
      --  Create a new event model based on "Event"
      Event_Models.Clone
        (Temp,
         Main_Model (Model (This).all)'Access,
         Event);

      --  Create its data entry view and controller
      Event_Views.Initialize (new Event_View, Temp);

      --  Create a view and controller for its table of positions
      Positions_Views.Initialize (new Positions_View, Temp);

      --  Create view and controller for primary directiosn
      Primaries_Views.Initialize (new Primaries_View, Temp);

      --  Add the new event to main's list of active events
      Extend (Main_Model (Model (This).all)'Access, Temp);
   end Clone;


   --  Make a copy of each selected event and add it to the main list
   procedure Clone_Selected_Events (This : access Main_Controller) is
   begin
      --  Iterate "Clone" over selected events with "Do_All"
      if Do_All (This, Clone'Access) then
         --  Unselect rows and update main display
         Unselect_All (This.Window.Event_Table);
         Update_All (Main_Model (Model (This).all)'Access);
      end if;
   end Clone_Selected_Events;


   --  Delete the selected events
   procedure Delete_Selected_Events (This : access Main_Controller) is
   begin
      --  Iterate "Delete" over selected events using "Do_All"
      if Do_All (This, Delete'Access) then
         --  Unselect rows and update display
         Unselect_All (This.Window.Event_Table);
         Update_All (Main_Model (Model (This).all)'Access);
      end if;
   end Delete_Selected_Events;


   --  Display the user-enabled views for the selected bodies
   procedure Display_Selected_Events (This : access Main_Controller) is
   begin
      --  Iterate "Display" over selected events using "Do_All"
      if Do_All (This, Display'Access) then
         --  Leave events selected
         null;
      end if;
   exception
      when E : Calculation_Failed =>
         Calculation_Error_Box (Exception_Message (E));
   end Display_Selected_Events;


   --  Open primaries filtering preferences dialog
   procedure Filter_Primaries (This : access Main_Controller) is
   begin
      Display_Filter_Primaries_Dialog (Get_View (This));
   end Filter_Primaries;


   --  Modify the selected events.  Open the event specification dialog
   --  for each event
   procedure Modify_Selected_Events (This : access Main_Controller) is
   begin
      --  Iterate "Modify" over selected events using "Do_All"
      if Do_All (This, Modify'Access) then
         --  Leave events selected
         null;
      end if;
   end Modify_Selected_Events;


   --  Create a new event and open its specification dialog
   procedure New_Event (This : access Main_Controller) is
      Temp : Event_Model_Ptr := new Event_Model;
   begin
      --  Initialize a model for the new event
      Event_Models.Initialize
        (Temp,
         Main_Model (Model (This).all)'Access, Use_Default_Location => True);

      --  Create its data entry view and controller
      Event_Views.Initialize (new Event_View, Temp);

      --  Create its positions table view and controller
      Positions_Views.Initialize (new Positions_View, Temp);

      --  Create view and controller for primary directiosn
      Primaries_Views.Initialize (new Primaries_View, Temp);

      --  Add the event model to main model's list of events
      Extend (Main_Model (Model (This).all)'Access, Temp);

      --  Update main display with the incomplete event description
      Update_All (Main_Model (Model (This).all)'Access);

      --  Display the data entry screen
      Update_All (Temp, new Modification_Event);
   end New_Event;


   --  Open primary directions list dialog
   procedure Primary_Directions_Event (This : access Main_Controller) is
   begin
      --  Iterate "Primary_Directions" over selected events using "Do_All"
      if Do_All (This, Primary_Directions'Access) then
         --  Leave events selected
         null;
      end if;
   exception
      when E : Calculation_Failed =>
         Calculation_Error_Box (Exception_Message (E));
   end Primary_Directions_Event;


   --  Quit the AstroFrames application
   procedure Quit_Event (This : access Main_Controller) is
      Action : Message_Dialog_Buttons;
      Any_Modified : Boolean := False;
      Event : Event_Model_Ptr;
   begin
      --  Check for one or more modified events.  GTK+ widget selected rows
      --  are indexed from 0
      for Row in 0 .. Get_Rows (This.Window.Event_Table) - 1 loop
         Event := Get (This.Window.Event_Table, Row);
         if Modified (Event) then
            Any_Modified := True;
            exit;
         end if;
      end loop;

      if Any_Modified then
         --  Prompt for decision whether to save all modified events
         Action := Quit_Option_Box;
         case Action is
            when Button_Cancel =>
               return;
            when Button_Yes =>
               Save_Modified_Events (This);
            when others =>
               null;
         end case;
      end if;

      --  Persist AstroFrames preferences
      Persist (Main_Model (Model (This).all)'Access);

      --  Exit main GTK+ event loop
      Gtk.Main.Main_Quit;
   end Quit_Event;


   --  Restore an event from disk
   procedure Restore_Event (This : access Main_Controller) is
      Spec : String :=
        File_Selection_Dialog
        (Title => "Select file to open for event",
         Default_Dir =>
           Event_Directory (Main_Model (Model (This).all)'Access));
   begin
      if not Gnat.OS_Lib.Is_Regular_File (Spec) then
         Invalid_Input_Box ("No file selected");
      else
         declare
            Temp : Event_Model_Ptr;
         begin
            --  Get event data from file.  Catch exception in block
            --  handler if configuration file is from an incompatible
            --  version of the program (or is some other kind of file)
            Temp :=  Restore (Spec);

            --  Put a reference to the main window model into the event
            --  model
            Event_Models.Initialize (Temp, Model (This));

            --  Create its data entry view and controller
            Event_Views.Initialize (new Event_View, Temp);

            --  Create its view and controller for a table of positions
            Positions_Views.Initialize (new Positions_View, Temp);

            --  Create view and controller for primary directiosn
            Primaries_Views.Initialize (new Primaries_View, Temp);

            --  Add the new event to main's list and update main's display
            Extend (Main_Model (Model (This).all)'Access, Temp);
            Update_All (Main_Model (Model (This).all)'Access);
       end;
      end if;
   exception
      when others =>
         Invalid_Input_Box ("Invalid data file");
   end Restore_Event;


   --  Save all modified events to disk
   procedure Save_Modified_Events (This : access Main_Controller) is
      Event : Event_Model_Ptr;
   begin
      --  Save each modified event
      for Row in 0 .. Get_Rows (This.Window.Event_Table) - 1 loop
         Event := Get (This.Window.Event_Table, Row);
         if Modified (Event) then
            Save (This, Event);
         end if;
      end loop;

      --  Update "status" indicator in main window
      Update_All (Main_Model (Model (This).all)'Access);
   end Save_Modified_Events;


   --  Save the selected events to disk
   procedure Save_Selected_Events (This : access Main_Controller) is
   begin
      --  Iterate "Save" over selected events using "Do_All"
      if Do_All (This, Save'Access) then
         --  Unselect rows and update display
         Unselect_All (This.Window.Event_Table);
         Update_All (Main_Model (Model (This).all)'Access);
      end if;
   end Save_Selected_Events;


   --  Select topocentric apparent coordinates
   procedure Select_Apparent_Coordinates (This : access Main_Controller) is
   begin
      Set_Coordinates (Main_Model (Model (This).all)'Access,
                       Topocentric_Apparent);
      Update_Event_Views (This);
   end Select_Apparent_Coordinates;

   --  Display the body selection dialog
   procedure Select_Bodies_Event
     (This  : access Main_Controller) is
   begin
      Display_Bodies_Dialog (Get_View (This));
   end Select_Bodies_Event;

   --  Select Conventional Chart display
   procedure Select_Conventional_Chart (This : access Main_Controller) is
   begin
      Select_Display
        (Main_Model (Model (This).all)'Access,
         Conventional_Chart,
         Get_Active (This.Window.Conventional_Chart));
   end Select_Conventional_Chart;

   --  Select geocentric apparent coordinates
   procedure Select_Conventional_Coordinates (This : access Main_Controller) is
   begin
      Set_Coordinates (Main_Model (Model (This).all)'Access,
                       Conventional);
      Update_Event_Views (This);
   end Select_Conventional_Coordinates;

   --  Select Lunar Zodiac display
   procedure Select_Lunar_Zodiac (This : access Main_Controller) is
   begin
      Select_Display
        (Main_Model (Model (This).all)'Access,
         Lunar_Zodiac,
         Get_Active (This.Window.Lunar_Zodiac));
   end Select_Lunar_Zodiac;

   --  Select Solar Zodiac display
   procedure Select_Solar_Zodiac (This : access Main_Controller) is
   begin
      Select_Display
        (Main_Model (Model (This).all)'Access,
         Solar_Zodiac,
         Get_Active (This.Window.Solar_Zodiac));
   end Select_Solar_Zodiac;

   --  Select Terrestrial Zodiac display
   procedure Select_Terrestrial_Zodiac (This : access Main_Controller) is
   begin
      Select_Display
        (Main_Model (Model (This).all)'Access,
         Terrestrial_Zodiac,
         Get_Active (This.Window.Terrestrial_Zodiac));
   end Select_Terrestrial_Zodiac;

   --  Select Table of Positions display
   procedure Select_Table_Of_Positions (This : access Main_Controller) is
   begin
      Select_Display
        (Main_Model (Model (This).all)'Access,
         Table_Of_Positions,
         Get_Active (This.Window.Table_Of_Positions));
   end Select_Table_Of_Positions;

   --  Select geocentric apparent coordinates
   procedure Select_True_Coordinates (This : access Main_Controller) is
   begin
      Set_Coordinates (Main_Model (Model (This).all)'Access,
                       Topocentric_True);
      Update_Event_Views (This);
   end Select_True_Coordinates;

   --  Set the default location
   procedure Set_Default_Location (This : access Main_Controller) is
   begin
      Display_Location_Dialog (Get_View (This));
   end Set_Default_Location;


   --  Set the root directory for saving and restoring events
   procedure Set_Event_Directory (This : access Main_Controller) is
      Event_Dir : String :=
        File_Selection_Dialog
        (Title => "Select Root Directory for Event Storage",
         Default_Dir =>
           Event_Directory (Main_Model (Model (This).all)'Access),
        Dir_Only => True);
   begin
      --  The second condition is probably not strictly necessary,
      --  as "File_Selector" should return a directory when
      --  "Dir_Only" is set.  Did not use Gnat.Os_Lib.Is_Directory
      --  because it fails for undetermined reasons on windows
      if Event_Dir /= "" and then
        Event_Dir (Event_Dir'Last) = Gnat.Os_Lib.Directory_Separator then
         Set_Event_Directory (Main_Model (Model (This).all)'Access, Event_Dir);
      else
         Invalid_Input_Box ("No valid directory selected");
      end if;
   end Set_Event_Directory;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Iterate over all rows selected in the dialog event list,
   --  applying "Operation" to them
   function Do_All (This : access Main_Controller; Operation : Action)
                   return Boolean is
      use Gtk.Enums.Gint_List;
      Row_List : Glist := First (Get_Selection (This.Window.Event_Table));
      Row : Gint;
   begin
      if Get_Rows (This.Window.Event_Table) = 1 then
         --  Special case when only one event in the list: selection
         --  is unnecessary
         Operation.all (This, Get (This.Window.Event_Table, 0));
         return True;
      elsif Length (Row_List) = 0 then
         --  Inform user that one or more rows must be selected
         Show_Information;
         return False;
      else
         --  Apply the operation to each selected event
         for I in 1 .. Length (Row_List) loop
            Row := Nth_Data (Row_List, I - 1);
            Operation.all (This, Get (This.Window.Event_Table, Row));
         end loop;
         return True;
      end if;
   end Do_All;


   --  Display an information dialog
   procedure Show_Information is
   begin
      Information_Box
        (Message => "Select one or more events on which to operate.",
         Help_Message =>
           "All toolbar operations in this window except New and Open " &
         ASCII.LF &
         "require selection of one or more events on which to perform.");
   end Show_Information;

   --  Delete an event
   procedure Delete (This : access Main_Controller; Event : Event_Model_Ptr) is
   begin
      Remove (Main_Model (Model (This).all)'Access, Event);

      -- Close all views on the deleted event
      Set_Views_Undisplayable (Event);
   end Delete;

   --  Display the user-enabled views for an event
   procedure Display
     (This : access Main_Controller; Event : Event_Model_Ptr) is
   begin
      Set_Views_Displayable (Event);
   end Display;

   --  Open the specification dialog for an event
   procedure Modify
     (This : access Main_Controller; Event : Event_Model_Ptr) is
   begin
      Update_All (Event,
                  new Modification_Event,
                  Suppress_Main_Update => True);
   end Modify;

   --  Open the primary directions list dialog for an event
   procedure Primary_Directions
     (This : access Main_Controller; Event : Event_Model_Ptr) is
   begin
      --  Have event open the dialog
      Update_All (Event,
                  new Primaries_Open_Event,
                  Suppress_Main_Update => True);
   end Primary_Directions;


   --  Save an event
   procedure Save
     (This : access Main_Controller; Event : Event_Model_Ptr) is
   begin
      if File_Spec (Event) = "" then
         --  Attempt to get a file specification
         declare
            Spec : String :=
              File_Selection_Dialog
              (Title => "Designate file to save " &
                 Description (Event),
               Default_Dir =>
                 Event_Directory (Main_Model (Model (This).all)'Access));
         begin
            if Spec /= "" and then
              Spec (Spec'Last) /= GNAT.Os_Lib.Directory_Separator then

               Set_File_Spec (Event, Spec);
            else
               Invalid_Input_Box
                 ("Event not saved: invalid file specification");
            end if;
         end;
      end if;

      if File_Spec (Event) /= "" then
         Persist (Event);
      end if;
   end Save;

   --  Update any open displays for all events
   procedure Update_Event_Views (This : access Main_Controller) is
      Event : Event_Model_Ptr;
   begin
      --  Update open views for all events
      for Row in 0 .. Get_Rows (This.Window.Event_Table) - 1 loop
         Event := Get (This.Window.Event_Table, Row);
         Update_All (Event, new Event_Models.Update_Event);
      end loop;
   end Update_Event_Views;

end Main_Controllers;
