------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                               MAIN_MODELS                                --
--                                                                          --
--                                B o d y                                   --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ephemeris;

--  Model of the main window event list
package body Main_Models is

   ----------------------
   --  Initialization  --
   ----------------------

   --  Set a reference to the list of bodies selected for display in event
   --  views
   procedure Initialize (This : access Main_Model; Bodies : Body_Model_Ptr) is
   begin
      This.Body_Model := Bodies;

      --  Set ephemeris to proper coordinates mode
      Set_Coordinates (This, This.Coordinates);
   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   --  Selected coordinate system
   function Coordinate_System (This : access Main_Model)
                              return Coordinate_Systems is
   begin
      return This.Coordinates;
   end Coordinate_System;

   --  Reference to selected bodies model
   function Body_Model (This : access Main_Model) return Body_Model_Ptr is
   begin
      return This.Body_Model;
   end Body_Model;


   --  Default location
   function Default_Location (This : access Main_Model)
                             return Location_Record is
   begin
      return This.Default_Location;
   end Default_Location;


   --  Default Time Zone
   function Default_Time_Zone (This : access Main_Model) return Time_Zone is
   begin
      return This.Default_Time_Zone;
   end Default_Time_Zone;

   --  Display type selected?
   function Displayable (This : access Main_Model; Display : Displays)
                        return Boolean is
   begin
      return This.Selected_Displays (Display);
   end Displayable;


   --  Get the path of the root directory for storing events
   function Event_Directory (This : access Main_Model) return String is
   begin
      return To_String (This.Root_Event_Directory);
   end Event_Directory;


   --  Cursor off the list of events?
   function Off (This : access Main_Model) return Boolean is
   begin
      return Off (This.Events_List);
   end Off;

   --  The event at the list cursor
   function Item (This : access Main_Model) return Event_Model_Ptr is
      pragma Assert
        (not Off (This.Events_List),
         "Attempt to get item when iterator is off list of events");
   begin
      return Item (This.Events_List);
   end Item;

   --  Generate an event for the time of program startup?
   function Startup_Event (This : access Main_Model) return Boolean is
   begin
      return This.Startup_Event;
   end Startup_Event;



   -------------------------
   --  Basic Operations   --
   -------------------------

   --  Add an event to the end of the list
   procedure Extend
     (This : access Main_Model; E : Event_Model_Ptr) is
   begin
      Extend (This.Events_List, E);
   end Extend;

   --  Advance the cursor one position
   procedure Forth (This : access Main_Model) is
   begin
      Forth (This.Events_List);
   end Forth;

   --  Delete the specified event.  It's assumed that there are no
   --  duplicate items on the list, so only one event is removed
   procedure Remove (This : access Main_Model; E : Event_Model_Ptr) is
   begin
      Start (This);
      while not Off (This) and then Item (This) /= E loop
         Forth (This);
      end loop;

      if not Off (This) then
         Remove (This.Events_List);
      end if;
   end Remove;

   --  Select or deselect a display kind
   procedure Select_Display
     (This : access Main_Model;
      Display : Displays;
      Selected : Boolean := True) is
   begin
      This.Selected_Displays (Display) := Selected;
   end Select_Display;


   --  Set the coordinate system for calculations
   procedure Set_Coordinates
     (This : access Main_Model;
      Coordinates : Coordinate_Systems) is

      use Ephemeris;
   begin
      This.Coordinates := Coordinates;
      case Coordinates is
         when Conventional => Make_Conventional_Positions;
         when Topocentric_Apparent => Make_Apparent_Positions;
         when Topocentric_True => Make_True_Positions;
      end case;
   end Set_Coordinates;

   --  Set the default location
   procedure Set_Default_Location
     (This : access Main_Model;
      Location : Location_Record;
      Startup_Event : Boolean := False) is
   begin
      This.Default_Location := Location;
      This.Startup_Event := Startup_Event;
   end Set_Default_Location;


   --  Set the root directory for saving events
   procedure Set_Event_Directory (This : access Main_Model; Path : String) is
   begin
      This.Root_Event_Directory := To_Unbounded_String (Path);
   end Set_Event_Directory;

   --  Set default time zone
   procedure Set_Default_Zone
     (This : access Main_Model;
      Zone : Time_Zone) is
   begin
      This.Default_Time_Zone := Zone;
   end Set_Default_Zone;

   --  Set whether a startup event should be generated at program startup
   procedure Set_Startup_Event
     (This : access Main_Model; Generate : Boolean) is
   begin
      This.Startup_Event := Generate;
   end Set_Startup_Event;


   --  Move cursor to the start of the list
   procedure Start (This: access Main_Model) is
   begin
      Start (This.Events_List);
   end Start;


   --  Update the main window
   procedure Update_All (This : access Main_Model) is
   begin
      Notify_Observers (This, new Update_Event);
   end Update_All;


end Main_Models;
