------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                               MAIN_MODELS                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with MVC; use MVC;
with Publish_Subscribe; use Publish_Subscribe;
with Times; use Times;
with Locations; use Locations;

with Lists;
with Event_Models; use Event_Models;
with Body_Models; use Body_Models;

with Gnat.OS_Lib;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;

--  Model for main window list of events.
--  This model has a single implicit iterator available in its interface,
--  as only a singleton controller for the main window will
--  use it
package Main_Models is
   type Main_Model is new Model_Obj with private;
   type Main_Model_Ptr is access all Main_Model'Class;

   --  Event indicating the need to update display of the event list
   type Update_Event is new Event with private;

   --  Available coordinate systems.  Apparent coordinates are "visual" in
   --  that they adjust for light-time and aberration.  True coordinates
   --  are spatial.  Geocentric are the conventional coordinates used by
   --  astrologers.  Topocentric are local, being adjusted for parallax.
   type Coordinate_Systems is
     (Conventional,          --  Geocentric apparent
      Topocentric_Apparent,
      Topocentric_True);     --  No adjustment for light-time, aberration

   --  Available displays of events.  Selected displays are shown
   --  when an event is to be displayed
   type Displays is
     (Table_Of_Positions, Conventional_Chart, Solar_Zodiac,
      Lunar_Zodiac, Terrestrial_Zodiac);

   ----------------------
   --  Initialization  --
   ----------------------

   --  Set a reference to the selected bodies model, for passing to event views
   procedure Initialize (This : access Main_Model; Bodies : Body_Model_Ptr);


   ---------------
   --  Queries  --
   ---------------

   --  The selected bodies model
   function Body_Model (This : access Main_Model) return Body_Model_Ptr;

   --  Selected coordinate system
   function Coordinate_System (This : access Main_Model)
                              return Coordinate_Systems;

   --  Default location.  New event dialogs are initialized with this location
   function Default_Location (This : access Main_Model) return Location_Record;

   --  Default Time Zone
   function Default_Time_Zone (This : access Main_Model) return Time_Zone;

   --  Display type selected?
   function Displayable (This : access Main_Model; Display : Displays)
     return Boolean;

   --  Get the path of the root directory for storing events
   function Event_Directory (This : access Main_Model) return String;

   --  The event at the current list cursor position
   function Item (This : access Main_Model) return Event_Model_Ptr;

   --  Cursor off the list of events?
   function Off (This : access Main_Model) return Boolean;

   --  Generate event for the time of program startup?
   function Startup_Event (This : access Main_Model) return Boolean;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Add an event to the end of the list
   procedure Extend (This : access Main_Model; E : Event_Model_Ptr);

   --  Advance the list cursor one position
   procedure Forth (This : access Main_Model);

   --  Delete the specified event from the list
   procedure Remove (This : access Main_Model; E : Event_Model_Ptr);

   --  Select or deselect a display kind
   procedure Select_Display
     (This : access Main_Model;
      Display : Displays;
      Selected : Boolean := True);

   --  Set the coordinate system for calculations
   procedure Set_Coordinates
     (This : access Main_Model;
      Coordinates : Coordinate_Systems);

   --  Set the default location
   procedure Set_Default_Location
     (This : access Main_Model;
      Location : Location_Record;
      Startup_Event : Boolean := False);

   --  Set the default time zone
   procedure Set_Default_Zone
     (This : access Main_Model;
      Zone : Time_Zone);

   --  Set the root directory for saving events
   procedure Set_Event_Directory (This : access Main_Model; Path : String);

   --  Set whether a program startup event should be generated
   procedure Set_Startup_Event (This : access Main_Model; Generate : Boolean);

   --  Set the cursor to the start of the list of events
   procedure Start (This: access Main_Model);

   --  Update the main window
   procedure Update_All (This : access Main_Model);

private
   --  List of events
   package Event_Lists is new Lists (Event_Model'Class, Event_Model_Ptr);
   use Event_Lists;

   --  Selected Displays
   type Display_Array is array (Displays) of Boolean;

   type Main_Model is new Model_Obj with record
      Events_List :  List;
      Root_Event_Directory : Unbounded_String
        := To_Unbounded_String ("" & GNAT.OS_Lib.Directory_Separator);
      Default_Location : Location_Record;
      Default_Time_Zone : Time_Zone := 0.0;
      Startup_Event : Boolean := False;
      Body_Model : Body_Model_Ptr;
      Selected_Displays : Display_Array :=
        (Table_Of_Positions => True, others => False);
      Coordinates : Coordinate_Systems := Conventional;
   end record;

   type Update_Event is new Event with null record;

   pragma Inline (Initialize);
   pragma Inline (Body_Model, Event_Directory,
                  Default_Location, Startup_Event);
   pragma Inline (Off, Item, Forth, Start, Extend);
   pragma Inline (Update_All);
   pragma Inline (Set_Event_Directory, Set_Default_Location,
                  Set_Startup_Event, Default_Time_Zone,
                  Set_Default_Zone);
   pragma Inline (Displayable, Select_Display, Coordinate_System);
end Main_Models;
