------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                             POSITIONS_VIEWS                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Event_Models; use Event_Models;
with Main_Models;
with Astroframes_Intl; use Astroframes_Intl;

with Times.Strings; use Times, Times.Strings;
with Locations.Strings; use Locations.Strings;

with Reporters.Header;
with Reporters.Ecliptic;
with Reporters.Equator;
with Reporters.Terra;

with Glib; use Glib;
with Gtk.Clist; use Gtk.Clist;

with Ada.Text_IO; use Ada.Text_IO;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;
use Ada.Strings;

--  View for table of positions display
package body Positions_Views is

   ----------------------
   --  Implementation  --
   ----------------------

   --  Dump current positions to a file
   procedure Export_Positions (
                    This : access Positions_View;
                    Model : access Event_Model'Class;
                    Evt: access Export_Event'Class);

   --  Write event description
   procedure Write_Event_Info
     (F : File_Type; Event : access Event_Model'Class);


   -----------------------
   --  Initialization   --
   -----------------------

   procedure Initialize
     (This : access Positions_View;
      M : access Model_Obj'Class)
   is
   begin
      --  Super initialization operation
      MVC.Initialize (View_Obj (This.Viewer)'Access, M);
      Add_Observer (M, View_Obj (This.Viewer)'Access);

      --  Create position table dialog and controller
      Gtk_New (This.Dialog);
      This.Controller := new Positions_Controller;

      --  Initialize position table dialog
      Positions_Dialog_Pkg.Initialize
        (This.Dialog, Controller_Ptr'(This.Controller.all'Access));

      --  Create and initialize views for components of positions table
      This.Header := new Reporters.Header.Reporter;
      Reporters.Initialize (This.Header, This);
      This.Ecliptic := new Reporters.Ecliptic.Reporter;
      Reporters.Initialize (This.Ecliptic, This);
      This.Equator := new Reporters.Equator.Reporter;
      Reporters.Initialize (This.Equator, This);
      This.Terra := new Reporters.Terra.Reporter;
      Reporters.Initialize (This.Terra, This);

      --  Initialize the dialog controller
      Positions_Controllers.Initialize
        (This.Controller, M, This.Viewer'Access);

   end Initialize;


   ---------------
   --  Queries  --
   ---------------

   --  Reference to positions table dialog
   function Dialog (This : access Positions_View)
                   return Positions_Dialog_Access is
   begin
      return This.Dialog;
   end Dialog;

   --  Reference to positions table dialog
   function Dialog
     (This : access View_Role)
     return Positions_Dialog_Access is
   begin
      return This.Full_View.Dialog;
   end Dialog;


   ----------------------
   --  Status Setting  --
   ----------------------

   --  Disable updates.  Note that this behavior can be
   --  controlled from the event model through the "Update"
   --  operation when "Display" or "Delete" is called
   --  on the event, as well as from freeze/unfreeze in a view.
   --  In both cases, the command goes through these two routines
   procedure Disable (This : access View_Role) is
   begin
      This.Full_View.Updatable := False;
   end Disable;


   --  Enable updates
   procedure Enable (This : access View_Role) is
   begin
      This.Full_View.Updatable := True;
   end Enable;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Update the positions table to reflect changes to the event data
   --  or to selected bodies, or to the view status
   procedure Update
     (This : access View_Role;
      Source : access Observable'Class;
      Evt : access Event'Class)
   is
      Event_Info : Update_Event_Ptr;
      Event : Event_Model_Ptr;
      Main : Main_Models.Main_Model_Ptr;
   begin
      if Evt.all in Update_Event then
         -- Downcast "Evt" to retrieve status info
         Event_Info := Update_Event (Evt.all)'Access;
         Event :=  Event_Model (Source.all)'Access;
         Main := Main_Models.Main_Model (Event_Models.Main (Event).all)'Access;

         if Main_Models.Displayable
           (Main, Main_Models.Table_Of_Positions) then
            --  This display kind is selected.  Check display status
            --  of the event
            if Display_Status (Event_Info.all) = Force_Off then
               --  Hide the display and ignore updates
               Disable (This);
               Hide_All (This.Full_View.Dialog);
               return;
            elsif Display_Status (Event_Info.all) = Freeze then
               --  Ignore updates
               Disable (This);
               return;
            elsif Display_Status (Event_Info.all) = Force_On then
               --  Signal to receive updates
               Enable (This);
            end if;

            if This.Full_View.Selected and then This.Full_View.Updatable then
               --  Update the positions table dialog and display it
               declare
                  Info : Report_Event_Ptr := new Report_Event;
               begin

                  Set_Title (This.Full_View.Dialog,
                             - ("AstroFrames:  Positions for " &
                                Description (Event)));

                  --  Forward event info to coordinate reporters
                  Info.Source := Event;
                  Info.Evt := Event_Info;
                  Notify_Observers (This.Full_View, Info);

                  Show_All (This.Full_View.Dialog);
               end;
            end if;
         end if;
      elsif Evt.all in Export_Event'Class then
            declare
               Model : Event_Model_Ptr := Event_Model (Source.all)'Access;
            begin
               Export_Positions
                 (This.Full_View,
                  Model,
                  Export_Event (Evt.all)'Access);
            end;
       end if;
   end Update;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Dump the positions into a designated file
   procedure Export_Positions (
                    This: access Positions_View;
                    Model : access Event_Model'Class;
                    Evt: access Export_Event'Class) is

      Header_Width : constant := 16;
      Coordinate_Width : constant := 13;

      Row_Header : String (1 .. Header_Width);
      Row_Field : String (1 ..Coordinate_Width);

      F : File_Type;
   begin
      Create (F, Name => File_Spec (Evt));

      Write_Event_Info (F, Model);

      --  Write column headers
      Put (F, "Point");
      Move ("Longitude", Row_Field, Justify => Right);
      Set_Col (F, Header_Width + 1);
      Put (F, Row_Field);
      Move ("Latitude", Row_Field, Justify => Right);
      Set_Col (F, Header_Width + Coordinate_Width + 4);
      Put (F, Row_Field);
      Move ("RA", Row_Field, Justify => Center);
      Set_Col (F,  Header_Width + 2 * Coordinate_Width + 4);
      Put (F, Row_Field);
      Move ("Declination", Row_Field, Justify => Right);
      Set_Col (F, Header_Width + 3 * Coordinate_Width + 4);
      Put (F, Row_Field);
      Move ("Terrestrial", Row_Field, Justify => Right);
      Set_Col (F,  Header_Width + 4 * Coordinate_Width + 4);
      Put (F, Row_Field);
      New_Line (F);
      New_Line (F);

      for Row in 0 .. Get_Rows (Table (This.Header)) - 1 loop
         --  Row header
         Move
           (Get_Text (Table (This.Header), Row, 0),
            Row_Header, Justify => Left);
         Put (F, Row_Header (1 .. Header_Width - 1));


         -- Ecliptic positions
         Move
           (Get_Text (Table (This.Ecliptic), Row, 0),
            Row_Field, Justify => Right);
         Set_Col (F, Header_Width + 1);
         Put (F, Row_Field & ' ' & Get_Text (Table (This.Ecliptic), Row, 1));

         Move
           (Get_Text (Table (This.Ecliptic), Row, 2),
            Row_Field, Justify => Right);
         Set_Col (F, Header_Width + Coordinate_Width + 4);
         Put (F, Row_Field);

         -- Equatorial positions
         Move
           (Get_Text (Table (This.Equator), Row, 0),
            Row_Field, Justify => Right);
         Set_Col (F, Header_Width + 2 * Coordinate_Width + 4);
         Put (F, Row_Field);

         Move
           (Get_Text (Table (This.Equator), Row, 1), Row_Field,
            Justify => Right);
         Set_Col (F, Header_Width + 3 * Coordinate_Width + 4);
         Put (F, Row_Field);

         -- Terrestrial positions
         Move
           (Get_Text (Table (This.Terra), Row, 0),
            Row_Field, Justify => Right);
         Set_Col (F, Header_Width + 4 * Coordinate_Width + 4);
         Put (F, Row_Field);

         New_Line (F);
      end loop;

      --  Write notes if there are any
      if Notes (Model) /= "" then
         New_Line (F);
         New_Line (F);
         Put_Line (F, "Notes:");
         New_Line (F);
         Put_Line (F, Notes (Model));
      end if;

      Close (F);
   end Export_Positions;

   --  Write event description
   procedure Write_Event_Info
     (F : File_Type; Event : access Event_Model'Class) is
      -- Format various strings
      Description : String :=
        Event_Models.Description (Event);

      Time : String :=
        Time_String (Date_Time (Event).Local_Time);

      Zone : String :=
        " Zone: " & Zone_String (Date_Time (Event).Zone);

      Date : String :=
        Date_String
        (Date_Time (Event).D,
         Date_Time (Event).M,
         Date_Time (Event).Y);

      Latitude : String :=
        "Lat: " & Latitude_String (Location (Event).Lat);
      Longitude : String :=
        " Long: " & Longitude_String (Location (Event).Long);
      Altitude : String :=
        " Alt: " & Altitude_String (Location (Event).Alt);
   begin
      Put_Line (F, Description);
      New_Line (F);

      Put_Line (F, Date  & Time & Zone);
      Put_Line (F, Latitude & Longitude & Altitude);

      New_Line (F);
      New_Line (F);
   end Write_Event_Info;

end Positions_Views;
