------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                      PRIMARIES_CONTROLLERS.EVENTS
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Primaries_Views.Calculations; use Primaries_Views.Calculations;

--  Set up the event for primary directions calculations
package body Primaries_Controllers.Events is

   ----------------------
   --  Implementation  --
   ----------------------

   Aspect_Angles : constant array (Aspects) of Long_Float :=
     (Conjunction => 0.0,
      Semisquare => 45.0,
      Sextile => 60.0,
      Square => 90.0,
      Trine => 120.0,
      Sesquare => 135.0,
      Opposition => 180.0,
      Sesquare_S => 225.0,
      Trine_S => 240.0,
      Square_S => 270.0,
      Sextile_S => 300.0,
      Semisquare_S => 315.0,
      others => 0.0);


   ------------------------------
   --  Primaries Export Event  --
   ------------------------------

   --  Initialize event
   function New_Primaries_Export_Event (File_Spec : String)
                                       return Primaries_Export_Event_Access is
      Result : Primaries_Export_Event_Access :=
        new Primaries_Export_Event;
   begin
      Result.File_Spec := To_Unbounded_String (File_Spec);
      return Result;
   end New_Primaries_Export_Event;

   --  Get file spec for text file to write directions list
   function File_Spec (E : access Primaries_Export_Event) return String is
   begin
      return To_String (E.File_Spec);
   end File_Spec;


   ------------------------------------
   --  Primary Update Event Queries  --
   ------------------------------------

   --  Start date:
   function Start_Date
     (E : access Primaries_Update_Event'Class) return Julian_Day is
   begin
      return E.Start_Date;
   end Start_Date;

   --  End date:
   function End_Date
     (E : access Primaries_Update_Event'Class) return Julian_Day is
   begin
      return E.End_Date;
   end End_Date;

   --  Key, in degrees per year
   function Key (E : access Primaries_Update_Event'Class) return Keys is
   begin
      return E.Key;
   end Key;


   --------------------------------
   --  Initialization - Zodiacal --
   --------------------------------

   --  When filtering dialog is implemented, extract common constructor
   --  operations to parent type
   function Create_Zodiacal (Start_Date, End_Date : Julian_Day;
                            Key : Keys)
                            return Zodiacal_Primaries_Event_Access is
      Result : Zodiacal_Primaries_Event_Access := new Zodiacal_Primaries_Event;
   begin
      Result.Start_Date := Start_Date;
      Result.End_Date := End_Date;
      Result.Key := Key;
      return Result;
   end Create_Zodiacal;


   --------------------------
   --  Queries - Zodiacal  --
   --------------------------

   --  String identifying directional system
   function System_String (E : access Zodiacal_Primaries_Event)
                          return String is
   begin
      return "zo";
   end System_String;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a promissor
   function To_Promissor
     (E : access Zodiacal_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d is

      Ecliptic : Coord_2d;

   begin
      Ecliptic := To_Ecliptic (Point, Obliquity);
      Ecliptic.Lat := 0.0;

      --  Parallels and contraparallels require special handling
      case Aspect is
         when Parallel | Contraparallel =>
            declare
               --  "Event" is used only for dispatching, and need not
               --  be initialized
               Event : aliased Mundo_Primaries_Event;
            begin
               --  Use mundo transformation to find reflection point
               --  for zodiacal projection of "Point"
               Ecliptic :=
                 To_Promissor
                 (Event'Access,
                  To_Equatorial (Ecliptic, Obliquity),
                  Aspect, Obliquity);
               return Ecliptic;
            end;

         when others =>
            Ecliptic.Long := Ecliptic.Long + Aspect_Angles (Aspect);
            return To_Equatorial (Ecliptic, Obliquity);
      end case;

   end To_Promissor;


   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a significator
   function To_Significator
     (E : access Zodiacal_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d renames To_Promissor;

   --  String showing latitude used for promissor.
   function Promissor_Lat_String (E : access Zodiacal_Primaries_Event)
                                 return String is
   begin
      return "(l = 0)";
   end Promissor_Lat_String;

   --  String showing latitude used for significator.
   function Significator_Lat_String (E : access Zodiacal_Primaries_Event)
                                    return String
     renames Promissor_Lat_String;


   --------------------------------
   --  Initialization - Mundo --
   --------------------------------

   --  When filtering dialog is implemented, extract common constructor
   --  operations to parent type
   function Create_Mundo (Start_Date, End_Date : Julian_Day;
                          Key : Keys)
                         return Mundo_Primaries_Event_Access is
      Result : Mundo_Primaries_Event_Access := new Mundo_Primaries_Event;
   begin
      Result.Start_Date := Start_Date;
      Result.End_Date := End_Date;
      Result.Key := Key;
      return Result;
   end Create_Mundo;


   --------------------------
   --  Queries - Mundo  --
   --------------------------

   --  String identifying directional system
   function System_String (E : access Mundo_Primaries_Event)
                          return String is
   begin
      return "mu";
   end System_String;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a promissor
   function To_Promissor
     (E : access Mundo_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d is

      Result : Coord_2d;

      --  Proportional point of aspect
      Proportion : Normal_UMD_Range;

      --  Upper meridian distance of radix body
      UMD_R : Normal_UMD_Range :=
        Point.Long - Primaries_Views.Calculations.RAMC;

      --  Ascensional difference on primary path of radix body
      AD_R : Normal_UMD_Range := AD (Point.Lat);

   begin
      Result.Lat := Point.Lat;

      --  Find proportional point for aspect
      if Above_Horizon (Point) then
         Proportion := UMD_R / (90.0 + AD_R) +
           Normal_UMD_Range (Aspect_Angles (Aspect) / 90.0);
      else
         if UMD_R < 0.0 then
            --  West
            Proportion := - 2.0 +
              (180.0 + UMD_R) / (90.0 - AD_R) +
              Normal_UMD_Range (Aspect_Angles (Aspect) / 90.0);
         else
            --  East
            Proportion := 2.0 -
              (180.0 - UMD_R) / (90.0 - AD_R) +
              Normal_UMD_Range (Aspect_Angles (Aspect) / 90.0);
         end if;
      end if;

      if Proportion >= 2.0 then
         Proportion := -2.0 + (Proportion - 2.0);
      elsif Proportion < -2.0 then
         Proportion := 2.0 + (Proportion + 2.0);
      end if;

      --  Parallels and contraparallels require special handling
      case Aspect is
         when Parallel =>
            Proportion := - Proportion;

         when ContraParallel =>
            if Proportion < 0.0 then
               Proportion := -2.0 - Proportion;
            else
               Proportion := 2.0 - Proportion;
            end if;

         when others => null;
      end case;


      pragma Assert (-2.0 <= Proportion and Proportion <= 2.0,
                    "proportion out of range");

      --  Convert to RA
      if Proportion in - 1.0 .. 1.0 then
         -- Above horizon
         Result.Long :=
           Primaries_Views.Calculations.RAMC +
           Long_Float (Proportion * (90.0 + AD_R));
      elsif Proportion in - 2.0 .. - 1.0 then
         --  Below horizon west
         Result.Long :=
           Primaries_Views.Calculations.RAMC +
           (-180.0 - Long_Float ((-2.0 - Proportion) * (90.0 - AD_R)));
      elsif Proportion in 1.0 .. 2.0 then
         --  Below horizon east
         Result.Long :=
           Primaries_Views.Calculations.RAMC +
           180.0 - Long_Float ((2.0 - Proportion) * (90.0 - AD_R));
      end if;

      --  Replace with cyclic operations
      if Result.Long <= 0.0 then
         Result.Long := Result.Long + 360.0;
      elsif Result.Long > 360.0 then
         Result.Long := Result.Long - 360.0;
      end if;
      return Result;
   end To_Promissor;


   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a significator
   function To_Significator
     (E : access Mundo_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d renames To_Promissor;

   --  String showing latitude used for promissor.
   function Promissor_Lat_String (E : access Mundo_Primaries_Event)
                                 return String is
   begin
      return "(l = r)";
   end Promissor_Lat_String;

   --  String showing latitude used for significator.
   function Significator_Lat_String (E : access Mundo_Primaries_Event)
                                    return String
     renames Promissor_Lat_String;

end Primaries_Controllers.Events;
