------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                      PRIMARIES_CONTROLLERS.EVENTS
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Times; use Times;
with Coordinate_Conversions; use Coordinate_Conversions;
with Publish_Subscribe; use Publish_Subscribe;

with Event_Models; use Event_Models;

with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;

package Primaries_Controllers.Events is

   ------------------------------
   --  Primaries_Export_Event  --
   ------------------------------

   --  Export displayed directions to a file
   type Primaries_Export_Event is new Event with private;
   type Primaries_Export_Event_Access is
     access all Primaries_Export_Event'Class;

   --  Initialize event
   function New_Primaries_Export_Event (File_Spec : String)
                                       return Primaries_Export_Event_Access;

   --  Get file spec
   function File_Spec (E : access Primaries_Export_Event) return String;


   ------------------------------
   --  Primaries_Update_Event  --
   ------------------------------

   --  Parameters for primary directions calculations
   type Primaries_Update_Event is abstract new Update_Event with private;
   type Primaries_Update_Event_Ptr is access all Primaries_Update_Event'Class;

   type Aspects is
     --  Dexter aspects
     (Conjunction, Semisquare, Sextile, Square, Trine, Sesquare, Opposition,
      --  Sinister aspects
      Sesquare_S, Trine_S, Square_S, Sextile_S, Semisquare_S,
      --  Reflections
      Parallel, ContraParallel);

   type Keys is (Ptolemy, Synodic, Cardan, Naibod);

   ---------------
   --  Queries  --
   ---------------

   --  Start date:
   function Start_Date (E : access Primaries_Update_Event'Class)
                       return Julian_Day;

   --  End date:
   function End_Date (E : access Primaries_Update_Event'Class)
                     return Julian_Day;

   --  Key, in degrees per year
   function Key (E : access Primaries_Update_Event'Class) return Keys;

   --  String identifying directional systems
   function System_String (E : access Primaries_Update_Event)
                          return String is abstract;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a promissor
   function To_Promissor
     (E : access Primaries_Update_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d is abstract;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a significator
   function To_Significator
     (E : access Primaries_Update_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_float) return Coord_2d is abstract;

   --  String showing latitude used for promissor.
   function Promissor_Lat_String (E : access Primaries_Update_Event)
                                 return String is abstract;

   --  String showing latitude used for significator.
   function Significator_Lat_String (E : access Primaries_Update_Event)
                                    return String is abstract;

   --------------------------------
   --  Zodiacal_Primaries_Event  --
   --------------------------------

   type Zodiacal_Primaries_Event is new Primaries_Update_Event with private;
   type Zodiacal_Primaries_Event_Access is access all Zodiacal_Primaries_Event;


   ----------------------
   --  Initialization  --
   ----------------------

   function Create_Zodiacal (Start_Date, End_Date : Julian_Day;
                             Key : Keys)
                            return Zodiacal_Primaries_Event_Access;

   ---------------
   --  Queries  --
   ---------------

   --  String identifying directional system
   function System_String (E : access Zodiacal_Primaries_Event)
                          return String;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a promissor
   function To_Promissor
     (E : access Zodiacal_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a significator
   function To_Significator
     (E : access Zodiacal_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d;

   --  String showing latitude used for promissor.
   function Promissor_Lat_String (E : access Zodiacal_Primaries_Event)
                                 return String;

   --  String showing latitude used for significator.
   function Significator_Lat_String (E : access Zodiacal_Primaries_Event)
                                    return String;

   --------------------------------
   --  Mundo_Primaries_Event  --
   --------------------------------

   type Mundo_Primaries_Event is new Primaries_Update_Event with private;
   type Mundo_Primaries_Event_Access is access all Mundo_Primaries_Event;


   ----------------------
   --  Initialization  --
   ----------------------

   function Create_Mundo (Start_Date, End_Date : Julian_Day;
                          Key : Keys)
                            return Mundo_Primaries_Event_Access;

   ---------------
   --  Queries  --
   ---------------

   --  String identifying directional system
   function System_String (E : access Mundo_Primaries_Event)
                          return String;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a promissor
   function To_Promissor
     (E : access Mundo_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d;

   --  Convert equatorial coordinates of a point to the appropriate
   --  coordinates for use as a significator
   function To_Significator
     (E : access Mundo_Primaries_Event;
      Point : Coord_2d;
      Aspect : Aspects;
      Obliquity : Long_Float) return Coord_2d;

   --  String showing latitude used for promissor.
   function Promissor_Lat_String (E : access Mundo_Primaries_Event)
                                 return String;

   --  String showing latitude used for significator.
   function Significator_Lat_String (E : access Mundo_Primaries_Event)
                                    return String;

private

   --  Export displayed directions to a file
   type Primaries_Export_Event is new Event with record
      File_Spec : Unbounded_String;
   end record;

   --  Parameters for primary directions calculations
   type Primaries_Update_Event is abstract new Update_Event with record
      Start_Date, End_Date : Julian_day;
      Key : Keys;
   end record;

   --  Zodiacal directions
   type Zodiacal_Primaries_Event is
     new Primaries_Update_Event with null record;

   --  Mundo directions
   type Mundo_Primaries_Event is
     new Primaries_Update_Event with null record;

   pragma Inline (Start_Date, End_Date, Key);
   pragma Inline (File_Spec);
end Primaries_Controllers.Events;
