------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                          PRIMARIES_CONTROLLERS                           --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.4 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Filter_Primaries_Views; use Filter_Primaries_Views;
with Primaries_Views; use Primaries_Views;
with Primaries_Controllers.Events; use Primaries_Controllers.Events;
with Event_Models; use Event_Models;
with Main_Models; use Main_Models;

with Ephemeris; use Ephemeris;
with Message_Boxes; use Message_Boxes;
with Field_Validation; use Field_Validation;
with Times.Strings; use Times, Times.Strings;
with Decimal_Sexagesimal; use Decimal_Sexagesimal;

with Gtk.Button; use Gtk.Button;
with Gtk.Clist; use Gtk.Clist;
with Gtk.Gentry; use Gtk.Gentry;
with Gtk.Radio_Button; use Gtk.Radio_Button;
with Glib; use Glib;
with Gtkada.File_Selection; use Gtkada.File_Selection;

with GNAT.Os_Lib; use GNAT.OS_Lib;

with Ada.Exceptions; use Ada.Exceptions;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;

--  Controller for Primaries List dialog
package body Primaries_Controllers is

   ----------------------
   --  Initialization  --
   ----------------------

   --  Set up controller
   procedure Initialize
     (C : access Primaries_Controller;
      M : access Model_Obj'Class;
      V : access View_Obj'Class) is
   begin
      --  Call super initialization operation
      Initialize (Controller_Obj (C.all)'Access, M, V);

      C.Dialog := Dialog (Primaries_View (V.all)'Access);
   end Initialize;


   -----------------------
   --  Basic Operations --
   -----------------------

   --  Generate and display the primaries direction list
   procedure Calculate (This : access Primaries_Controller) is
      Start_Month : Months;
      Start_Year : Years;

      End_Month : Months;
      End_Year : Years;

      Radix_Components : Date_Time_Components :=
        Date_Time (Event_Model (Model (This).all)'Access);

      Start_Date, End_Date, Radix_Date : Julian_Day;

      Evt : Primaries_Update_Event_Ptr;

      Key_String : String := Get_Text (This.Dialog.Key);

      Key : Keys :=
        Keys'Value (Key_String (1 .. Index (Key_String, " ") - 1));

   begin
      --  Get dates into Julian day format and validate
      Radix_Date :=
        Gregorian_To_Julian_Day
        (Radix_Components.D, Radix_Components.M, Radix_Components.Y,
         Radix_Components.Local_Time);

      --  Adjust for zone:
      Radix_Date := Radix_Date + Radix_Components.Zone / 24.0;

      Start_Month := Month_Value (Get_Text (This.Dialog.Start_Month));
      Start_Year := Year_Value (Get_Text (This.Dialog.Start_Year));
      Start_Date :=
        Gregorian_To_Julian_Day (1, Start_Month, Start_Year, 0.0);

      --  Must start the list no earlier than radix. Substitute radix
      --  date if start date is earlier
      if Start_Date < Radix_Date then
         Start_Date := Radix_Date;
      end if;


      End_Month :=  Month_Value (Get_Text (This.Dialog.End_Month));
      End_Year := Year_Value (Get_Text (This.Dialog.End_Year));

      --  Adjust to use last day of month
      if End_Month = Dec then
         End_Date :=
           Gregorian_To_Julian_Day
           (31, End_Month, End_Year,
            To_Decimal (Make_Sexagesimal (23, 59, 59.0)));
      else
         End_Date :=
           Gregorian_To_Julian_Day
           (1, Months'Succ (End_Month), End_Year, 0.0) - 1.0;
      end if;

      --  Makes no sense to continue.  Notify user.
      if End_Date <= Start_Date then
         Invalid_Input_Box ("Invalid date range");
      else
         --  Create notification and pass to radix event for its observers
         if Is_Active (This.Dialog.Zodiacal) then
            Evt :=
              Primaries_Update_Event_Ptr'(
                                          Create_Zodiacal
                                          (Start_Date,
                                           End_Date,
                                           Key).all'Access);
         elsif Is_Active (This.Dialog.Mundo) then
            Evt :=
              Primaries_Update_Event_Ptr'(
                                          Create_Mundo
                                          (Start_Date,
                                           End_Date,
                                           Key).all'Access);
         else
            raise Calculation_Failed;
         end if;
         Update_All (Event_Model (Model (This).all)'Access,
                     Evt, Suppress_Main_Update => True);

         --  If here, calculation succeeded
         if Get_Rows (This.Dialog.Direction_List) > 0 then
            --  Enable export
            Set_Sensitive (This.Dialog.Export_Button, True);
         end if;
      end if;

   exception
      --  Intercept bad input or calculation errors, and pop
      --  up a message box

      when E : Calculation_Failed =>
         Calculation_Error_Box (Exception_Message (E));

      when Invalid_Date =>
         Invalid_Input_Box ("Invalid date field");
   end Calculate;


   --  Update primaries list if already displayed
   procedure Conditional_Update (This : access Primaries_Controller) is
      Rows : Gint := Get_Rows (This.Dialog.Direction_List);
   begin
      if Rows > 0 then
         Calculate (This);
      end if;
   end Conditional_Update;

   --  Close the dialog
   procedure Close (This : access Primaries_Controller) is
   begin
      Close (Primaries_View (View (This).all)'Access);
   end Close;

   --  Export list of directions if available
   procedure Export (This : access Primaries_Controller) is
      Event : Event_Model_Ptr :=
        Event_Model (Model (This).all)'Access;

      Spec : String :=
        File_Selection_Dialog
        (Title => "Save Primaries for " &
           Description (Event) &
           " to File",
         Default_Dir =>
           Event_Directory
           (Main_Model (Event_Models.Main (Event).all)'access));
   begin
      if Spec = "" or else
        Spec (Spec'Last) = GNAT.Os_Lib.Directory_Separator then
         Invalid_Input_Box
           ("Primaries not exported: invalid file specification");
      else
         --  Export current positions to file "Spec"
         Update_All (Event_Model (Model (This).all)'Access,
                     New_Primaries_Export_Event (Spec),
                     Suppress_Main_Update => True);
      end if;

   end Export;

   --  Open filtering dialog
   procedure Open_Filter (This : access Primaries_Controller) is
   begin
      Open (Filter (Primaries_View (View (This).all)'Access));
   end Open_Filter;

end Primaries_Controllers;
