------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                      PRIMARIES_VIEWS.CALCULATIONS                        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Strings.Fixed; use Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants; use Ada.Strings.Maps.Constants;
with Elementary_Functions; use Elementary_Functions;
with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Locations; use Locations;

with Gnat.Heap_Sort_G;
with Lists;

with Ephemeris.Flags; use Ephemeris.Flags;

with Body_Models; use Body_Models;

--  Primary direction calculations. This unit implements Placidus zodiacal
--  and mundo directions.  It should also provide the framework for
--  Svarogich conform directions (NYI).
--  It's currently messy due to being developed quickly.
package body Primaries_Views.Calculations is

   type Direction_Access is access all Direction;

   --  Lists to hold directions as they're generated
   package Direction_Lists is new Lists (Direction, Direction_Access);
   use Direction_Lists;

   --  Input to primaries calculations.  May turn into a tagged type
   --  to discriminate different systems of primaries.  At the moment
   --  this can be considered a partial speculum line.
   type Coordinates is record
      --  Needed by all systems
      Name : Unbounded_String;
      Equator : Coord_2d;
      --  Diurnal and nocturnal semiarcs. Precalculate (maybe - currently
      --  not used)
      DSA, NSA : Normal_UMD_Range;
      --  Radix upper meridian distance. Currently not used
      UMD : Normal_UMD_Range;

      --  Only needed for zodiacal directions
      Ecliptic : Long_Float;
   end record;

   --  Virtual speculum
   type Positions_Array is array (Natural range <>) of Coordinates;

   --  Obliquity of the ecliptic.  This, RAMC, Geo_Lat, Dynamic_Angle and
   --  Tan_DA are calculated when Generate is called.  The package is
   --  therefore not reentrant.
   Obliquity : Long_Float;

   --  Perpendicular to gravitational horizon (standin for geo latitude
   --  in Svarogich system.  Will use when conform directions are implemented
   --    Dynamic_Angle : Long_Float;

   Geo_Lat : Latitudes;

   --  Tangent of geographic latitude
   Tan_Lat : Long_Float;

   --  Aspects used. Filter dialog will specify for a given call to
   --  Generate
   type Aspect_Description is record
      Name : Unbounded_String;
      Arc : Long_Float;
   end record;

   type Aspects_Array is array (Aspects range <>) of Aspect_Description;

   Aspects_Table : constant Aspects_Array (Aspects'Range) :=
     ((To_Unbounded_String ("  "), 0.0),
      (To_Unbounded_String ("45"), 45.0),
      (To_Unbounded_String ("sx"), 60.0),
      (To_Unbounded_String ("sq"), 90.0),
      (To_Unbounded_String ("tr"), 120.0),
      (To_Unbounded_String ("135"), 135.0),
      (To_Unbounded_String ("opp"), 180.0),
      (To_Unbounded_String ("135"), 225.0),
      (To_Unbounded_String ("tr"), 240.0),
      (To_Unbounded_String ("sq"), 270.0),
      (To_Unbounded_String ("sx"), 300.0),
      (To_Unbounded_String ("45"), 315.0),
      (To_Unbounded_String ("||"), -1.0),   --  parallel
      (To_Unbounded_String ("=="), -1.0));  --  contraparallel

   Cusp_Labels : constant array (1 ..13) of Unbounded_String :=
     (To_Unbounded_String ("ASC"),
      To_Unbounded_String ("II"),
      To_Unbounded_String ("III"),
      To_Unbounded_String ("IC"),
      To_Unbounded_String ("V"),
      To_Unbounded_String ("VI"),
      To_Unbounded_String ("DSC"),
      To_Unbounded_String ("VIII"),
      To_Unbounded_String ("IX"),
      To_Unbounded_String ("MC"),
      To_Unbounded_String ("XI"),
      To_Unbounded_String ("XII"),
      To_Unbounded_String ("PF"));

   Key_Values : constant array (Keys) of Long_Float :=
     (Ptolemy => 1.0,
      Synodic => To_Decimal (Make_Sexagesimal (1, 0, 57.2)),
      Cardan  => To_Decimal (Make_Sexagesimal (0, 59, 12.0)),
      Naibod  => To_Decimal (Make_Sexagesimal (0, 59, 8.33)));


   ----------------------
   --  Implementation  --
   ----------------------

   --  Generate a set of directions, and add to Direction_List.  This one
   --  is potentially sensitive to the directional system used.  Obviously,
   --  all these parameters are exposing the opportunity for a tagged
   --  type for better organization.  Defer modifying until the needs of
   --  the filtering mechanism are better understood.
   procedure Add_Directions
     (Direction_List : in out List;
      Promissors : Positions_Array;
      Promissors_Aspects : Aspects_Array;
      Show_Promissor_Latitude: Boolean;
      Significators : Positions_Array;
      Significators_Aspects : Aspects_Array;
      Show_Significator_Latitude : Boolean;
      Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class);


   --  Generate array of directions for the specified range of dates in "Evt"
   function Get_Directions
     (Positions : access Positions_Array;
      Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) return Directions_Array;

   --  Fill in table of coordinates for selected bodies.  This largely
   --  overlaps Reporters.Equator.  Should combine the calculation routines
   --  of both together at some point.
   function Get_Positions
     (Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) return Positions_Array;

   --  Distance of a point from east point on the equator expressed in
   --  the range -180.0 < HD <= 180.0 of RA. Depends on global RAMC. This
   --  is also coded in Reporters.Terra.Calculations, and should be
   --  consolidated.
   function Horizon_Distance (RA : Long_Float) return Long_Float;

   --  Make string description of a point
   function Make_Id (Body_Code : Natural) return Unbounded_String;

   --  Determine a primary arc. Independent of selection of zodiacal
   --  vs mundo directions.  Not used for conform directions, which
   --  are not based on arcs.
   function Primary_Arc (Promissor, Significator : Coord_2D)
                        return Normal_UMD_Range;


   ---------------
   --  Queries  --
   ---------------

   --  Directions for "Evt" relative to "Radix"
   function Generate
     (Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) return Directions_Array is

      --  Speculum equivalent
      Positions : aliased Positions_Array := Get_Positions (Radix, Evt);

      --  Filtered directions
      Directions : Directions_Array :=
        Get_Directions (Positions'Access, Radix, Evt);


      --  Sort package  --

      --  "Move" actual parameter
      procedure Move (From : Natural; To : Natural) is
         Temp : Direction := Directions (To);
      begin
         Directions (To) := Directions (From);
         Directions (From) := Temp;
      end Move;

      --  "Lt" actual parameter
      function "<" (Left, Right : Natural) return Boolean is
      begin
         return Directions (Left).Date < Directions (Right).Date;
      end "<";

      package Heap_Sort is new GNAT.Heap_Sort_G (Move, "<");

   begin
      Heap_Sort.Sort (Directions'Last);
      return Directions;
   end Generate;


   ----------------------
   --  Implementation  --
   ----------------------

   --  Cyclic subtraction. Returns a value in the range -180 .. 180
   --  expressing the position of Left relative to Right.
   function "-" (Left, Right : Long_Float) return Normal_UMD_Range is
      Result : Long_Float;
   begin
      Result := Left - Right;
      if Result > 180.0 then
         Result := Result - 360.0;
      elsif Result <= -180.0 then
         Result :=  Result + 360.0;
      end if;
      return Normal_UMD_Range (Result);
   end "-";

   --  Determine whether a pair of equatorial coordinates is above the horizon.
   --  Depends on globals "RAMC" and "Dynamic_Angle".  Duplication of routine
   --  in "Reporters.Terra.Calculations" - should be consolidated.
   function Above_Horizon (Eq : Coord_2D) return Boolean is
      -- Distance in RA from eastern horizon intersect on the equator
      HD : Long_Float := Horizon_Distance (Eq.Long);
      Result : Long_Float :=
        Cos (Geo_Lat, 360.0) * Sin (HD, 360.0) * Cos (Eq.Lat, 360.0) -
        Sin (Geo_Lat, 360.0) * Sin (Eq.Lat, 360.0);
   begin
      return Result < 0.0;
   end Above_Horizon;

   --  Ascensional difference for an arbitrary declination.
   --  Depends on global "Tan_Lat".
   function AD (Declination : Long_Float) return Normal_UMD_Range is
      Result : Long_Float :=
        Arcsin (Tan (Declination, 360.0) * Tan_Lat, 360.0);
   begin
      return Normal_UMD_Range (Result);
   end AD;

   --  Generate a set of directions, and add to "Direction_List"
   procedure Add_Directions
     (Direction_List : in out List;
      Promissors : Positions_Array;
      Promissors_Aspects : Aspects_Array;
      Show_Promissor_Latitude: Boolean;
      Significators : Positions_Array;
      Significators_Aspects : Aspects_Array;
      Show_Significator_Latitude : Boolean;
      Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) is

      Days_In_Year : constant Long_Float := 365.2422;

      --  Arc to time conversion, expressed in degrees / year
      Key : constant Long_Float :=
        Key_Values (Primaries_Controllers.Events.Key (Evt));

      --  Range of applicable arcs
      subtype Filter_Arc is Long_Float range
         (Start_Date (Evt) - JD_UT (Evt.all)) / Days_In_Year * Key
         ..
         (End_Date (Evt) - JD_UT (Evt.all)) / Days_In_Year * Key;

      --  Short and long arcs of a single direction.  Not applicable for
      --  conform directions, which don't use arcs.
      Arc, Long_Arc : Long_Float;

      P_Lat_String, S_Lat_String : String := 7 * ' ';

      procedure Gen_Directions
        (Arc : Long_Float;
         Promissor : Natural;
         P_Aspect : Aspects;
         Significator : Natural;
         S_Aspect : Aspects) is

         Working_Arc : Long_Float := Arc;
      begin
         --  Iterate to determine all dates within the specified range
         --  at which the arc occurs
         while abs (Working_Arc) <= Filter_Arc'Last loop
            --  Filter the arc, format and add to list if
            --  in range
            if abs (Working_Arc) in Filter_Arc then
               --  Arc is in desired range of years.  Set its
               --  fields and add to directions list
               declare
                  D : Direction_Access := new Direction;
                  Class_String : String := " D";
               begin
                  --  By convention, negative arcs are direct;
                  --  positive arcs are converse
                  if Working_Arc > 0.0 then
                     Class_String (Class_String'Last) := 'C';
                  end if;

                  D.Date := JD_UT (Evt.all) +
                    abs (Working_Arc) / Key * Days_In_Year;

                  D.Description :=
                    To_Unbounded_String (System_String (Evt)) &
                    ' ' &
                    Promissors_Aspects (P_Aspect).Name & " " &
                    Promissors (Promissor).Name  & ' ' &
                    P_Lat_String &
                    Class_String & " -> " &
                    Significators_Aspects (S_Aspect).Name &
                    " " &
                    Significators (Significator).Name & ' ' &
                    S_Lat_String;

                  Direction_Lists.Extend (Direction_List, D);
               end;
            end if;
            --  Generate next occurrence of the arc
            Working_Arc :=
              Working_Arc + Long_Float'Copy_Sign (360.0, Working_Arc);
         end loop;
      end Gen_Directions;


   begin
      --  These two parameters can be eliminated.  The condition is
      --  implicit in the information passed in "Evt".  Initialize
      --  the result of "*_Lat_String" to eliminate these conditionals.
      if Show_Promissor_Latitude then
         P_Lat_String := Promissor_Lat_String (Evt);
      end if;

      if Show_Significator_Latitude then
         S_Lat_String := Significator_Lat_String (Evt);
      end if;

      --  Generate directions based on selected promissors, significators
      --  and aspects
      for Promissor in Promissors'Range loop
         for P_Aspect in Promissors_Aspects'Range loop

            for Significator in Significators'Range loop
               for S_Aspect in Significators_Aspects'Range loop

                  if Promissor /= Significator or else
                    (P_Aspect /= Conjunction and then
                     S_Aspect /= Conjunction) then

                     Arc := Long_Float (Primary_Arc
                       (To_Promissor
                        (Evt, Promissors (Promissor).Equator,
                         P_Aspect, Obliquity),
                        To_Significator
                        (Evt, Significators (Significator).Equator,
                         S_Aspect, Obliquity)));

                     --  Determine the arc for opposite direction
                     Long_Arc := Arc + Long_Float'Copy_Sign (360.0, - Arc);

                     Gen_Directions
                       (Arc, Promissor, P_Aspect, Significator, S_Aspect);
                     Gen_Directions
                       (Long_Arc, Promissor, P_Aspect, Significator, S_Aspect);
                  end if;
               end loop;
            end loop;
         end loop;
      end loop;
   end Add_Directions;


   --  Fill in array of directions
   function Get_Directions
     (Positions : access Positions_Array;
      Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) return Directions_Array is

      --  List of directions filtered for the desired range of dates
      Directions_List : List;

      --  Index of last selected body in Positions array.
      --  Following that are the 12 house cusps and part of fortune
      Last_Body_Index : constant Natural :=
        Count (Event_Models.Body_Model (Radix).all);

      --  Index range in Positions array for house cusps
      subtype Cusps_Range is Positive
        range Last_Body_Index + 1 .. Last_Body_Index + 12;

      --  Some specific indexes in Positions
      PF : constant Positive := Cusps_Range'Last + 1;
      ASC : constant Cusps_Range := Cusps_Range'First;
      MC : constant Cusps_Range := ASC + 9;

   begin
      --  Fill out necessary bits of speculum
      for Point in Positions'First .. Last_Body_Index loop
         --  For now, just fill in longitude.  This is only needed for
         --  zodiacal directions.  Not sure there's enough here to
         --  justify turning the loop into a dispatching operation
         Positions (Point).Ecliptic :=
           To_Ecliptic (Positions (Point).Equator, Obliquity).Long;

         --  Figure out what to precalculate later, after adding variants
      end loop;

      Start (Directions_List);

      --  *** The sequence of directions generation here is based on
      --  what is typically used for zodiacal directions.  Some of
      --  these don't make sense for mundo directions.  When the
      --  "filter directions" dialog is implemented, what's done here
      --  will be controlled from info passed in Evt about selections
      --  Otherwise, this code does not depend on the kind of directions
      --  being generated.

      --  Direct bodies to bodies and to part of fortune
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (Positions'First .. Last_Body_Index) &
           Positions (PF .. PF),
         Promissors_Aspects =>
           Aspects_Table (Aspects'First .. Semisquare_S),
         Show_Promissor_Latitude => True,
         Significators => Positions (Positions'First .. Last_Body_Index),
         Significators_Aspects =>
           Aspects_Table (Conjunction .. Conjunction),
         Show_Significator_Latitude => True,
         Radix => Radix,
         Evt => Evt);

      --  Generate parallels and contraparallels of bodies
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (Positions'First .. Last_Body_Index) &
           Positions (PF .. PF),
         Promissors_Aspects =>
           Aspects_Table (Conjunction .. Conjunction),
         Show_Promissor_Latitude => True,
         Significators => Positions (Positions'First .. Last_Body_Index),
         Significators_Aspects =>
           Aspects_Table (Parallel .. Contraparallel),
         Show_Significator_Latitude => True,
         Radix => Radix,
         Evt => Evt);

      --  Direct bodies to ASC and MC
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (Positions'First .. Last_Body_Index) &
           Positions (PF .. PF),
         Promissors_Aspects =>
           Aspects_Table (Aspects_Table'First .. Semisquare_S),
         Show_Promissor_Latitude => True,
         Significators => Positions (ASC) & Positions (MC),
         Significators_Aspects =>
           Aspects_Table (Conjunction .. Conjunction),
         Show_Significator_Latitude => True,
         Radix => Radix,
         Evt => Evt);

      --  Direct bodies to intermediate cusps and part of fortune
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (Positions'First .. Last_Body_Index),
         Promissors_Aspects => Aspects_Table (Conjunction .. Conjunction),
         Show_Promissor_Latitude => True,
         Significators =>
           Positions (ASC + 1 .. MC - 1) &
           Positions (MC + 1 .. PF),
         Significators_Aspects => Aspects_Table (Conjunction .. Conjunction),
         Show_Significator_Latitude => False,
         Radix => Radix,
         Evt => Evt);

      --  Direct MC, ASC to bodies
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (ASC .. ASC) & Positions (MC .. MC),
         Promissors_Aspects =>
           Aspects_Table (Aspects_Table'First .. Semisquare_S),
         Show_Promissor_Latitude => False,
         Significators => Positions (Positions'First .. Last_Body_Index),
         Significators_Aspects => Aspects_Table (Conjunction .. Conjunction),
         Show_Significator_Latitude => True,
         Radix => Radix,
         Evt => Evt);

      --  Direct MC, ASC to Part of Fortune
      Add_Directions
        (Direction_List => Directions_List,
         Promissors => Positions (ASC .. ASC) & Positions (MC .. MC),
         Promissors_Aspects =>
           Aspects_Table (Aspects_Table'First .. Semisquare_S),
         Show_Promissor_Latitude => False,
         Significators => Positions (PF .. PF),
         Significators_Aspects => Aspects_Table (Conjunction .. Conjunction),
         Show_Significator_Latitude => False,
         Radix => Radix,
         Evt => Evt);

      --  Move filtered list of directions into array
      declare
         --  Element at index 0 is a temp location used by
         --  GNAT.Heap_Sort_G.  Crummy abstraction.
         Result : Directions_Array (0 .. Count (Directions_List));
      begin
         Start (Directions_List);
         for I in 1 .. Result'Last loop
            pragma Assert (not Off (Directions_List));
            Result (I) := Item (Directions_List).all;
            Forth (Directions_List);
         end loop;
         --  Deallocate
         Wipe_Out (Directions_List);
         return Result;
      end;
   end Get_Directions;

   --  Fill in table of coordinates for selected bodies.
   --  Likely candidate for per directional system dispatching operation
   function Get_Positions
     (Radix : access Event_Model'Class;
      Evt : access Primaries_Update_Event'Class) return Positions_Array is

      --  Index of last body in Result.  Following last body
      --  are the 12 cusps and PoF.
      Last_Body_Index : constant Natural :=
        Count (Event_Models.Body_Model (Radix).all);

      --  Index range of cusps in Result
      subtype Cusps_Range is Positive
        range Last_Body_Index + 1 .. Last_Body_Index + 12;

      --  Indices of Part of Fortune & Ascendant in Result
      PF : constant Positive := Cusps_Range'Last + 1;
      ASC : constant Cusps_Range := Cusps_Range'First;

      --  Return selected bodies, plus house cusps & Part of Fortune
      Result : Positions_Array (1 .. PF);

      --  Iterator over currently selected bodies and points
      Bodies : Body_Iterator :=
        Initialize (Event_Models.Body_Model (Radix).all);

      --  Temps
      Current_Body : Integer;

      --  Ecliptic longitudes for calculation of Part of Fortune
      Sun, Moon : Long_Float;
   begin
      --  Set some globals for this round of calculations.  Assuming
      --  non-concurrent calculation of primaries, a reasonably safe
      --  bet.
      --       Dynamic_Angle :=
 --      Event_Models.Dynamic_Angle (Update_Event (Evt.all));

      Geo_Lat := Latitude (Evt.all);
      Tan_Lat := Tan (Geo_Lat, 360.0);
      Calculate_UT (JD_UT (Evt.all), Obliquity_And_Nutation);
      Obliquity := Longitude;

      --  Get positions of selected bodies
      Use_Equatorial;
      Ephemeris.Set_Topocentric_Parameters
        (Longitude (Evt.all),
         Geo_Lat,
         Altitude (Evt.all));

      Start (Bodies);
      for I in Result'First .. Last_Body_Index loop
         Current_Body := Item (Bodies);
         Calculate_UT (JD_UT (Evt.all), Current_Body);
         Result (I).Name := Make_Id (Current_Body);
         Result (I).Equator.Long := Longitude;
         Result (I).Equator.Lat := Latitude;
         Forth (Bodies);
      end loop;

      --  Calculate cusps
      Calculate_Cusps (Radix, Geo_Lat, Ephemeris.Placidus);
      RAMC := Event_Models.RAMC (Radix);

      for I in Cusps_Range loop
         Result (I).Name := Cusp_Labels (I - Last_Body_Index);
         Result (I).Ecliptic := Cusp (Radix, I - Last_Body_Index);
         Result (I).Equator :=
           To_Equatorial ((Result (I).Ecliptic, 0.0), Obliquity);
      end loop;

      MC := Result (10 + Last_Body_Index).Ecliptic;

      --  Calculate Part of Fortune, and place at end of Result
      --  array
      Use_Ecliptical;
      Calculate_UT (JD_UT (Evt.all), Body_Models.Sun);
      Sun := Longitude;
      Calculate_UT (JD_UT (Evt.all), Body_Models.Moon);
      Moon := Longitude;

      if Above_Horizon (To_Equatorial ((Sun, 0.0), Obliquity)) then
         --  Use diurnal formula
         Result (PF).Ecliptic := Result (ASC).Ecliptic + Moon - Sun;
      else
         --  Use nocturnal formula
         Result (PF).Ecliptic := Result (ASC).Ecliptic - Moon + Sun;
      end if;

      --  Normalize longitude to range 0 .. 360  Should add a cyclic
      --  addition operator on type "Longitudes"
      if Result (PF).Ecliptic < 0.0 then
         Result (PF).Ecliptic := Result (PF).Ecliptic + 360.0;
      elsif Result (PF).Ecliptic >= 360.0 then
         Result (PF).Ecliptic := Result (PF).Ecliptic - 360.0;
      end if;

      Result (PF).Equator :=
        To_Equatorial ((Result (PF).Ecliptic, 0.0), Obliquity);
      Result (PF).Name := Cusp_Labels (PF - Last_Body_Index);

      return Result;
   end Get_Positions;

   --  Distance of a point from the east point on the equator expressed in
   --  the range -180.0 < HD <= 180.0 of RA. Duplicate of code in
   --  "Reporters.Terra.Calculations" - should consolidate.
   function Horizon_Distance (RA : Long_Float) return Long_Float is
      EP : Long_Float := RAMC + 90.0;
      HD : Long_Float;
   begin
      if EP >= 360.0 then
         EP := EP - 360.0;
      end if;

      HD := RA - EP;

      --  Normalize to range -180 .. 180
      --  Replace with cyclic subtraction already defined on Normal_UMD,
      --  which should be broken out into the utilities area as a generally
      --  useful facility.
      if HD > 180.0 then
         HD := HD - 360.0;
      elsif HD <= -180.0 then
         HD :=  HD + 360.0;
      end if;

      --  Ensure in range
      pragma Assert (-180.0 < HD and HD <= 180.0, "HD out of range");
      return HD;
   end Horizon_Distance;

   --  Make string for description of a point
   function Make_Id (Body_Code : Natural) return Unbounded_String is
      Result : Unbounded_String;
   begin
      if Is_Basic (Body_Code) then
         --  Use a 2 character string to identify the point
         declare
            Image : String := Basic_Bodies'Image (To_Body (Body_Code));

            --  Find underscore in body name, if any
            Pos: Natural := Index (Image, "_");
         begin
            if Pos = 0 then
               --  No underscore. Use first two characters of image for id
               Result :=
                 To_Unbounded_String
                 (Image (1 .. 1) & Translate (Image (2 .. 2), Lower_Case_Map));
            else
               --  Two word image. Use first letter of each word as id
               Result :=
                 To_Unbounded_String
                 (Image (1 .. 1) & Image (Pos + 1 .. Pos + 1));
            end if;
         end;
      else
         --  Extended body. Use catalog number as id
         Result :=
           To_Unbounded_String
           (Overwrite (Natural'Image (Body_Code - 10_000), 1, "#"));
      end if;
      return Result;
   end Make_Id;

   --  Determine primary arc.  This is independent of whether latitude
   --  is used for zodiacal and mundo directions, as the proper
   --  coordinates of either the body or its zodiacal projection are
   --  passed in the arguments. Conform directions will use a completely
   --  different mechanism for determining hit dates, based on the time
   --  when a promissor has the same terrestrial longitude as a
   --  significator, followed by the symbolic time determination
   function Primary_Arc (Promissor,Significator : Coord_2D)
                        return Normal_UMD_Range is

      --  Ascensional difference of promissor
      AD_P : Normal_UMD_Range := AD (Promissor.Lat);

      --  Upper meridian distance of promissor
      UMD_P : Normal_UMD_Range := Promissor.Long - RAMC;

      --  Ascensional difference of significator
      AD_S : Normal_UMD_Range := AD (Significator.Lat);

      --  Upper meridian distance of significator
      UMD_S : Normal_UMD_Range := Significator.Long - RAMC;

      --  Proportional point on primary path of the promissor
      PP : Normal_UMD_Range;
   begin
      --  Consider whether this can be simplified
      if Above_Horizon (Significator) then
         PP := (90.0 + AD_P) * UMD_S / (90.0 + AD_S);
      else
         if UMD_S < 0.0 then
            -- West
            PP := - 180.0 + (90.0 - AD_P) * (180.0 + UMD_S) / (90.0 - AD_S);
         else
            -- East
            PP := 180.0 - (90.0 - AD_P) * (180.0 - UMD_S) / (90.0 - AD_S);
         end if;
      end if;

      return Long_Float (PP) - Long_Float (UMD_P);
   end Primary_Arc;

end Primaries_Views.Calculations;
