------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                            REPORTERS.ECLIPTIC                            --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ephemeris.Flags; use Ephemeris.Flags;
with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Body_Models; use Body_Models;
with Event_Models;

with Positions_Dialog_Pkg; use Positions_Dialog_Pkg;
with Positions_Views; use Positions_Views;

--  View for GTK+ columned list display of ecliptic coordinates
package body Reporters.Ecliptic is

   ----------------------
   --  Implementation  --
   ----------------------

   function Motion_String
      (Evt : access Update_Event'Class;
       The_Body : Natural) return Motion_String_Result;


   ----------------------
   --  Initialization  --
   ----------------------

   --  Establish reference to model (Positions_View)
   procedure Initialize
     (This : access Reporter; M : access Model_Obj'Class) is
      Positions_Dialog : Positions_Dialog_Access :=
        Dialog (Positions_View (M.all)'Access);
   begin
      --  Super version of operation
      MVC.Initialize (View_Obj (This.all)'Access, M);
      Add_Observer (M, Observer (This.all)'Access);

      --  Associate with ecliptic display table
      This.Table := Positions_Dialog.Ecliptic_Table;
      Format_Table (This);
   end Initialize;


   --  Ecliptic coordinate strings for one cusp
   function Get_Coordinates
     (This : access Reporter;
      Source : access Event_Model'Class;
      Evt : access Update_Event'Class;
      Cusp : Cusps) return Coordinates is

      Direction_String : constant Motion_String_Result := (others => ' ');
      use Event_Models;
   begin
      --  Topocentric is closest to Svarogich.
      Calculate_Cusps (Source, Latitude (Evt.all), Ephemeris.Topocentric);
      return
           (Count => 3,
            Longitude =>
              Pad
              (To_Degree_String
               (Event_Models.Cusp
                (Source, ((Cusps'Pos (Cusp) + 9) mod 12) + 1)),
               Right),
            Direction => Direction_String,
            Latitude  => Pad (To_Degree_String (0.0), Right));
   end Get_Coordinates;

   --  Ecliptic coordinate strings for one body
   function Get_Coordinates
     (This : access Reporter;
      Source : access Event_Model'Class;
      Evt : access Update_Event'Class;
      The_Body : Natural) return Coordinates is
   begin
      --  Perform ecliptic calculations
      Use_Ecliptical;
      Generate_Velocities;
      Ephemeris.Set_Topocentric_Parameters
        (Longitude (Evt.all),
         Latitude (Evt.all),
         Altitude (Evt.all));
      Calculate_UT (JD_UT (Evt.all), The_Body);

      --  Format and return corresponding strings
      declare
         Long : String :=
           Pad (To_Degree_String (Longitude), Right);
         Motion : Motion_String_Result :=
           Motion_String (Evt, The_Body);
         Lat : String := Pad (To_Degree_String (Latitude), Right);
      begin
         return
           (Count => 3,
            Longitude => Long,
            Direction => Motion,
            Latitude  => Lat);
      end;
   end Get_Coordinates;


   ----------------------
   --  Implementation  --
   ----------------------

   --  String describing Retrograde or Stationary motion status
   function Motion_String
      (Evt : access Update_Event'Class;
       The_Body : Natural) return Motion_String_Result is

      Current_Velocity : Long_Float;
      Result : Motion_String_Result := "  ";
      Stationary : Boolean := False;
      Retrograde : Boolean := False;
   begin
      --  Recalculate velocity, since the program may eventually
      --  be multithreaded.  This is not enough for thread safety:
      --  will need to make calculation sequence a protected operation
      --  or task entry
      Use_Ecliptical;
      Generate_Velocities;
      Calculate_UT (JD_UT (Evt.all), The_Body);

      --  Determine whether body is retrograde
      if Velocity < 0.0 then
         if not Is_Basic (The_Body) or else
           To_Body (The_Body) /= Mean_Node then
            Retrograde := True;
         end if;
      end if;

      --  Determine whether body is stationary.  Velocity limits are
      --  "rules of thumb" from Alois Treindl, except for the asteroids.
      --  Of the basic bodies, Pholus is not considered.  Catalog
      --  bodies are not considered.  Since the mean lunar node is always
      --  retrograde, it doesn't get an indication.  We don't try to
      --  determine whether extended bodies are stationary
      if Is_Basic (The_Body) then
         case To_Body (The_Body) is
            when Mercury =>
               if abs
                 Velocity < To_Decimal (Make_Sexagesimal (0, 5, 0.0)) then
                  Stationary := True;
               end if;

            when Venus =>
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 3, 0.0)) then
                  Stationary := True;
               end if;

            when Mars =>
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 1, 30.0)) then
                  Stationary := True;
               end if;

            when Jupiter | Saturn =>
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 1, 0.0)) then
                  Stationary := True;
               end if;

            when Uranus | Chiron =>
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 0, 20.0)) then
                  Stationary := True;
               end if;

            when Neptune | Pluto =>
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 0, 10.0)) then
                  Stationary := True;
               end if;

            when Ceres | Pallas | Vesta | Juno =>
               --  Guess-timate value
               if abs
                 Velocity <= To_Decimal (Make_Sexagesimal (0, 1, 15.0)) then
                  Stationary := True;
               end if;

            when others =>
               null;
         end case;
      end if;

      --  Determine the proper string
      if Stationary then
         -- Get velocity for next day
         Current_Velocity := Velocity;
         Calculate_UT (JD_UT (Evt.all) + 1.0, The_Body);

         if Retrograde then
            if Velocity < Current_Velocity then
               --  Just past the retrograde station
               Result := "SR";
            else
               --  Approaching the direct station
               Result := "RS";
            end if;
         elsif Velocity > Current_Velocity then
            --  Just past the direct station
            Result := "SD";
         else
            --  Approaching the retrograde station
            Result := "DS";
         end if;

      elsif Retrograde then
         Result := "R ";
      end if;

      return Result;
   end Motion_String;

end Reporters.Ecliptic;
