------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                 --
--                                                                          --
--                        REPORTERS.TERRA.CALCULATIONS                      --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Exceptions; use Ada.Exceptions;
with Elementary_Functions; use Elementary_Functions;
with Ephemeris.Flags; use Ephemeris.Flags;
with Body_Models; use Body_Models;

--  Calculations for view displaying terrestrial longitudes in GTK+ CLIST
package body Reporters.Terra.Calculations is

   ------------------------------------
   --  Implementation  Declarations  --
   ------------------------------------

   --  Common fields extracted or derived from "Event" to make code more
   --  readable
   RAMC, Dynamic_Angle,
     RA, Decl : Long_Float;

   --  Forward declarations of implementation routines
   function Above_Horizon return Boolean;
   function Estimated_Tau (HD : Long_Float) Return Long_Float;
   function Horizon_Distance return Long_Float;
   function House_Position return Long_Float;
   function Normalize (Long : Long_Float) return Long_Float;

   --  Declarations related to binary search solution of house position
   --  formula
   function Delta_Fn (Tau : Long_Float) return Long_Float;
   type Fn is access function (Tau : Long_Float) return Long_Float;
   function Binary_Search (Func : Fn) return Long_Float;

   pragma Inline (Horizon_Distance);


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Svarogich terrestrial longitude for one body at time and location
   --  of "Event".  These positions are very close to Placidus and Topocentric,
   --  following S's refinement and reformulation of the latter
   function House_Position
     (Event : Update_Event_Ptr; A_Body : Natural)
     return Long_Float is
   begin
      -- Get Equatorial coordinates
      Dynamic_Angle := Event_Models.Dynamic_Angle (Event.all);
      Use_Equatorial;
      Ephemeris.Set_Topocentric_Parameters
        (Longitude (Event.all),
         Dynamic_Angle,
         Altitude (Event.all));
      Calculate_Ut (JD_UT (Event.all), A_Body);

      -- Set package body locals
      RA := Longitude;
      Decl := Latitude;
      RAMC := 15.0 * GST (JD_UT (Event.all)) + Longitude (Event.all);

      --  Calculate terrestrial longitude
      return House_Position;
   exception
      when E : others =>
         Raise_Exception
           (Exception_Identity (E),
            "House longitude for triplicity zone NYI");
   end House_Position;


   ----------------------
   --  Implementation  --
   ----------------------

   -- On or above the horizon?
   function Above_Horizon return Boolean is
      -- Distance in RA from eastern horizon intersect on the equator:
      HD : Long_Float := Horizon_Distance;
   begin
      return
        Cos (Dynamic_Angle, 360.0) * Sin (HD, 360.0) * Cos (Decl, 360.0) -
        Sin (Dynamic_Angle, 360.0) * Sin (Decl, 360.0) < 0.0;
   end Above_Horizon;


   --  First estimate for terrestrial longitude to plug into iterative
   --  solution. Main purpose is to start iteration in the correct quadrant
   function Estimated_Tau (HD : Long_Float) Return Long_Float is
      --  Start with horizon distance, then adjust for qudrant if necessary
      Result : Long_Float := HD;
   begin
      if Above_Horizon then
         if HD >= 0.0 then
            if HD <= 90.0 then
               Result := -1.0;
            else
               Result := -179.0;
            end if;
         end if;
      else
         --  Below horizon
         if HD < 0.0 then
            if HD > -90.0 then
               Result := 1.0;
            else
               Result := 179.0;
            end if;
         end if;
      end if;
      return Result;
   end Estimated_Tau;


   -- Distance of RA from east point on the equator expressed in
   -- the range -180.0 < HD <= 180.0
   function Horizon_Distance  return Long_Float is
      EP : Long_Float := RAMC + 90.0;
      HD : Long_Float;
   begin
      if EP >= 360.0 then
         EP := EP - 360.0;
      end if;

      HD := RA - EP;
      return Normalize (HD);
   end Horizon_Distance;


   --  The house position formula we try to solve.  In his thesis, the
   --  formula is given as:
   --
   --  Sin (tau - ra - ramc + 90) +
   --     Tan (d) (1 - abs tau / 90) Tan (dyn_angle) = 0
   --
   --  This has the problem that a couple of the trig functions used
   --  can blow up with invalid arguments at extreme latitudes.  The
   --  formula here factors Tan (dyn_angle) into Sin (dyn_angle) /
   --  Cos (dyn_angle), then multiplies the formula by Cos (dyn_angle),
   --  giving:
   --
   --  Cos (dyn_angle) Sin (tau - ra - ramc + 90) +
   --     Tan (d) (1 - abs tau / 90) Sin (dyn_angle) = 0
   --
   --  This is substantially better-behaved.  We're solving for Tau.
   function Delta_Fn (Tau : Long_Float) return Long_Float is
      HD : Long_Float := Horizon_Distance;
   begin
      return
        Cos (Dynamic_Angle, 360.0) * Sin (Tau - HD, 360.0)
             + Tan (Decl, 360.0) * (1.0 - abs Tau / 90.0)
             * Sin (Dynamic_Angle, 360.0);
   end Delta_Fn;


   --  Binary search.  If generally useful (likely), move to "utilites"
   --  as a global facility later.  Will need to generalize function
   --  format and arguments as it currently relies on globals of this
   --  package
   function Binary_Search (Func : Fn) return Long_Float is
      HD : Long_Float := Horizon_Distance;
      Lo, Hi, Tau, Delta_Tau, Lo_delta : Long_Float;
   begin
      -- Generate an estimate for terrestrial longitude
      Tau := Estimated_Tau (HD);

      --  Set iteration endpoints based on quadrant.  This approach
      --  fails to catch two of three solutions for triplicity zone,
      --  but beats the earlier implementation that raised an exception
      --  when taking tan (dynamic_angle) near 90, or arcsin for extreme
      --  latitudes.  "Delta_Fn" was modified to eliminate these problems
      if Tau in 0.0 .. 90.0 then
         Lo := 0.0;
         Hi := 90.0;
      elsif Tau in 90.0 .. 180.0 then
         Lo := 90.0;
         Hi := 180.0;
      elsif Tau in -90.0 .. 0.0 then
         Lo := -90.0;
         Hi := 0.0;
      else
         Lo := -180.0;
         Hi := -90.0;
      end if;

      --  Generate delta of function (from 0) for low bound of the search
      Lo_Delta := Func.all (Lo);

      loop
         --  Generate delta based on current estimated terrestrial longitude.
         --  The accuracy threshold is determined empirically to correspond
         --  to a calculation error of +/- .05" in tau, which is much more
         --  precise than the inputs for location and time warrant.
         --  Theoretical precision of tau is on the order of +/- 30", though
         --  I've seen differences from the Concept program on the order of
         --  up to 10'.

         Delta_Tau := Func.all (Tau);
         exit when abs Delta_Tau <= 5.0E-8;

         --  Swap estimated terrestrial longitude to appropriate bound
         --  and generate new estimate
         if Delta_Tau * Lo_Delta > 0.0 then
            Lo := Tau;  Lo_Delta := Delta_Tau;
         else
            Hi := Tau;
         end if;
         Tau := (Lo + Hi) / 2.0;
      end loop;

      return Tau;
   end Binary_Search;


   function House_Position return Long_Float is
      Result : Long_Float;
   begin
      --  Iterate for solution
      Result := Binary_Search (Delta_Fn'Access);

      --  Convert to range 0.0 <= Result < 360.0
      if Result < 0.0 then
         Result := Result + 360.0;
      end if;

      return Result;
   exception
      when others =>
         raise Calculation_Failed;
   end House_Position;


   --  Normalize value to range -180.0 < Result <= 180.0:
   function Normalize (Long : Long_Float) return Long_Float is
      Result : Long_Float := Long;
   begin
      if Result > 180.0 then
         Result := Result - 360.0;
      elsif Result <= -180.0 then
         Result :=  Result + 360.0;
      end if;

      --  Ensure in range
      pragma Assert (-180.0 < Result and Result <= 180.0, "HD out of range");
      return Result;
   end Normalize;

end Reporters.Terra.Calculations;
