-----------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                                REPORTERS                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Strings.Fixed; use  Ada.Strings.Fixed;

with Gtkada.Types; use Gtkada.Types;
with Glib; use Glib;

with Body_Models; use Body_Models;
with Positions_Views; use Positions_Views;

--  Common part of views on GTK+ Columned Lists used to display
--  coordinates in the positions table for an event
package body Reporters is

   ----------------------
   --  Implementation  --
   ----------------------

   --  Update the table for a coordinate set within the positions table
   --  dialog (template method)
   procedure Update_Table (
                    This : access Reporter;
                    Model : access Event_Model'Class;
                    Evt : access Event'Class);


   ---------------
   --  Queries  --
   ---------------

   function Table (This : access Reporter'Class) return Gtk_Clist is
   begin
      return This.Table;
   end Table;


   ------------------------
   --  Basic Operations  --
   ------------------------

   --  Format a table
   procedure Format_Table
     (R : access Reporter'Class) is
   begin
      for I in 0 .. Get_Columns (R.Table) loop
         Set_Column_Justification
           (R.Table, I, Justification (R));
      end loop;
   end Format_Table;


   --  Update the table for a coordinate set within the positions table
   --  dialog (template method)
   procedure Update (This: access Reporter;
                     Source: access Observable'Class;
                     Evt: access Event'Class) is

      --  Cast Evt to Report_Event
      Model : Event_Model_Ptr :=
        Event_Model (Report_Event (Evt.all).Source.all)'Access;

   begin
      Update_Table (This, Model, Evt);
   end Update;


   ----------------------
   --  Implementation  --
   ----------------------

   -- Format string to fit table column
   function Pad (S : String; Justification : Alignment) return String is
      Result : Result_String;
   begin
      Move (S, Result, Justify => Justification);
      return Result;
   end Pad;


   -- Default justification in CLIST column
   function Justification (R : access Reporter) return Gtk_Justification is
   begin
      return Justify_Right;
   end Justification;

   --  Update the table for a coordinate set within the positions table
   --  dialog (template method)
   procedure Update_Table (
                    This: access Reporter;
                    Model : access Event_Model'Class;
                    Evt : access Event'Class) is

      -- Independent iterator on selected bodies
      Bodies : Body_Iterator :=
        Initialize (Event_Models.Body_Model (Model).all);

      Current_Body : Integer;
      Row : Gint;

      type Results is access Coordinates;

      Result : Results;
   begin
      -- Suppress display updates of CLIST
      Freeze (This.Table);

      -- Prepare CLIST
      Clear (This.Table);

      --  Fill CLIST with cusps.  For now, just MC and ASC
      for Cusp in MC .. III loop
            --  Dispatch to specific coordinate kind reporter
            Result :=
              new Coordinates'(
                               Get_Coordinates
                               (Reporter_Ptr (This),
                                Model,
                                Update_Event
                                (Report_Event (Evt.all).Evt.all)'Access,
                                Cusp));
            --  Load result strings to CLIST
            case Result.Count is
               when 1 =>
                  Row := Append
                    (This.Table,
                     Result.Longitude + "");
               when 2 =>
                  Row := Append
                    (This.Table,
                     Result.Longitude + Result.Lat);
               when 3 =>
                  Row := Append
                    (This.Table,
                     Result.Longitude + Result.Direction + Result.Latitude);
            end case;
      end loop;


      -- Fill CLIST for selected bodies with the coordinate kind
      Start (Bodies);
      while not Off (Bodies) loop
         Current_Body := Item (Bodies);

         --  Dispatch to specific coordinate kind reporter
         Result :=
           new Coordinates'(
                            Get_Coordinates
                            (Reporter_Ptr (This),
                             Model,
                             Update_Event
                             (Report_Event (Evt.all).Evt.all)'Access,
                             Current_Body));

         --  Load result strings to CLIST
         case Result.Count is
            when 1 =>
               Row := Append
                 (This.Table,
                  Result.Longitude + "");
            when 2 =>
               Row := Append
                 (This.Table,
                  Result.Longitude + Result.Lat);
            when 3 =>
               Row := Append
                 (This.Table,
                  Result.Longitude + Result.Direction + Result.Latitude);
         end case;


         Forth (Bodies);
      end loop;

      --  Display CLIST
      Thaw (This.Table);
   end Update_Table;

end Reporters;
