-- Tests for Swiss Ephemeris Calculation Flag Management.

with AUnit.Test_Cases.Registration;
use AUnit.Test_Cases.Registration;

with AUnit.Assertions; use AUnit.Assertions;

package body Ephemeris.Flags.Test_Flags is

   function Test_Flag (Flags : Unsigned_32; Set : Unsigned_32)
                      return Boolean is
      use type Interfaces.Unsigned_32;
   begin
      return (Flags and Set) = Set;
   end Test_Flag;

   -------------------
   -- Test Routines --
   -------------------


   procedure Test_Swiss (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_JPL;
      Use_Moshier;
      Use_Swiss;
      Assert (Test_Flag (Calculation_Flags, Swiss),
              "Swiss flag not set");
      Assert (not Test_Flag (Calculation_Flags, JPL), "JPL flag  set");
      Assert (not Test_Flag (Calculation_Flags, Moshier),
              "Moshier flag set");
   end Test_Swiss;

   procedure Test_JPL (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Moshier;
      Use_Swiss;
      Use_JPL;
      Assert (not Test_Flag (Calculation_Flags, Swiss),
              "Swiss flag set");
      Assert (Test_Flag (Calculation_Flags, JPL), "JPL flag not set");
      Assert (not Test_Flag (Calculation_Flags, Moshier),
              "Moshier flag set");
   end Test_JPL;

   procedure Test_Moshier
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Swiss;
      Use_JPL;
      Use_Moshier;
      Assert (not Test_Flag (Calculation_Flags, Swiss),
              "Swiss flag set");
      Assert (not Test_Flag (Calculation_Flags, JPL), "JPL flag set");
      Assert (Test_Flag (Calculation_Flags, Moshier),
              "Moshier flag not set");
   end Test_Moshier;

   procedure Test_True_Positions
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Apparent_Positions;
      Use_True_Positions;
      Assert (Test_Flag (Calculation_Flags, True_Positions),
              "Light time set");
      Assert (Test_Flag (Calculation_Flags,
                         Gravitational_Deflection_Off),
              "Gravitational deflection set");
      Assert (Test_Flag (Calculation_Flags, Aberration_Off),
              "Aberration set");
   end Test_True_Positions;

   procedure Test_Apparent_Positions
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_True_Positions;
      Use_Apparent_Positions;
      Assert (not Test_Flag (Calculation_Flags, True_Positions),
              "Light time not set");
      Assert (not Test_Flag (Calculation_Flags,
                             Gravitational_Deflection_Off),
              "Gravitational deflection not set");
      Assert (not Test_Flag (Calculation_Flags, Aberration_Off),
              "Aberration not set");
   end Test_Apparent_Positions;

   procedure Test_Generate_Velocities
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      No_Velocities;
      Generate_Velocities;
      Assert (Test_Flag (Calculation_Flags, Velocities),
              "Velocities not set");
   end Test_Generate_Velocities;

   procedure Test_No_Velocities
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Generate_Velocities;
      No_Velocities;
      Assert (not Test_Flag (Calculation_Flags, Velocities),
              "Velocities set");
   end Test_No_Velocities;

   procedure Test_Nutation
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      No_Nutation;
      Use_Nutation;
      Assert (not Test_Flag (Calculation_Flags, Nutation_Off),
              "Nutation not set");
   end Test_Nutation;

   procedure Test_No_Nutation
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Nutation;
      No_Nutation;
      Assert (Test_Flag (Calculation_Flags, Nutation_Off),
              "Nutation set");
   end Test_No_Nutation;

   procedure Test_Cartesian
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Polar;
      Use_Cartesian;
      Assert (Test_Flag (Calculation_Flags, Cartesian),
              "Cartesian not set");
   end Test_Cartesian;

   procedure Test_Polar (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Cartesian;
      Use_Polar;
      Assert (not Test_Flag (Calculation_Flags, Cartesian),
              "Polar not set");
   end Test_Polar;

   procedure Test_Degrees
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Radians;
      Use_Degrees;
      Assert (not Test_Flag (Calculation_Flags, Radians),
              "Degrees not set");
   end Test_Degrees;

   procedure Test_Radians
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Degrees;
      Use_Radians;
      Assert (Test_Flag (Calculation_Flags, Radians),
              "Radians not set");
   end Test_Radians;

   procedure Test_Topocentric
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Heliocentric;
      Use_Topocentric;
      Assert (Test_Flag (Calculation_Flags, Topocentric),
              "Topocentric not set");
      Assert (not Test_Flag (Calculation_Flags, Heliocentric),
              "Heliocentric set");
   end Test_Topocentric;

   procedure Test_Heliocentric
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Topocentric;
      Use_Heliocentric;
      Assert (not Test_Flag (Calculation_Flags, Topocentric),
              "Topocentric set");
      Assert (Test_Flag (Calculation_Flags, Heliocentric),
              "Heliocentric not set");
   end Test_Heliocentric;

   procedure Test_Geocentric
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Heliocentric;
      Use_Geocentric;
      Assert (not Test_Flag (Calculation_Flags, Topocentric),
              "Topocentric set");
      Assert (not Test_Flag (Calculation_Flags, Heliocentric),
              "Heliocentric set");

      Use_Topocentric;
      Use_Geocentric;
      Assert (not Test_Flag (Calculation_Flags, Topocentric),
              "Topocentric set");
      Assert (not Test_Flag (Calculation_Flags, Heliocentric),
              "Heliocentric set");
   end Test_Geocentric;

   procedure Test_Ecliptical
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Equatorial;
      Use_Ecliptical;
      Assert (not Test_Flag (Calculation_Flags, Equatorial),
              "Ecliptic not set");
   end Test_Ecliptical;

   procedure Test_Equatorial
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Ecliptical;
      Use_Equatorial;
      Assert (Test_Flag (Calculation_Flags, Equatorial),
              "Equatorial not set");
   end Test_Equatorial;

   procedure Test_Sidereal
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Tropical;
      Use_Sidereal;
      Assert (Test_Flag (Calculation_Flags, Sidereal),
              "Sidereal not set");
   end Test_Sidereal;

   procedure Test_Tropical
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Sidereal;
      Use_Tropical;
      Assert (not Test_Flag (Calculation_Flags, Sidereal),
              "Tropical not set");
   end Test_Tropical;

   procedure Test_Equinox_Of_Date
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_J2000_Equinox;
      Use_Equinox_Of_Date;
      Assert (not Test_Flag (Calculation_Flags, J2000),
              "Equinox of date not set");
   end Test_Equinox_Of_Date;

   procedure Test_J2000 (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Use_Equinox_Of_Date;
      Use_J2000_Equinox;
      Assert (Test_Flag (Calculation_Flags, J2000),
              "J2000 Equinox not set");
   end Test_J2000;

   procedure Register_Tests (T : in out Test_Case) is
   begin
      null;
      Register_Routine (T, Test_Swiss'Access, "Test Select Swiss");
      Register_Routine (T, Test_JPL'Access, "Test Select JPL");
      Register_Routine (T, Test_Moshier'Access, "Test Select Moshier");
      Register_Routine (T, Test_True_Positions'Access, "Test True Positions");
      Register_Routine (T, Test_Apparent_Positions'Access,
                        "Test Apparent Positions");
      Register_Routine (T, Test_Generate_Velocities'Access,
                        "Test Generate Velocities");
      Register_Routine (T, Test_No_Velocities'Access, "Test No Velocities");
      Register_Routine (T, Test_Nutation'Access, "Test Nutation");
      Register_Routine (T, Test_No_Nutation'Access, "Test Nutation Off");
      Register_Routine (T, Test_Cartesian'Access, "Test Cartesian");
      Register_Routine (T, Test_Polar'Access, "Test Polar");
      Register_Routine (T, Test_Degrees'Access, "Test Degrees");
      Register_Routine (T, Test_Radians'Access, "Test Radians");
      Register_Routine (T, Test_Topocentric'Access, "Test Topocentric");
      Register_Routine (T, Test_Heliocentric'Access, "Test Heliocentric");
      Register_Routine (T, Test_Geocentric'Access, "Test Geocentric");
      Register_Routine (T, Test_Ecliptical'Access,
                        "Test Ecliptical Coordinates");
      Register_Routine (T, Test_Equatorial'Access,
                        "Test Equatorial Coordinates");
      Register_Routine (T, Test_Sidereal'Access, "Test Sidereal");
      Register_Routine (T, Test_Tropical'Access, "Test Tropical");
      Register_Routine (T, Test_Equinox_Of_Date'Access, "Test Equinox of Date");
      Register_Routine (T, Test_J2000'Access, "Test J2000 Equinox");
   end Register_Tests;

   --  Identifier of test case:
   function Name (T : Test_Case) return String_Access is
   begin
      return new String'("Test Calculation Flags");
   end Name;

end Ephemeris.Flags.Test_Flags;
