with AUnit.Test_Cases.Registration;
use AUnit.Test_Cases.Registration;

with AUnit.Assertions; use AUnit.Assertions;

with Decimal_Sexagesimal;
use Decimal_Sexagesimal;
package body Test_Dec_Sex_Conversions is


   -------------------
   -- Test Routines --
   -------------------


   procedure Test_Sexagesimal (T : in out AUnit.Test_Cases.Test_Case'Class) is
      S : Sexagesimal;
   begin
      S := Make_Sexagesimal (24, 12, 33.555555);
      Assert
        (abs (Seconds (To_Sexagesimal (To_Decimal (S)))
              - Seconds (S)) <= 0.0000139,   -- 0.05s
         "Sex -> Dec -> Sex positive failed" & ASCII.LF &
         "Expected: " & To_Degree_String (S) & ASCII.LF &
         "Got:"  & To_Degree_String (To_Sexagesimal (To_Decimal (S))));


      S := Make_Sexagesimal (-89, -59, -0.12345);
      Assert
        (abs (Seconds (To_Sexagesimal (To_Decimal (S)))
              - Seconds (S)) <= 0.0000139,   -- 0.05s
         "Sex -> Dec -> Sex negative failed" & ASCII.LF &
         "Expected: " & To_Degree_String (S) & ASCII.LF &
         "Got: " & To_Degree_String (To_Sexagesimal (To_Decimal (S))));
   end Test_Sexagesimal;

   procedure Test_Decimal (T : in out AUnit.Test_Cases.Test_Case'Class) is
      D : Long_Float;
   begin
      D := 330.546;
      Assert
        (abs (To_Decimal (To_Sexagesimal (D))
              - D) <= 0.0000139,   -- 0.05s
         "Dec -> Sex -> Dec positive failed" & ASCII.LF &
         "Expected: " & Long_Float'Image (D) & ASCII.LF &
         "Got:"  & Long_Float'Image  (To_Decimal (To_Sexagesimal (D))));

      D := -89.33456;
      Assert
        (abs (To_Decimal (To_Sexagesimal (D))
              - D) <= 0.0000139,   -- 0.05s
         "Dec -> Sex -> Dec negative failed" & ASCII.LF &
         "Expected: " & Long_Float'Image (D) & ASCII.LF &
         "Got:"  & Long_Float'Image  (To_Decimal (To_Sexagesimal (D))));

   end Test_Decimal;

   procedure Test_Queries (T : in out AUnit.Test_Cases.Test_Case'Class) is
      S : Sexagesimal;
   begin
      S := Make_Sexagesimal (0, 0, 0.0);
      Assert (Integral (S) = 0 and Minutes (S) = 0 and Seconds (S) = 0.0,
              "Extraction of 0 fields failed");
      S := Make_Sexagesimal (359, 59, 11.456);
      Assert
        (Integral (S) = 359 and Minutes (S) = 59 and Seconds (S) = 11.456,
         "Extraction of positive fields failed");

      S := Make_Sexagesimal (-89, -59, -33.456);
      Assert
        (Integral (S) = -89 and Minutes (S) = -59 and Seconds (S) = -33.456,
         "Extraction of negative fields failed");
   end Test_Queries;

   procedure Test_Degree_String
     (T : in out AUnit.Test_Cases.Test_Case'Class) is
      S : Sexagesimal;
   begin
      Assert (To_Degree_String (-0.11) = "-0 06' 36""",
              "Negative value incorrectly reported" & ASCII.LF &
              "Expected: ""-0 06' 36""""" & ASCII.LF &
              "Got: """ & To_Degree_String (-0.11) & '"');

      S := Make_Sexagesimal (0, 0, -59.9999999);
      Assert (To_Degree_String (S) = "-0 01' 00""",
              "Negative value incorrectly rounded" & ASCII.LF &
              "Expected: ""-0 01' 00""""" & ASCII.LF &
              "Got: """ & To_Degree_String (S) & '"');

      Assert (To_Degree_String (Make_Sexagesimal (0, 0, 0.0)) = "0 00' 00""",
              "0 degree string incorrectly formatted" & ASCII.LF &
              "Expected: ""0 00' 00""" & ASCII.LF &
              "Got: """ & To_Degree_String (Make_Sexagesimal (0, 0, 0.0)) &
              '"');

      Assert
        (To_Degree_String (Make_Sexagesimal (356, 1, 22.5)) = "356 01' 23""",
         "Positive degree string incorrectly formatted" & ASCII.LF &
         "Expected: ""356 01' 23""" & ASCII.LF &
              "Got: """ & To_Degree_String (Make_Sexagesimal (356, 1, 22.5)) &
              '"');

      Assert
        (To_Degree_String (Make_Sexagesimal (-56, -1, -22.5)) = "-56 01' 23""",
         "Negative degree string incorrectly formatted" & ASCII.LF &
         "Expected: ""-56 01' 23""" & ASCII.LF &
              "Got: """ & To_Degree_String (Make_Sexagesimal (-56, -1, -22.5)) &
              '"');

      Assert (To_Degree_String (Make_Sexagesimal (0, 0, 0.0)) =
              To_Degree_String (To_Decimal (Make_Sexagesimal (0, 0, 0.0))),
              "0 decimal degree string incorrectly formatted" & ASCII.LF &
              "Expected: ""0 00' 00""" & ASCII.LF &
              "Got: """ &
              To_Degree_String (To_Decimal (Make_Sexagesimal (0, 0, 0.0))) &
              '"');

      Assert
        (To_Degree_String (Make_Sexagesimal (356, 1, 22.5)) =
         To_Degree_String (To_Decimal (Make_Sexagesimal (356, 1, 22.5))),
         "Positive decimal degree string incorrectly formatted" & ASCII.LF &
         "Expected: ""356 01' 23""" & ASCII.LF &
         "Got: """ &
         To_Degree_String (To_Decimal (Make_Sexagesimal (356, 1, 22.5))) &
              '"');

      Assert
        (To_Degree_String (Make_Sexagesimal (-56, -1, -22.5)) =
         To_Degree_String (To_Decimal (Make_Sexagesimal (-56, -1, -22.5))),
         "Negative decimal degree string incorrectly formatted" & ASCII.LF &
         "Expected: ""-56 01' 23""" & ASCII.LF &
         "Got: """ &
         To_Degree_String (To_Decimal (Make_Sexagesimal (-56, -1, -22.5))) &
              '"');

   end Test_Degree_String;

   procedure Test_Time_String (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Assert (To_Time_String (Make_Sexagesimal (0, 0, 0.0)) = "0:00:00",
              "0 time string incorrectly formatted" & ASCII.LF &
              "Expected: ""0:00:00" & ASCII.LF &
              "Got: """ & To_Time_String (Make_Sexagesimal (0, 0, 0.0)) &
              '"');

      Assert
        (To_Time_String (Make_Sexagesimal (23, 1, 22.5)) = "23:01:23",
         "Positive time string incorrectly formatted" & ASCII.LF &
         "Expected: ""23:01:23" & ASCII.LF &
              "Got: """ & To_Time_String (Make_Sexagesimal (23, 1, 22.5)) &
              '"');

      Assert (To_Time_String (Make_Sexagesimal (0, 0, 0.0)) =
              To_Time_String (To_Decimal (Make_Sexagesimal (0, 0, 0.0))),
              "0 decimal time string incorrectly formatted" & ASCII.LF &
              "Expected: ""0:00:00" & ASCII.LF &
              "Got: """ &
              To_Time_String (To_Decimal (Make_Sexagesimal (0, 0, 0.0))) &
              '"');

      Assert
        (To_Time_String (Make_Sexagesimal (23, 1, 22.5)) =
         To_Time_String (To_Decimal (Make_Sexagesimal (23, 1, 22.5))),
         "Positive decimal time string incorrectly formatted" & ASCII.LF &
         "Expected: ""23:01:23" & ASCII.LF &
         "Got: """ &
         To_Time_String (To_Decimal (Make_Sexagesimal (356, 1, 22.5))) &
              '"');

   end Test_Time_String;

   procedure Register_Tests (T : in out Test_Case) is
   begin
      null;
      Register_Routine (T, Test_Sexagesimal'Access, "Test Sexagesimal Conversions");
      Register_Routine (T, Test_Decimal'Access, "Test Decimal Conversion");
      Register_Routine (T, Test_Queries'Access, "Test Field Queries");
      Register_Routine (T, Test_Degree_String'Access, "Test Generation of Degree String");
      Register_Routine (T, Test_Time_String'Access, "Test Time String Generation");
   end Register_Tests;

   --  Identifier of test case:
   function Name (T : Test_Case) return String_Access is
   begin
      return new String'("Test Decimal / Sexagesimal Conversions");
   end Name;

end Test_Dec_Sex_Conversions;
