with AUnit.Test_Cases.Registration;
use AUnit.Test_Cases.Registration;

with AUnit.Assertions; use AUnit.Assertions;

with Field_Validation; use Field_Validation;
with Times; use Times;

with Ephemeris; use Ephemeris;
package body Test_Field_Validation is


   -------------------
   -- Test Routines --
   -------------------


   procedure Test_Day (T : in out AUnit.Test_Cases.Test_Case'Class) is
      S : String (1..3);
   begin
      S := "001";
      S := Validate_Day (S);
      Assert (S = "001", "Validation changed string value");

      S := "031";
      S := Validate_Day (S);
      Assert (S = "031", "Validation changed string value");

      declare
      begin
         S := "-01";
         S := Validate_Day (S);
         Assert (False, "Negative day string accepted");
      exception
         when Invalid_Date => null;
      end;

      declare
      begin
         S := "000";
         S := Validate_Day (S);
         Assert (False, "Zero day string accepted");
      exception
         when Invalid_Date => null;
      end;

      declare
      begin
         S := "032";
         S := Validate_Day (S);
         Assert (False, "Out of range day string accepted");
      exception
         when Invalid_Date => null;
      end;

   exception
      when Invalid_Date =>
         Assert (False, "Rejected valid day string " & S);
   end Test_Day;

   procedure Test_Month (T : in out AUnit.Test_Cases.Test_Case'Class) is
      S : String (1..3);
   begin
      for M in Months loop
         S := Months'Image (M);
         S := Validate_Month (S);
         Assert (S = Validate_Month (S),
                 "Validate changed value of month string");
      end loop;

      declare
         S2 : String := "January";
      begin
         S2 := Validate_Month (S2);
         Assert (False, "Validate accepted invalid month string");
      exception
         when Invalid_Date => null;
      end;
   exception
      when Invalid_Date =>
         Assert (False, "Rejected valid month string " & S);
   end Test_Month;


   procedure Test_Year (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Assert (Validate_Year ("") = "0", "Failed to zero empty year string");
      Assert (Validate_Year ("  ") = "0", "Failed to zero blank year string");
      Assert (Validate_Year ("1951") = "1951", "Changed positive year string");
      Assert (Validate_Year ("-10") = "-10", "Changed negative year string");

      declare
         S : String := "x";
      begin
         S := Validate_Year (S);
         Assert (False, "Invalid year string accepted");
      exception
         when Invalid_Date => null;
      end;

   exception
      when Invalid_Date =>
         Assert (False, "Rejected valid year string ");
   end Test_Year;

   procedure Test_Latitude (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      -- Valid strings:
      Assert (Validate_Latitude ("0") = "0",
              "Changed zero latitude");
      Assert (Validate_Latitude ("90") = "90",
              "Changed positive latitude");

      -- Invalid strings:
      declare
         S : String := "";
      begin
         S := Validate_Latitude (S);
         Assert (False, "Empty latitude accepted");
      exception
         when Invalid_Latitude => null;
      end;

      declare
         S : String := "  ";
      begin
         S := Validate_Latitude (S);
         Assert (False, "Blank latitude accepted");
      exception
         when Invalid_Latitude => null;
      end;

      declare
         S : String := "91";
      begin
         S := Validate_Latitude (S);
         Assert (False, "Positive out of range latitude accepted");
      exception
         when Invalid_Latitude => null;
      end;

      declare
         S : String := "-1";
      begin
         S := Validate_Latitude (S);
         Assert (False, "Negative out of range latitude accepted");
      exception
         when Invalid_Latitude => null;
      end;

      declare
         S : String := "x";
      begin
         S := Validate_Latitude (S);
         Assert (False, "Alpha latitude accepted");
      exception
         when Invalid_Latitude => null;
      end;
   exception
      when Invalid_Latitude =>
         Assert (False, "Rejected valid latitude string ");
   end Test_Latitude;

   procedure Test_Longitude (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      -- Valid strings:
      Assert (Validate_Longitude ("0") = "0",
              "Changed zero longitude");
      Assert (Validate_Longitude ("180") = "180",
              "Changed positive longitude");

      -- Invalid strings:
      declare
         S : String := "";
      begin
         S := Validate_Longitude (S);
         Assert (False, "Empty longitude accepted");
      exception
         when Invalid_longitude => null;
      end;

      declare
         S : String := "  ";
      begin
         S := Validate_Longitude (S);
         Assert (False, "Blank longitude accepted");
      exception
         when Invalid_longitude => null;
      end;

      declare
         S : String := "181";
      begin
         S := Validate_Longitude (S);
         Assert (False, "Positive out of range longitude accepted");
      exception
         when Invalid_longitude => null;
      end;

      declare
         S : String := "-1";
      begin
         S := Validate_Longitude (S);
         Assert (False, "Negative out of range longitude accepted");
      exception
         when Invalid_longitude => null;
      end;

      declare
         S : String := "x";
      begin
         S := Validate_Longitude (S);
         Assert (False, "Alpha longitude accepted");
      exception
         when Invalid_longitude => null;
      end;
   exception
      when Invalid_longitude =>
         Assert (False, "Rejected valid longitude string ");
   end Test_Longitude;

   procedure Test_Altitude (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Assert (Validate_Altitude ("") = "0",
              "Failed to zero empty altitude string");
      Assert (Validate_Altitude ("  ") = "0",
              "Failed to zero blank altitude string");
      Assert (Validate_Altitude ("1951") = "1951",
              "Changed positive altitude string");
      Assert (Validate_Altitude ("-10") = "-10",
              "Changed negative altitude string");

      declare
         S : String := "x";
      begin
         S := Validate_Altitude (S);
         Assert (False, "Alpha altitude string accepted");
      exception
         when Invalid_Altitude => null;
      end;

   exception
      when Invalid_Altitude =>
         Assert (False, "Rejected valid altitude string ");
   end Test_Altitude;

   procedure Test_Hour (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Assert (Validate_Hour ("") = "0",
              "Failed to zero empty hour string");
      Assert (Validate_Hour ("  ") = "0",
              "Failed to zero blank hour string");
      Assert (Validate_Hour ("23") = "23",
              "Changed positive hour string");
      Assert (Validate_Hour ("00") = "00",
              "Changed zero hour string");

      declare
         S : String := "-1";
      begin
         S := Validate_Hour (S);
         Assert (False, "Negative hour string accepted");
      exception
         when Invalid_Time => null;
      end;

      declare
         S : String := "24";
      begin
         S := Validate_Hour (S);
         Assert (False, "Out of range hour string accepted");
      exception
         when Invalid_Time => null;
      end;

      declare
         S : String := "x";
      begin
         S := Validate_Hour (S);
         Assert (False, "Alpha hour string accepted");
      exception
         when Invalid_Time => null;
      end;

   exception
      when Invalid_Time =>
         Assert (False, "Rejected valid hour string ");
   end Test_Hour;

   procedure Test_Sexagesimal (T : in out AUnit.Test_Cases.Test_Case'Class) is
   begin
      Assert (Validate_Sexagesimal ("") = "0",
              "Failed to zero empty hour string");
      Assert (Validate_Sexagesimal ("  ") = "0",
              "Failed to zero blank hour string");
      pragma Assert (Validate_Sexagesimal ("0") = "0",
                     "Changed lower bound sexagesimal string");
      pragma Assert (Validate_Sexagesimal ("59") = "59",
                     "Changed lower bound sexagesimal string");

      -- Invalid strings:
      declare
         S : String := "-1";
      begin
         S := Validate_Sexagesimal (S);
         Assert (False, "Negative sexagesimal string accepted");
      exception
         when Invalid_Time => null;
      end;

      declare
         S : String := "60";
      begin
         S := Validate_Sexagesimal (S);
         Assert (False, "Out of range sexagesimal string accepted");
      exception
         when Invalid_Time => null;
      end;

      declare
         S : String := "x";
      begin
         S := Validate_Sexagesimal (S);
         Assert (False, "Alpha sexagesimal string accepted");
      exception
         when Invalid_Time => null;
      end;

    exception
      when Invalid_Time =>
         Assert (False, "Rejected valid sexagesimal string ");
   end Test_Sexagesimal;

   procedure Register_Tests (T : in out Test_Case) is
   begin
      null;
      Register_Routine (T, Test_Day'Access, "Test Day Strings");
      Register_Routine (T, Test_Month'Access, "Test Month String Validation");
      Register_Routine (T, Test_Year'Access, "Test year string validation");
      Register_Routine (T, Test_Latitude'Access, "Test latitude validation");
      Register_Routine (T, Test_Longitude'Access, "Test longitude strings");
      Register_Routine (T, Test_Altitude'Access, "Test altitude strings");
      Register_Routine (T, Test_Hour'Access, "Test hour strings");
      Register_Routine (T, Test_Sexagesimal'Access, "Test sexagesimal strings");
   end Register_Tests;

   --  Identifier of test case:
   function Name (T : Test_Case) return String_Access is
   begin
      return new String'("Test Validation of Input Strings");
   end Name;

end Test_Field_Validation;
