------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                            DECIMAL_SEXAGESIMAL                           --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                             --
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Strings.Fixed;
use Ada.Strings, Ada.Strings.Fixed;

-- Decimal / Sexagesimal Conversions.
package body Decimal_Sexagesimal is

   ----------------------
   --  Initialization  --
   ----------------------

   function Make_Sexagesimal
     (Integral : Integer; Minutes : Minutes_Part; Seconds : Long_Float)
     return Sexagesimal is

      pragma Assert
        ((not (Integral < 0) or (Minutes <= 0 and Seconds <= 0.0)) and then
         (not (Minutes < 0) or (Integral <= 0 and Seconds <= 0.0)) and then
         (not (Seconds < 0.0) or (Integral <= 0 and Minutes <= 0)),
         "Components have inconsistent signs");
   begin
      return (Integral, Minutes, Seconds);
   end Make_Sexagesimal;


   ---------------
   --  Queries  --
   ---------------

   function Integral (S : Sexagesimal) return Integral_Part is
   begin
      return S.Integral;
   end Integral;


   function Minutes (S : Sexagesimal) return Minutes_Part is
   begin
      return S.Minutes;
   end Minutes;


   function Seconds (S : Sexagesimal) return Long_Float is
   begin
      return S.Seconds;
   end Seconds;

   -- Round up, carrying overflows.  Restrict integral to mod "Cycle"
   procedure Round (S : in out Sexagesimal; Cycle : Integer) is
      pragma Assert (S.Integral >= 0, "Attempt to round negative Sexagesimal");
      Temp_Minutes : Integer := Integer (S.Minutes);
   begin
      if Integer (S.Seconds) >= 60 then
         S.Seconds := S.Seconds - 60.0;
         Temp_Minutes := Temp_Minutes + 1;
      end if;

      if Temp_Minutes >= 60 then
         S.Minutes := Minutes_Part (Temp_Minutes - 60);
         S.Integral := (S.Integral + 1) mod Cycle;
      else
         S.Minutes := Minutes_Part (Temp_Minutes);
      end if;
   end Round;


   function To_Degree_String (S : Sexagesimal) return String is
      Sign_String : String (1..1) := " ";
      Seconds_String : String (1..3);
      Minutes_String : String (1..3);
      Temp : Sexagesimal := S;
      Negative : Boolean :=
        S.Integral < 0 or else S.Minutes < 0 or else S.Seconds < 0.0;
   begin
      if Negative then
         Temp.Integral := abs S.Integral;
         Temp.Minutes := abs S.Minutes;
         Temp.Seconds := abs S.Seconds;
         Sign_String := "-";
      end if;

      if Integer (Temp.Seconds) = 60 then
         Round (Temp, 360);
      end if;

      Move (Trim (Minutes_Part'Image (Minutes_Part (Temp.Seconds)),
                  Both) & '"',
                  Seconds_String,
                  Justify => Right, Pad => '0');

      Move (Trim (Minutes_Part'Image (Temp.Minutes), Both) & "'",
            Minutes_String,
            Justify => Right, Pad => '0');

      return
        Trim (Sign_String, Both) &
        Trim (Integer'Image (Temp.Integral), Both) & " " &
        Minutes_String & " " &
        Seconds_String;
   end To_Degree_String;


   function To_Degree_String (D : Long_Float) return String is
   begin
      return To_Degree_String (To_Sexagesimal (D));
   end To_Degree_String;


   function To_Time_String (S : Sexagesimal) return String is
      Minutes_String : String (1..2);
      Seconds_String : String (1..2);
      Temp : Sexagesimal := S;
   begin
      if Integer (S.Seconds) = 60 then
         Round (Temp, 24);
      end if;

      Move (Trim (Minutes_Part'Image (Minutes_Part (Temp.Seconds)),
                  Both),
                  Seconds_String,
                  Justify => Right, Pad => '0');

      Move (Trim (Minutes_Part'Image (Temp.Minutes), Both),
            Minutes_String,
            Justify => Right, Pad => '0');

      return
        Trim (Integer'Image (Temp.Integral), Both) & ":" &
        Minutes_String & ":" &
        Seconds_String;
   end To_Time_String;

   function To_Time_String (D : Long_Float) return String is
   begin
      return To_Time_String (To_Sexagesimal (D));
   end To_Time_String;


   ------------------------
   --  Basic Operations  --
   ------------------------

   function To_Decimal (S : Sexagesimal) return Long_Float is
      Negative : Boolean :=
        S.Integral < 0 or else S.Minutes < 0 or else S.Seconds < 0.0;
      Result : Long_Float;
   begin
      Result := abs S.Seconds / 3600.0 + Long_Float (abs S.Minutes) / 60.0 +
        Long_Float (abs (S.Integral));
      if Negative then
         Result := - Result;
      end if;
      return Result;
   end To_Decimal;



   function To_Sexagesimal (D : Long_Float) return Sexagesimal is
      Negative : Boolean := D < 0.0;
      Result : Sexagesimal;
      Temp : Long_Float := abs (D);
   begin
      Result.Integral := Integer (Long_Float'Truncation (Temp));

      Temp := (Temp - Long_Float'Truncation (Temp)) * 60.0;

      Result.Minutes := Minutes_Part (Long_Float'Truncation (Temp));
      Result.Seconds := (Temp - Long_Float'Truncation (Temp)) * 60.0;

      if Negative then
         Result.Integral := - Result.Integral;
         Result.Minutes := - Result.Minutes;
         Result.Seconds := - Result.Seconds;
      end if;

      return Result;
   end To_Sexagesimal;

end Decimal_Sexagesimal;
