------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                              TIMES.STRINGS                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Strings.Fixed, Ada.Strings.Maps.Constants;
use Ada.Strings, Ada.Strings.Fixed;
use Ada.Strings.Maps, Ada.Strings.Maps.Constants;
with Ada.Exceptions; use Ada.Exceptions;

with Decimal_Sexagesimal; use Decimal_Sexagesimal;
with Field_Validation; use Field_Validation;

--  String / Value conversions for time and date types
package body Times.Strings is

   -------------
   --  Times  --
   -------------

   --  Formatted time string from HH.dddd format float
   function Time_String (Hours : Decimal_Hours) return String is
   begin
      return To_Time_String (Hours) & ' ';
   end Time_String;

   --  Formatted hour string from HH.dddd float
   function Hour_String (Hours : Decimal_Hours) return String is
      Temp : String := To_Time_String (Hours);
   begin
      return Temp (1 .. Index (Temp, ":") - 1);
   end Hour_String;

   --  Formatted minute string from HH.dddd float
   function Minute_String (Hours : Decimal_Hours) return String is
      Temp : String := To_Time_String (Hours);
   begin
      return
        Temp (Index (Temp, ":") + 1 .. Index (Temp ,":", Backward) - 1);
   end Minute_String;

   --  Formatted seconds string from HH.dddd float
   function Second_String (Hours : Decimal_Hours) return String is
      Temp : String := To_Time_String (Hours);
   begin
      return Temp (Index (Temp ,":", Backward) + 1 .. Temp'Last);
   end Second_String;

   --  Validate and convert strings to a time in a HH.dddd float
   function Time_Value (Hour, Minute, Second : String) return Decimal_Hours is
   begin
      return
        To_Decimal
        (Make_Sexagesimal (Natural'Value (Validate_Hour (Hour)),
                           Minutes'Value (Validate_Sexagesimal (Minute)),
                           Seconds'Value (Validate_Sexagesimal (Second))));
   exception
      when E : Invalid_Time =>
         Raise_Exception (Exception_Identity (E), "Bad Local Time Entered");
   end Time_Value;


   ------------------
   --  Time Zones  --
   ------------------

   --  Formatted zone string from +/- HH.dddd format float.
   --  Negative values are West by convention
   function Zone_String (Hours : Time_Zone) return String is
      Temp_Zone : String := To_Time_String (abs Hours);
      Zone : String := Temp_Zone (1 .. Temp_Zone'Last - 3) & "W ";
   begin
      if Hours > 0.0 then
         Zone (Zone'Last - 1) := 'E';
      end if;

      return Zone;
   end Zone_String;

   --  Zone hemisphere string from +/- HH.dddd format float
   function Zone_Hemi_String (Hours : Time_Zone) return String is
      Result : String := "West";
   begin
      if Hours > 0.0 then
         Result := "East";
      end if;
      return Result;
   end Zone_Hemi_String;

   --  Validate and convert strings to a time zone in +/- HH.dddd format
   function Zone_Value (Hour, Minute, Hemi : String) return Time_Zone is
      Zone_Hour : Time_Zone;
      Zone_Minute : Long_Float;
   begin
      Zone_Hour := Time_Zone (Hours'Value (Validate_Hour (Hour)));

      Zone_Minute :=
        Long_Float (Minutes'Value (Validate_Sexagesimal (Minute)));

      if Validate_EW (Hemi) = "East" then
         Zone_Hour := Zone_Hour + Zone_Minute / 60.0;
      else
         Zone_Hour := - Zone_Hour - Zone_Minute / 60.0;
      end if;

      return Zone_Hour;
   exception
      when E : Constraint_Error | Invalid_Time | Invalid_Longitude =>
         Raise_Exception (Exception_Identity (E), "Bad Time Zone Entered");
   end Zone_Value;



   -------------
   --  Dates  --
   -------------

   --  Formatted date string
   function Date_String (Day : Days; Month : Months; Year : Years)
                        return String is
      Temp : String :=
              Days'Image (Day) & " " &
              Months'Image (Month) (1..1) &
              Translate (Months'Image (Month) (2..3),
                         Lower_Case_Map) & " " &
              Trim (Years'Image (Year), Left) & ' ';

   begin
      return Temp (2 .. Temp'Last);
   end Date_String;

   --  Formatted date string with month and year only
   function Date_String (Month : Months; Year : Years)
                        return String is
      Temp : String :=
              Months'Image (Month) (1..1) &
              Translate (Months'Image (Month) (2..3),
                         Lower_Case_Map) & " " &
              Trim (Years'Image (Year), Left) & ' ';

   begin
      return Temp;
   end Date_String;

   --  Formatted day string
   function Day_String (Day : Days) return String is
   begin
      return Trim (Days'Image (Day), Both);
   end Day_String;

   --  Formatted month string
   function Month_String (Month : Months) return String is
   begin
      return
        Months'Image (Month) (1) &
        Translate (Months'Image (Month) (2..3), Lower_Case_Map);
   end Month_String;

   --  Formatted year string
   function Year_String (Year : Years) return String is
   begin
      return Trim (Years'Image (Year), Both);
   end Year_String;

   --  Validate and return day value
   function Day_Value (Day : String) return Days is
   begin
      return Days'Value (Validate_Day (Day));
   end Day_Value;

   --  Validate and return month value
   function Month_Value (Month : String) return Months is
   begin
      return Months'Value (Validate_Month (Month));
   end Month_Value;

   --  Validate and return year value
   function Year_Value (Year : String) return Years is
   begin
      return Years'Value (Validate_Year (Year));
   end Year_Value;

end Times.Strings;
