------------------------------------------------------------------------------
--                                                                          --
--                               AstroFrames                                --
--                                                                          --
--                                  TIMES                                   --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $
--                                                                          --
--                       Copyright (C) 2001 Ed Falis                        --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- This software  is maintained by Ed Falis (falis@adelphia.net)            --
--                                                                          --
------------------------------------------------------------------------------

--  Time and Date types and operations
package Times is

   --  Date / Time in Julian Day format
   subtype Julian_Day is Long_Float;

   --  Time components
   subtype Decimal_Hours is Long_Float range 0.0 .. 24.0;

   subtype Hours is Natural range 0 .. 23;
   subtype Minutes is Natural range 0 .. 59;
   subtype Seconds is Long_Float range 0.0 .. 60.0;

   --  Representation of a time zone.  Negative values are West
   subtype Time_Zone is Long_Float range -12.0 .. 12.0;

   --  Date components
   subtype Days is Positive range 1..31;
   subtype Years is Integer;

   --  Enumeration of months
   type Months is (Jan, Feb, Mar, Apr, May, Jun, Jul, Aug, Sep, Oct, Nov, Dec);

   --  Numerical representation of months
   for Months use
      (Jan => 1,
       Feb => 2,
       Mar => 3,
       Apr => 4,
       May => 5,
       Jun => 6,
       Jul => 7,
       Aug => 8,
       Sep => 9,
       Oct => 10,
       Nov => 11,
       Dec => 12);

   --  Numerical equivalents of months
   subtype Month_Range is Positive range 1 .. 12;

   --  Composite Date / Time
   type Date_Time_Components is record
      D : Days := 1;
      M : Months := Jan;
      Y : Years := 1900;
      Local_Time : Decimal_Hours := 0.0;
      Zone : Time_Zone := 0.0;
   end record;


   ---------------
   --  Queries  --
   ---------------

   -- Check validity of date.  Gregorian only.
   function Valid_Date (D : Days; M : Months; Y : Years) return Boolean;


   -------------------
   --  Conversions  --
   -------------------

   --  Convert date in Gregorian calendar to Julian day
   function Gregorian_To_Julian_Day
     (D : Days; M : Months; Y : Years; H : Decimal_Hours) return Julian_Day;

   --  Convert date in Julian calendar to Julian Day
   function Julian_To_Julian_Day
     (D : Days; M : Months; Y : Years; H : Decimal_Hours) return Julian_Day;


   -------------------
   --  Measurement  --
   -------------------

   --  Hours part of JD in H.dddd format
   function JD_Hours (JD : Julian_Day) return Decimal_Hours;

   --  Extract Gregorian calendar components from Julian day
   function Day (JD : Julian_Day) return Days;
   function Month (JD : Julian_Day) return Months;
   function Year (JD : Julian_Day) return Years;

   function Hour (JD : Julian_Day) return Hours;
   function Minute (JD : Julian_Day) return Minutes;
   function Second (JD : Julian_Day) return Seconds;

end Times;
