/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           ApplicationWindow.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "ApplicationWindow.h"
#endif

#include "ApplicationWindow.h"

#include <wx/string.h>

#include <wx/cmdline.h>
#include <wx/config.h>
#include <wx/filedlg.h>
#include <wx/filename.h>
#include <wx/image.h>
#include <wx/listbase.h>
#include <wx/mimetype.h>
#include <wx/log.h>
#include <wx/sizer.h>
#include <wx/textctrl.h>

#include "ChildWindow.h"
#include "Document.h"
#include "IconProvider.h"
#include "MainWindowControl.h"
#include "MenuProvider.h"
#include "ToolPanel.h"

#include "dialogs/AboutDialog.h"
#include "dialogs/ConfigDialog.h"
#include "dialogs/LocationDialog.h"

#include "config.h"
#include "constants.h"
#include "func.h"
#include "Base.h"
#include "Calculator.h"
#include "Conf.h"
#include "Session.h"
#include "Varga.h"

Config *config;

IMPLEMENT_APP( MaitreyaApp )

IMPLEMENT_CLASS( ApplicationWindow, wxFrame )
BEGIN_EVENT_TABLE( ApplicationWindow, wxFrame )
	EVT_MENU ( APP_FILENEW,  ApplicationWindow::OnFileNew )
	EVT_MENU ( APP_FILEOPEN,  ApplicationWindow::OnFileOpen )
	EVT_MENU ( APP_FILEQUIT,  ApplicationWindow::OnFileQuit )
	EVT_MENU ( APP_SHOWSTATUSBAR,  ApplicationWindow::OnToggleStatusBar )
	EVT_MENU ( APP_SHOWMAINTOOLBAR, ApplicationWindow::OnToggleToolBars )
	EVT_MENU ( APP_SAVECONFIG,  ApplicationWindow::OnSaveConfig )
	EVT_MENU ( APP_CONFIGURE,  ApplicationWindow::OnConfigure )
	EVT_MENU ( APP_LOCATION,  ApplicationWindow::OnLocationDialog )
	EVT_MENU ( APP_ABOUT,  ApplicationWindow::OnAppAbout )
	EVT_MENU ( APP_HELP,  ApplicationWindow::OnAppHelp )
	EVT_MENU_RANGE( APP_LIST_BIGICONS, APP_LIST_TABLE, ApplicationWindow::OnToggleIcons )
	EVT_MENU_RANGE( APP_SHOWHORA, APP_SHOWEPHEM, ApplicationWindow::OnNewChild )
	EVT_MENU_RANGE( CHILD_CLOSE, CHILD_NEW_RASI+30, ApplicationWindow::OnChildCommand )
	EVT_MENU_RANGE ( APP_FILERECENT, APP_FILERECENT+35, ApplicationWindow::OnOpenRecent )
	EVT_COMMAND( wxID_ANY, CONFIG_CHANGED, ApplicationWindow::OnConfigChanged )
	EVT_COMMAND( wxID_ANY, CHILD_SETACTIVE, ApplicationWindow::OnChildFocusEvent )
	EVT_COMMAND( wxID_ANY, CHILD_CLOSED, ApplicationWindow::OnChildCloseEvent )
	EVT_COMMAND( wxID_ANY, CREATE_ENTRY_CHART, ApplicationWindow::OnCreateEntryChart )
	EVT_COMMAND( wxID_ANY, DOC_SAVED, ApplicationWindow::OnDocSavedEvent )
	EVT_COMMAND( wxID_ANY, DOC_UPDATED, ApplicationWindow::OnDocUpdateEvent )
	EVT_CLOSE( ApplicationWindow::OnClose )
	EVT_DROP_FILES( ApplicationWindow::OnDropFilesEvent )
	EVT_LIST_ITEM_SELECTED( LISTCTRL_ID, ApplicationWindow::OnListItemSelected )
END_EVENT_TABLE()

/*****************************************************
**
**   MaitreyaApp   ---   Constructor 
**
******************************************************/
MaitreyaApp::MaitreyaApp()
 : wxApp()
{ 
} 

/*****************************************************
**
**   MaitreyaApp   ---   OnInit 
**
******************************************************/
bool MaitreyaApp::OnInit()
{
	wxString s;
	ApplicationWindow *frame = 0;

	SetAppName( APP_NAME );
	SetVendorName( VENDOR_NAME );

	Session *session = Session::get();
	session->init( argv[0] );

	::wxInitAllImageHandlers();
	frame = new ApplicationWindow( session->getGuiAppName(), config->xPosMasterWindow, config->yPosMasterWindow,
		config->xMasterWindow, config->yMasterWindow );
  frame->Show( true );
  SetTopWindow(frame);

	if ( argc <= 1 )
		{ if ( config->openNewDocOnStart ) frame->NewFile(); }
	for( int i = 1; i < argc; i++ ) frame->OpenFile( argv[i] ); 
#ifdef __WXMSW__
	frame->DragAcceptFiles( true );
#endif
	return true;
}

/*****************************************************
**
**   MaitreyaApp   ---   OnExit 
**
******************************************************/
int MaitreyaApp::OnExit()
{
	return 0;
}

/*****************************************************
**
**   ApplicationWindow   ---   Constructor 
**
******************************************************/
ApplicationWindow::ApplicationWindow(const wxChar *title, int xpos, int ypos, int width, int height)
	: wxFrame( 0, -1, title, wxPoint(xpos, ypos), wxSize(width, height))
{
	wxString rfiles, s;

	docommands = true;
	listctrl = new MainWindowControl( this, -1 );
	listctrl->Show( true );
	if ( config->logMode )
	{
		m_text = new wxTextCtrl(this, wxID_ANY, wxT("Log window.\n"),
			wxPoint(0, 250), wxSize(100, 150), wxTE_MULTILINE);
		m_logTargetOld = wxLog::SetActiveTarget(new wxLogTextCtrl(m_text));
		wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
		sizer->Add( listctrl, 1, wxEXPAND | wxALL, config->childFrameWidth );
		sizer->Add( m_text, 0, wxEXPAND, 0 );
		SetSizer( sizer );
		sizer->Fit( this );
		Layout();
	}
	else
	{
		(void)new wxLogNull();   
		//wxLog *log = new wxLogStream( &cout );
		//wxLog::SetActiveTarget( log );
	}
	SetIcon( IconProvider::getIcon( BITMAP_MAITREYA  ));

	docmanager = new DocumentManager;
	menubar = new AppMenuBar( wxMB_DOCKABLE );
	SetMenuBar( menubar );
	maintoolbar = new MyToolBar( this, -1, wxDefaultPosition, wxDefaultSize, wxTB_FLAT | wxTB_DOCKABLE );
	maintoolbar->Realize();
	SetToolBar( maintoolbar );
	showToolBars();
	statusbar = (MyStatusbar*)CreateStatusBar();
	showStatusBar();
	updateMenusAndToolbars( 0 );
}

/*****************************************************
**
**   ApplicationWindow   ---   Destructor 
**
******************************************************/
ApplicationWindow::~ApplicationWindow()
{
	delete docmanager;
	delete config;

}

/*****************************************************
**
**   ApplicationWindow   ---   OnOpenRecent 
**
******************************************************/
void ApplicationWindow::OnOpenRecent( wxCommandEvent &event )
{
	wxLogMessage( wxT( "ApplicationWindow::OnOpenRecent eventid %d" ), event.GetId() );
	OpenFile( menubar->getRecentFilesEntry( event.GetId()) );
}


/*****************************************************
**
**   ApplicationWindow   ---   SetActiveChild 
**
******************************************************/
void ApplicationWindow::SetActiveChild( ChildWindow *child )
{
	if ( ! listctrl->getNumberOfEntries() ) return;
	for( long i = 0; i < listctrl->getNumberOfEntries(); i++ )
	{
		if ( listctrl->getChild(i) == child )
		{
			listctrl->selectItem(i );
			updateMenusAndToolbars( child );
			return;
		}
	}
//	assert( 0 );
}

/*****************************************************
**
**   ApplicationWindow   ---   updateMenusAndToolbars 
**
******************************************************/
void ApplicationWindow::updateMenusAndToolbars( ChildWindow *child )
{
	bool isWin, isDoc, animated;
	int checkEW = 0;

	docommands = false;
	isWin = isDoc = animated = false;
	if ( child )
	{
		isWin = true;
		if ( child->getDoc() ) isDoc = true;
		if ( child->supportsEWToggle())
			{ checkEW = ( child->getPreferVedic() ? 1 : 2 ); }
		if ( isDoc && child->getDoc()->isAnimated() ) animated = true;
	}
	menubar->updateMenus( isWin, isDoc, animated, checkEW );
	maintoolbar->updateTools( isWin, isDoc, animated, checkEW );

	docommands = true;
}

/*****************************************************
**
**   ApplicationWindow   ---   OnChildClose 
**
******************************************************/
void ApplicationWindow::OnChildClose( ChildWindow *child )
{
  listctrl->deleteEntry( child );
	Document *doc = child->getDoc();
	if ( doc )
	{
		if ( child == doc->getMainWindow() )
		{
			docmanager->deleteDoc( doc );
		}
	}
	int nb = listctrl->getNumberOfEntries();
	ChildWindow *achild = 0;
	if ( nb > 0 ) achild = listctrl->getChild( 0 );
	updateMenusAndToolbars( achild );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnFileNew 
**
******************************************************/
void ApplicationWindow::OnFileNew( wxCommandEvent &event )
{
	NewFile();
}

/*****************************************************
**
**   ApplicationWindow   ---   NewFile 
**
******************************************************/
void ApplicationWindow::NewFile()
{
	ChildWindowFactory factory;

	Document *doc = WXDEBUG_NEW Document();
	doc->update();
	docmanager->addDoc( doc );
	ChildWindow *child = factory.createMainWindow( this, doc );
	AddChild( child );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnFileOpen 
**
******************************************************/
void ApplicationWindow::OnFileOpen( wxCommandEvent &event )
{
	wxString filetypes = wxT( "Maitreya (*.mtx)|*.mtx| Jagannatha Hora (*.jhd)|*.jhd| All files (*.*)|*.*" );
	wxFileDialog openFileDialog( this, _( "Open file"), config->defOpenPath, wxT( "" ),
		filetypes, wxOPEN, wxDefaultPosition );

	if (openFileDialog.ShowModal() == wxID_OK )
	{
		if ( OpenFile( openFileDialog.GetDirectory() + wxFileName::GetPathSeparator()  + openFileDialog.GetFilename() ))
			{ config->defOpenPath = openFileDialog.GetDirectory(); }
	}
}

/*****************************************************
**
**   ApplicationWindow   ---   OpenFile 
**
******************************************************/
bool ApplicationWindow::OpenFile( wxString filename )
{
	wxLogMessage( wxT( "ApplicationWindow::OpenFile %s" ), filename.c_str() );
	wxChar buf[256];

	Document *doc = WXDEBUG_NEW Document();
	if ( ! doc->openFile( filename ))
	{
		delete doc;
		wxSprintf( buf, wxT( "%s %s" ), _( "Cannot open file" ), filename.c_str() );
		doMessageBox( this, buf, wxOK | wxICON_ERROR );
		return false;
	}
	menubar->addToRecentFiles( filename );
	CreateNewChild( doc );
	return true;
}

/*****************************************************
**
**   ApplicationWindow   ---   CreateNewChild
**
******************************************************/
void ApplicationWindow::CreateNewChild( Document *doc )
{
	doc->update();
	docmanager->addDoc( doc );
	ChildWindow *child = ChildWindowFactory().createMainWindow( this, doc );
	AddChild( child );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnFileQuit 
**
******************************************************/
void ApplicationWindow::OnFileQuit( wxCommandEvent &event )
{
	Close( true );
}

/*****************************************************
**
**   ApplicationWindow   ---   AddChild 
**
******************************************************/
void ApplicationWindow::AddChild( ChildWindow *child )
{
	if ( child == 0 )
	{
		printf( "Error: child is null!\n" );
		return;
	}
	listctrl->addEntry( child );
	SetActiveChild( child );
	child->Show( true );
	child->postCreate();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnNewChild 
**
******************************************************/
void ApplicationWindow::OnNewChild( wxCommandEvent &event )
{
	ChildWindow *child = 0;
	wxLogMessage( wxT( "ApplicationWindow::OnNewChild %d" ), event.GetId() );
	ChildWindowFactory factory;

	switch( event.GetId() )
	{
		case APP_NEW_PARTNER:
			child = factory.createPartnerWindow( this, docmanager );
		break;
		default:
			child = factory.createChild( this, 0, event.GetId(), true );
		break;
	}
	AddChild( child );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnChildCommand 
**
******************************************************/
void ApplicationWindow::OnChildCommand( wxCommandEvent &event )
{
	int id = event.GetId();
	wxLogMessage( wxT( "ApplicationWindow::OnChildCommand %d" ), id );
	if ( ! docommands ) return;
	ChildWindow *child = listctrl->getActiveChild();

	if ( ! child )
	{
		wxLogMessage( wxT( "ApplicationWindow::OnChildCommand No active child" ));
		return;
	}
	if ( id < CHILD_NEW_TEXT )
	{
		if ( ! child->dispatchCommand( id ))
			wxLogError( wxT( "ApplicationWindow::OnChildCommand cannot dispatch command %d" ), id );
		if ( id == CHILD_EDITDATA ) listctrl->updateItemLabels();
	}
	else
	{
		// avoid that document dependent view is opened based upon non doc view, ugly but works
		if ( child->getDoc() == 0 &&
		( id == CHILD_NEW_TEXT || id == CHILD_NEW_TRANSIT || id == CHILD_NEW_SBC || id == CHILD_NEW_SOLAR
			|| id == CHILD_NEW_YOGA || id == CHILD_NEW_GRAPHICALDASA || id == CHILD_NEW_DASA
			|| id == CHILD_NEW_MIDPOINT || id == CHILD_NEW_WCHART || id == CHILD_NEW_PLANETLIST 
			|| id >= CHILD_NEW_RASI )) return;

		// may be done now
		AddChild( ChildWindowFactory().createChild( this, child->getDoc(), id, true ));
	}
}

/*****************************************************
**
**   ApplicationWindow   ---   saveConfig 
**
******************************************************/
void ApplicationWindow::saveConfig()
{
	wxLogMessage( wxT( "ApplicationWindow::saveConfig" ));

	wxRect rect = GetRect();
	config->xPosMasterWindow = rect.x;
	config->yPosMasterWindow = rect.y;
	config->xMasterWindow = rect.width;
	config->yMasterWindow = rect.height;
	menubar->saveRecentFiles();
	config->writeAll();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnClose 
**
******************************************************/
void ApplicationWindow::OnClose( wxCloseEvent &event )
{
	wxLogMessage( wxT( "Start ApplicationWindow::OnClose" ));
	long i;
	Document *doc;

	if ( config->askOnQuit )
	{
		if ( doMessageBox( this, _( "Do you want to quit?" ), wxYES_NO | wxCENTRE | wxICON_QUESTION ) == wxID_NO ) return;
	}
	wxLogMessage( wxT( "ApplicationWindow::OnClose -- Closing" ));

	for( i = 0; i < docmanager->getNbDocuments(); i++ )
	{
		doc = docmanager->getDocument( i );
		if ( ! doc->queryClose() ) return;
	}
	for( i = 0; i < listctrl->getNumberOfEntries(); i++ )
	{
		if ( ! listctrl->getChild( i )->Close()) return;
	}
	saveConfig();
	Destroy();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnSaveConfig 
**
******************************************************/
void ApplicationWindow::OnSaveConfig( wxCommandEvent &event )
{
	saveConfig();
	doMessageBox( this, _( "Configuration saved." ));
}

/*****************************************************
**
**   ApplicationWindow   ---   OnToggleIcons 
**
******************************************************/
void ApplicationWindow::OnToggleIcons( wxCommandEvent &event )
{
	config->listStyle = event.GetId() - APP_LIST_BIGICONS;
	listctrl->recreate();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnConfigure 
**
******************************************************/
void ApplicationWindow::OnConfigure( wxCommandEvent &event )
{
	wxLogMessage( wxT( "ApplicationWindow::OnConfigure" ));
	ConfigDialog( this ).ShowModal();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnLocationDialog 
**
******************************************************/
void ApplicationWindow::OnLocationDialog( wxCommandEvent &event )
{
  LocationDialog dialog( this );
	dialog.ShowModal();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnAppHelp 
**
******************************************************/
void ApplicationWindow::OnAppHelp( wxCommandEvent &event )
{
	if ( ! wxLaunchDefaultBrowser( URL )) doMessageBox( this, _( "Cannot launch Browser." ) );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnAppAbout 
**
******************************************************/
void ApplicationWindow::OnAppAbout( wxCommandEvent &event )
{
	AboutDialog( this ).ShowModal();
}

/**************************************************************
***
**  STATUS BAR
***
***************************************************************/
wxStatusBar *ApplicationWindow::OnCreateStatusBar(int number, long style, wxWindowID id, const wxString& name)
{
	return new MyStatusbar( this );
}

/*****************************************************
**
**   ApplicationWindow   ---   showStatusBar 
**
******************************************************/
void ApplicationWindow::showStatusBar()
{
	wxCoord w, h;
	statusbar->StartPlanetTimer( config->showStatusBar && config->showStatusInfo );
	statusbar->Show( config->showStatusBar );
	// Workaround 
	GetSize( &w, &h );
	SetSize( w+1, h );
	SetSize( w, h );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnToggleStatusBar 
**
******************************************************/
void ApplicationWindow::OnToggleStatusBar( wxCommandEvent &event )
{
	config->showStatusBar = event.IsChecked();
	showStatusBar();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnToggleToolBars 
**
******************************************************/
void ApplicationWindow::OnToggleToolBars( wxCommandEvent &event )
{
	switch( event.GetId() )
	{
		case APP_SHOWMAINTOOLBAR:
			config->showMainToolbar = event.IsChecked();
		break;
		default:
			assert( 0 );
		break;
	}
	showToolBars();
}

/*****************************************************
**
**   ApplicationWindow   ---   showToolBars 
**
******************************************************/
void ApplicationWindow::showToolBars()
{
	wxCoord w, h;
	maintoolbar->Show( config->showMainToolbar );
	// Workaround
	GetSize( &w, &h );
	SetSize( w+1, h );
	SetSize( w, h );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnDropFilesEvent 
**
******************************************************/
void ApplicationWindow::OnDropFilesEvent( wxDropFilesEvent &event )
{
	wxLogMessage( wxT( "ApplicationWindow::OnDropFilesEvent" ));
	for( int i = 0; i < event.GetNumberOfFiles(); i++ )
	{
		wxLogMessage( wxT( "ApplicationWindow::OnDropFilesEvent open %s"), (event.GetFiles())[i].c_str() );
		OpenFile( (event.GetFiles())[i].c_str() );
	}
}

/*****************************************************
**
**   ApplicationWindow   ---   OnConfigChanged 
**
******************************************************/
void ApplicationWindow::OnConfigChanged( wxCommandEvent &event )
{
	Document *doc;
	int i;
	Session::get()->getCalculator()->init();
	for( i = 0; i < docmanager->getNbDocuments(); i++ )
	{
		doc = docmanager->getDocument( i );
		if ( doc )
		{
			doc->update();
			doc->updateAllChildWindows();
		}
	}
	if ( config->showStatusBar )
	{
		((MyStatusbar*)GetStatusBar())->StartPlanetTimer( config->showStatusInfo );
	}
}

/*****************************************************
**
**   ApplicationWindow   ---   OnChildFocusEvent 
**
******************************************************/
void ApplicationWindow::OnChildFocusEvent( wxCommandEvent &event )
{
	ChildWindow *child = (ChildWindow*)event.GetEventObject();
	SetActiveChild( child );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnChildCloseEvent 
**
******************************************************/
void ApplicationWindow::OnChildCloseEvent( wxCommandEvent &event )
{
	ChildWindow *child = (ChildWindow*)event.GetEventObject();
	OnChildClose( child );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnNewChildEvent 
**
******************************************************/
void ApplicationWindow::OnNewChildEvent( wxCommandEvent &event )
{
	assert( 0 );
	ChildWindow *child = (ChildWindow*)event.GetEventObject();
	AddChild( child );
	echo( wxT( "New child created" ));
}

/*****************************************************
**
**   ApplicationWindow   ---   OnListItemSelected
**
******************************************************/
void ApplicationWindow::OnListItemSelected( wxListEvent& event )
{
	listctrl->setActiveItem( event.GetIndex() );
	updateMenusAndToolbars( listctrl->getChild( event.GetIndex() ));
}

/*****************************************************
**
**   ApplicationWindow   ---   OnDocSavedEvent 
**
******************************************************/
void ApplicationWindow::OnDocSavedEvent( wxCommandEvent &event )
{
	Document *doc = (Document*)event.GetEventObject();
	menubar->addToRecentFiles( ((FileDataSet*)doc->getDataSet())->getFilename() );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnDocUpdateEvent 
**
******************************************************/
void ApplicationWindow::OnDocUpdateEvent( wxCommandEvent &event )
{
	listctrl->arrangeIcons();
}

/*****************************************************
**
**   ApplicationWindow   ---   OnCreateEntryChart
**
******************************************************/
void ApplicationWindow::OnCreateEntryChart( wxCommandEvent &event )
{
	Document *d = (Document*)event.GetEventObject();
	CreateNewChild( d );
}

/*****************************************************
**
**   ApplicationWindow   ---   OnEventTest 
**
******************************************************/
void ApplicationWindow::OnEventTest( wxCommandEvent &event )
{
	//wxString *s = (wxString*)event.GetEventObject();
}

/*****************************************************
**
**   ApplicationWindow   ---   echo 
**
******************************************************/
void ApplicationWindow::echo( const wxChar *message )
{
	//((MyStatusbar*)GetStatusBar())->echo( message );
}

