/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Base.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Base.h"
#endif

#include "Base.h"

#include <wx/clipbrd.h>
#include <wx/dataobj.h>
#include <wx/dc.h>
#include <wx/dcmemory.h>
#include <wx/filedlg.h>
#include <wx/filename.h>
#include <wx/log.h>
#include <wx/msgdlg.h>
#include <wx/timer.h>
#include <wx/settings.h>
#include <wx/sizer.h>
#include <wx/html/htmlwin.h>

#include "constants.h"
#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "DataSet.h"
#include "FontProvider.h"
#include "Horoscope.h"
#include "Lang.h"
#include "Session.h"
#include "Writer.h"
#include "WriterFactory.h"
#include "dialogs/ExportDialog.h"

extern Config *config;

enum { STATUSBAR_PTIMER = wxID_HIGHEST + 1, STATUSBAR_MSGTIMER };

IMPLEMENT_CLASS( MyStatusbar, wxStatusBar )
BEGIN_EVENT_TABLE( MyStatusbar, wxStatusBar )
  EVT_TIMER( STATUSBAR_PTIMER, MyStatusbar::OnPlanetTimer )
  EVT_TIMER( STATUSBAR_MSGTIMER, MyStatusbar::OnMessageTimer )
END_EVENT_TABLE()

IMPLEMENT_CLASS( MyEventPropagator, wxEvtHandler )
BEGIN_EVENT_TABLE( MyEventPropagator, wxEvtHandler)
  EVT_RIGHT_DOWN( MyEventPropagator::OnContextMenuEvent )
	EVT_KEY_DOWN( MyEventPropagator::OnKeyDown )
END_EVENT_TABLE()

IMPLEMENT_CLASS( TextWidget, wxWindow )
BEGIN_EVENT_TABLE( TextWidget, wxWindow )
	EVT_SIZE( TextWidget::OnSize )
END_EVENT_TABLE()

IMPLEMENT_CLASS( PlanetaryPositionTextCtrl, TextWidget )

IMPLEMENT_CLASS( ExportableScrollWidget, wxScrolledWindow )
BEGIN_EVENT_TABLE( ExportableScrollWidget, wxScrolledWindow )
	EVT_PAINT( ExportableScrollWidget::OnPaint )
	EVT_MOTION( ExportableScrollWidget::OnMouseMotion )
	EVT_LEFT_DOWN( ExportableScrollWidget::OnMouseLeftDown )
	EVT_LEFT_UP( ExportableScrollWidget::OnMouseLeftUp )
	EVT_LEAVE_WINDOW( ExportableScrollWidget::OnMouseLeave )
END_EVENT_TABLE()

IMPLEMENT_CLASS( MySplitter, wxSplitterWindow )
BEGIN_EVENT_TABLE( MySplitter, wxSplitterWindow )
	EVT_RIGHT_DOWN( MySplitter::OnMouseRight )
	//EVT_SPLITTER_SASH_POS_CHANGED(wxID_ANY, MySplitter::OnPositionChanged)
	EVT_SIZE( MySplitter::OnSize )
END_EVENT_TABLE()

/**************************************************************
***
**  Static ids for text alignement
***
***************************************************************/
int Align::VCenter =  1;
int Align::Bottom  =  2;
int Align::Top     =  4; 
int Align::HCenter =  8;
int Align::Left    = 16;
int Align::Right   = 32;
int Align::Center  =  9; // AlignHCenter+AlignVCenter;

/**************************************************************
***
**
***
***************************************************************/
int mapCommand2ViewId( const int &command, int &vedic, int &subtype )
{
	switch( command )
	{
		case CHILD_NEW_TEXT:           return VIEW_TEXT; break;
		case CHILD_NEW_TRANSIT:        return VIEW_TRANSIT; break;
		case CHILD_NEW_SBC:            return VIEW_SBC; break;
		case CHILD_NEW_SOLAR:          return VIEW_SOLAR; break;
		case CHILD_NEW_DASA:           return VIEW_DASA; break;
		case CHILD_NEW_GRAPHICALDASA:  return VIEW_GRAPHICALDASA; break;
		case CHILD_NEW_MIDPOINT:       return VIEW_MIDPOINT; break;
		case CHILD_NEW_PLANETLIST:     return VIEW_PLANETLIST; break;
		case APP_SHOWHORA:             return VIEW_HORA; break;
		case APP_SHOWECLIPSE:          return VIEW_ECLIPSE; break;
		case APP_NEW_PARTNER:          return VIEW_PARTNER; break;
		case APP_SHOWEPHEM:            return VIEW_EPHEM; break;
		case CHILD_NEW_WCHART:
			vedic = false;
			subtype = 0;
			return VIEW_GRAPHIC;
		break;
		default:
			if ( command >= CHILD_NEW_RASI && command <= CHILD_NEW_RASI+30 )
			{
				vedic = true;
				subtype = command-CHILD_NEW_RASI;
				return VIEW_GRAPHIC;
			}
		break;
	}
	return 0;
}

/**************************************************************
***
**    move a window to the centre of the screen
***
***************************************************************/
void moveWindow2Center( wxWindow *w )
{
	int x = wxSystemSettings::GetMetric( wxSYS_SCREEN_X );
	int y = wxSystemSettings::GetMetric( wxSYS_SCREEN_Y );
	wxSize sz = w->GetSize();
	w->Move( ( x - sz.x ) / 2, ( y - sz.y ) / 2 );
}

/**************************************************************
***
**    doMessageBox
***
***************************************************************/
int doMessageBox( wxWindow *parent, const wxString message, const long style )
{
	Session *session = Session::get();
	return wxMessageDialog( parent, message, session->getGuiAppName(), style | wxCENTRE ).ShowModal();
}

/**************************************************************
***
**    getSizeForViewId
***
***************************************************************/
wxSize getSizeForViewId( const int &viewId, const bool vedic )
{
	wxSize size;
	switch( viewId )
	{
		case VIEW_PLANETLIST:
			size = wxSize( config->xMainWindow, config->yMainWindow );
		break;
		case VIEW_GRAPHIC:
			if ( vedic )
				size = wxSize( config->xVedicGraphicWindow, config->yVedicGraphicWindow );
			else
				size = wxSize( config->xWesternGraphicWindow, config->yWesternGraphicWindow );
		break;
		case VIEW_TEXT:
			size = wxSize( config->xTextWindow, config->yTextWindow );
		break;
		case VIEW_TRANSIT:
			size = wxSize( config->xTransitWindow, config->yTransitWindow );
		break;
		case VIEW_SBC:
			size = wxSize( config->xSbcWindow, config->ySbcWindow );
		break;
		case VIEW_SOLAR:
			size = wxSize( config->xSolarWindow, config->ySolarWindow );
		break;
		case VIEW_GRAPHICALDASA:
			size = wxSize( config->xGraphicalDasaWindow, config->yGraphicalDasaWindow );
		break;
		case VIEW_DASA:
			size = wxSize( config->xDasaWindow, config->yDasaWindow );
		break;
		case VIEW_MIDPOINT:
			size = wxSize( config->xMidpointWindow, config->yMidpointWindow );
		break;
		case VIEW_HORA:
			size = wxSize( config->xHoraWindow, config->yHoraWindow );
		break;
		case VIEW_ECLIPSE:
			size = wxSize( config->xEclipseWindow, config->yEclipseWindow );
		break;
		case VIEW_EPHEM:
			size = wxSize( config->xEphemWindow, config->yEphemWindow );
		break;
		case VIEW_PARTNER:
			size = wxSize( config->xPartnerWindow, config->yPartnerWindow );
		break;
		case VIEW_MULTIPLE:
			size = wxSize( config->xMultipleView, config->yMultipleView );
		break;
		default:
			printf( "Error: wrong view id %d in getSizeForViewId\n", viewId );
			assert( false );
		break;
	}

	//printf( "RETURN GetConfigSizeForViewId id %d x %d y %d\n", viewId, size.x, size.y );
	return size;
}

/**************************************************************
***
**    getSizeForViewId
***
***************************************************************/
void setConfigSizeForViewId( const int &viewId, const wxSize &size, const bool vedic )
{
	//printf( "Start setConfigSizeForViewId id %d x %d y %d\n", viewId, size.x, size.y );
	switch ( viewId )
	{
		case VIEW_PLANETLIST:
			config->xMainWindow = size.x;
			config->yMainWindow = size.y;
		break;
		case VIEW_GRAPHIC:
			if ( vedic )
			{
				config->xVedicGraphicWindow = size.x;
				config->yVedicGraphicWindow = size.y;
			}
			else
			{
				config->xWesternGraphicWindow = size.x;
				config->yWesternGraphicWindow = size.y;
			}
		break;
		case VIEW_TEXT:
			config->xTextWindow = size.x;
			config->yTextWindow = size.y;
		break;
		case VIEW_DASA:
			config->xDasaWindow = size.x;
			config->yDasaWindow = size.y;
		break;
		case VIEW_GRAPHICALDASA:
			config->xGraphicalDasaWindow = size.x;
			config->yGraphicalDasaWindow = size.y;
		break;
		case VIEW_TRANSIT:
			config->xTransitWindow = size.x;
			config->yTransitWindow = size.y;
		break;
		case VIEW_SBC:
			config->xSbcWindow = size.x;
			config->ySbcWindow = size.y;
		break;
		case VIEW_SOLAR:
			config->xSolarWindow = size.x;
			config->ySolarWindow = size.y;
		break;
		case VIEW_MIDPOINT:
			config->xMidpointWindow = size.x;
			config->yMidpointWindow = size.y;
		break;
		case VIEW_HORA:
			config->xHoraWindow = size.x;
			config->yHoraWindow = size.y;
		break;
		case VIEW_ECLIPSE:
			config->xEclipseWindow = size.x;
			config->yEclipseWindow = size.y;
		break;
		case VIEW_PARTNER:
			config->xPartnerWindow = size.x;
			config->yPartnerWindow = size.y;
		break;
		case VIEW_EPHEM:
			config->xEphemWindow = size.x;
			config->yEphemWindow = size.y;
		break;
		case VIEW_MULTIPLE:
			config->xMultipleView = size.x;
			config->yMultipleView = size.y;
		break;
		default:
			printf( "Error: wrong view id %d in setConfigSizeForViewId\n", viewId );
			exit( 1 );
		break;
	}
}

/**************************************************************
***
**    tellThatRestartIsRecommended
***
***************************************************************/
void tellThatRestartIsRecommended( wxWindow *parent )
{
    static bool alreadysaidthat = false;
    if ( ! alreadysaidthat ) doMessageBox( parent, _( "Restart of the application is recommended."));
    alreadysaidthat = true;
}

/*****************************************************
**
**   MyEventPropagator   ---   Constructor
**
******************************************************/
MyEventPropagator::MyEventPropagator()
 : wxEvtHandler()
{
}

/*****************************************************
**
**   MyEventPropagator   ---   OnContextMenuEvent
**
******************************************************/
void MyEventPropagator::OnContextMenuEvent( wxMouseEvent &event )
{
	event.ResumePropagation( wxEVENT_PROPAGATE_MAX );
	event.Skip();
}

/*****************************************************
**
**   MyEventPropagator   ---   OnKeyDown
**
******************************************************/
void MyEventPropagator::OnKeyDown( wxKeyEvent &event )
{
	event.ResumePropagation( wxEVENT_PROPAGATE_MAX );
	event.Skip();
}

/**************************************************************
***
**
***
***************************************************************/
void getRGB( const int &color, unsigned char &red, unsigned char &green, unsigned char &blue )
{
	int i;
	blue = (unsigned char)( color % 256 );
	i = ( color - blue ) / 256;
	green = (unsigned char)(i % 256);
	red = (unsigned char)(( i - green ) / 256 );
}

/**************************************************************
***
**
***
***************************************************************/
int setRGB( const unsigned char &red, const unsigned char &green, const unsigned char &blue )
{
	return (int)blue + 256 * ( (int)green + 256 * (int)red );
}

/*****************************************************
**
**   MySplitter   ---   Constructor 
**
******************************************************/
MySplitter::MySplitter( wxWindow *parent, const wxWindowID id, const wxPoint &pos, const wxSize size, long style )
 : wxSplitterWindow( parent, id, pos, size, style )
{
}

/*****************************************************
**
**   MySplitter   ---   OnMouseRight
**
******************************************************/
void MySplitter::OnMouseRight( wxMouseEvent& event)
{
	//event.Skip();
	int x = event.m_x;
	int y = event.m_y;
	wxWindow *window = (wxWindow*)event.GetEventObject();
	window->ClientToScreen( &x, &y );
	this->ScreenToClient( &x, &y );
	event.m_x = x;
	event.m_y = y;
	event.SetEventObject( this );
	GetParent()->ProcessEvent( event );
}

/*****************************************************
**
**   MySplitter   ---   OnPaint
**
******************************************************/
void MySplitter::OnPaint( wxPaintEvent &event )
{
	//printf( "MySplitter::OnPaint\n" );
}

/*****************************************************
**
**   MySplitter   ---   OnSize 
**
******************************************************/
void MySplitter::OnSize( wxSizeEvent &event )
{
	SizeWindows();
	if ( GetWindow1() ) GetWindow1()->Refresh();
	if ( GetWindow2()) GetWindow2()->Refresh();
	event.Skip();
}

/*****************************************************
**
**   MySplitter   ---   OnPositionChanged
**
******************************************************/
void MySplitter::OnPositionChanged(wxSplitterEvent& event)
{
	wxSize size1 = GetSize();
	size1.x = event.GetSashPosition();
	GetWindow1()->SetSize( size1 );
	wxSize size2 = GetSize();
	size2.x -= event.GetSashPosition();
	GetWindow2()->SetSize( size2 );
	event.Skip();
}

/*****************************************************
**
**   MyStatusbar   ---   Constructor 
**
******************************************************/
MyStatusbar::MyStatusbar( wxWindow *parent )
 : wxStatusBar( parent, -1, wxST_SIZEGRIP )
{
	SetFieldsCount( 2 );
	SetFont( *FontProvider().getDefaultFont());
	planettimer = new wxTimer( this, STATUSBAR_PTIMER );
	msgtimer = new wxTimer( this, STATUSBAR_MSGTIMER );
  PushEventHandler( new MyEventPropagator() );
}

/*****************************************************
**
**   MyStatusbar   ---   Destructor 
**
******************************************************/
MyStatusbar::~MyStatusbar()
{
	if ( planettimer->IsRunning() ) planettimer->Stop();
	if ( msgtimer->IsRunning() ) msgtimer->Stop();
}

/*****************************************************
**
**   MyStatusbar   ---   echo 
**
******************************************************/
void MyStatusbar::echo( const wxChar *message, const int timeout )
{
	//if ( msgtimer->IsRunning() ) msgtimer->Stop();
	//SetStatusText( message, 0 );
	//msgtimer->Start( 1000 * timeout, true );
}

/*****************************************************
**
**   MyStatusbar   ---   StartPlanetTimer 
**
******************************************************/
void MyStatusbar::StartPlanetTimer( bool b )
{
	if ( b )
	{
		if ( ! planettimer->IsRunning() ) planettimer->Start( 1000 );
	}
	else
	{
		if ( planettimer->IsRunning() ) planettimer->Stop();
    SetStatusText( wxT( "" ), 0 );
    SetStatusText( wxT( "" ), 1 );
	}
}

/*****************************************************
**
**   MyStatusbar   ---   OnMessageTimer 
**
******************************************************/
void MyStatusbar::OnMessageTimer( wxTimerEvent &event )
{
	//SetStatusText( "", 0 );
}

/*****************************************************
**
**   MyStatusbar   ---   OnPlanetTimer 
**
******************************************************/
void MyStatusbar::OnPlanetTimer( wxTimerEvent &event )
{
	wxString s;
  double mlen, asclen, dummy, nakportion;
  Lang *lang = Lang::get();
  Calculator *calculator = Session::get()->getCalculator();
  Formatter *formatter = Formatter::get();
	int naktype = ( config->iNakshatraMode28 ? N28 : N27 );
	DataSet d;

	d.setActualDate();
	d.setLocation( config->defaultLocation );
  asclen = calculator->calcAscendantAya( d.getJD(), d.getLocation()->getLatitude(),
		d.getLocation()->getLongitude(), config->preferVedic );

	calculator->calcPosition( &d, OMOON, mlen, dummy, true, config->preferVedic );

  s = lang->getObjectName( ( config->preferVedic ? IASCENDANT : WASCENDANT ), TSHORT );
	s << wxT( " " ) << formatter->getPosFormatted( asclen );
  SetStatusText( s, 0 );

  if ( config->preferVedic )
	{
		/*
    s.sprintf( wxT( "%s %s  -  %2.2f%% %s" ), lang->getObjectName( IMOON, TSHORT ), tmp,
			getNakshatraLength( mlen, naktype ) * 100 /NAKSHATRA_LEN,
			lang->getNakshatraName( getNakshatra( mlen, naktype ), naktype, TMEDIUM )  );
		*/
		nakportion = (int)(getNakshatraLength( mlen, naktype ) * 10000 /NAKSHATRA_LEN );
		nakportion /= 100;
    s = lang->getObjectName( IMOON, TSHORT );
		s << wxT( " " ) << formatter->getPosFormatted( mlen ) << wxT( " - " ) <<  nakportion
			<< wxT( "% " ) << lang->getNakshatraName( getNakshatra( mlen, naktype ), naktype, TMEDIUM );
	}
  else
	{
    //s.sprintf( wxT( "%s %s" ), lang->getObjectName( WMOON, TSHORT ), tmp );
    s = lang->getObjectName( WMOON, TSHORT );
		s << wxT( " " ) << formatter->getPosFormatted( mlen );
	}
  SetStatusText( s, 1 );
}

/*****************************************************
**
**   TextWidget   ---   Constructor 
**
******************************************************/
TextWidget::TextWidget( wxWindow *parent, const int &hmode, wxWindowID id, const wxPoint& pos,
	const wxSize& size )
 : wxWindow( parent, id, pos, size )
{
	FontProvider f;
	wxFont *defaultFont = f.getDefaultFont();
	wxFont *textFont = f.getTextFont();
	int ss = defaultFont->GetPointSize();
	int sizes[7] = { ss-4, ss-2, ss, ss+1, ss+2, ss+3, ss+5};

	htmlmode = hmode;
	if ( htmlmode )
	{
		textctrl = 0;
		htmlctrl = new wxHtmlWindow( this, -1, pos, size );
		htmlctrl->SetFonts( defaultFont->GetFaceName(), textFont->GetFaceName(), sizes );
		htmlctrl->PushEventHandler( new MyEventPropagator() );
		htmlctrl->Show( true );
	}
	else
	{
		htmlctrl = 0;
		textctrl = new wxTextCtrl( this, -1, wxT( "" ), pos, size, wxTE_MULTILINE|wxTE_READONLY|wxSUNKEN_BORDER );
		textctrl->SetFont( *textFont );
		textctrl->PushEventHandler( new MyEventPropagator() );
		textctrl->Show( true );
	}
	writer = WriterFactory().getWriter( htmlmode ? WRITER_HTML : WRITER_TEXT );
}

/*****************************************************
**
**   TextWidget   ---   selectAll
**
******************************************************/
void TextWidget::selectAll()
{
	if ( htmlmode )
	{
		htmlctrl->SelectAll();
	}
	else
	{
		textctrl->SetSelection( -1, -1 );
	}
}

/*****************************************************
**
**   TextWidget   ---   doCopy
**
******************************************************/
void TextWidget::doCopy()
{
  wxString s;
  if ( wxTheClipboard->Open() )
  {
		if ( htmlmode )
		{
			s = htmlctrl->ToText();
		}
		else
		{
			s = textctrl->GetStringSelection();
		}
    if ( s.IsEmpty() ) s = textctrl->GetValue();
    wxTheClipboard->SetData( new wxTextDataObject( s ));
    wxTheClipboard->Close();
  }
}

/*****************************************************
**
**   TextWidget   ---   setContents
**
******************************************************/
void TextWidget::setContents( const wxString &s )
{
	if ( htmlmode )
	{
		assert( htmlctrl );
		htmlctrl->SetPage( s );
	}
	else
	{
		assert( textctrl );
		textctrl->SetValue( s );
	}
}

/*****************************************************
**
**   TextWidget   ---   OnSize 
**
******************************************************/
void TextWidget::OnSize( wxSizeEvent &event )
{
	SetSize( event.GetSize() );
	if ( htmlctrl )
	{
		htmlctrl->SetSize( event.GetSize() );
		htmlctrl->Refresh();
	}
	else textctrl->SetSize( event.GetSize() );
}

/*****************************************************
**
**   PlanetaryPositionTextCtrl   ---   Constructor 
**
******************************************************/
PlanetaryPositionTextCtrl::PlanetaryPositionTextCtrl( wxWindow *parent, Horoscope *h, const int htmlmode,
	wxWindowID id, const wxPoint& pos, const wxSize& size )
 : TextWidget( parent, htmlmode, id, pos, size )
{
	horoscope = h;
	preferVedic = config->preferVedic;
	istyle = config->iGraphicStyle;
	wstyle = config->wGraphicStyle;
	isEmpty = false;
	emptyMessage = wxT( "empty" );
	prefix = _( "Chart" );
}

/*****************************************************
**
**   PlanetaryPositionTextCtrl   ---   dump
**
******************************************************/
void PlanetaryPositionTextCtrl::dump()
{
	wxString s1;
  Formatter *formatter = Formatter::get();
  Lang *lang = Lang::get();
  int i;
  double len;
  int nak_mode = ( config->iNakshatraMode28 ? N28 : N27 );

	writer->beginWriting();
	if ( isEmpty )
	{
		writer->writeHeader1( emptyMessage );
	}
	else
	{
		writer->writeHeader1( prefix );
				 
		int line = 1;
		if ( preferVedic )
		{
			Table table( 3, 11 );
			table.setHeader( 0, _( "Planet" ) );
			table.setHeader( 1, _( "Length" ) );
			table.setHeader( 2, _( "Nakshatra" ) );
			for( i = ISUN; i <= IASCENDANT; i++ )
			{
				table.setEntry( 0, line, lang->getObjectName( i, TSHORT ));
				len =  horoscope->getObjectLength(i);
				table.setEntry( 1, line, formatter->getPosFormatted( len,
					(( istyle & IGRAPHIC_STYLE_SHOW_RETRO ) ? horoscope->isObjectRetrograde(i) : false )));
				table.setEntry( 2, line, lang->getNakshatraName( getNakshatra( len, nak_mode ), nak_mode, TMEDIUM ) );
				line++;
			}
			writer->writeTable( table );
		}
		else
		{
			int numrows = 14;
			if ( wstyle & WGRAPHIC_STYLE_SHOW_URANIAN ) numrows += 8;
			if ( wstyle & WGRAPHIC_STYLE_SHOW_HOUSES ) numrows += 4;
			Table table( 2, numrows );
			table.setHeader( 0, _( "Planet" ) );
			table.setHeader( 1, _( "Length" ) );
			for( i = WSUN; i <= WPOSEIDON; i++ )
			{
				if ( ! ( wstyle & WGRAPHIC_STYLE_SHOW_URANIAN ) && i >= WCUPIDO ) continue;
				table.setEntry( 0, line, lang->getObjectName( i, TMEDIUM ));
				len =  horoscope->getObjectLength(i);
				table.setEntry( 1, line, formatter->getPosFormatted( len,
					(( wstyle & WGRAPHIC_STYLE_SHOW_RETRO ) ? horoscope->isObjectRetrograde(i) : false )));
				line++;
			}
			if ( wstyle & WGRAPHIC_STYLE_SHOW_HOUSES )
			{
				table.setEntry( 0, line, wxT( "2" ));
				table.setEntry( 1, line++, formatter->getPosFormatted( horoscope->getHouse( HOUSE2, preferVedic, false ), false ));

				table.setEntry( 0, line, wxT( "3" ));
				table.setEntry( 1, line++, formatter->getPosFormatted( horoscope->getHouse( HOUSE3, preferVedic, false ), false ));

				table.setEntry( 0, line, wxT( "11" ));
				table.setEntry( 1, line++, formatter->getPosFormatted( horoscope->getHouse( HOUSE11, preferVedic, false ), false ));

				table.setEntry( 0, line, wxT( "12" ));
				table.setEntry( 1, line++, formatter->getPosFormatted( horoscope->getHouse( HOUSE12, preferVedic, false ), false ));
			}
			writer->writeTable( table );
		}
	}
	writer->endWriting();
	setContents( writer->getContents());
}

/*****************************************************
**
**   PlanetaryPositionTextCtrl   ---   dispatchCommand
**
******************************************************/
bool PlanetaryPositionTextCtrl::dispatchCommand( const int &command )
{
  switch( command )
	{
		case CHILD_WSHOWASPECTS:
			wstyle & WGRAPHIC_STYLE_SHOW_ASPECTS ? wstyle &= ~WGRAPHIC_STYLE_SHOW_ASPECTS : wstyle |= WGRAPHIC_STYLE_SHOW_ASPECTS;
		break;
		case CHILD_WSHOWRETRO:
			 wstyle & WGRAPHIC_STYLE_SHOW_RETRO ? wstyle &= ~WGRAPHIC_STYLE_SHOW_RETRO : wstyle |= WGRAPHIC_STYLE_SHOW_RETRO;
		break;
		case CHILD_WSHOWHOUSES:
			wstyle & WGRAPHIC_STYLE_SHOW_HOUSES ? wstyle &= ~WGRAPHIC_STYLE_SHOW_HOUSES : wstyle |= WGRAPHIC_STYLE_SHOW_HOUSES;
		break;
		case CHILD_WSHOWURANIAN:
			wstyle & WGRAPHIC_STYLE_SHOW_URANIAN ? wstyle &= ~WGRAPHIC_STYLE_SHOW_URANIAN : wstyle |= WGRAPHIC_STYLE_SHOW_URANIAN;
		break;
		case CHILD_ISHOWRETRO:
			istyle & IGRAPHIC_STYLE_SHOW_RETRO ? istyle &= ~IGRAPHIC_STYLE_SHOW_RETRO : istyle |= IGRAPHIC_STYLE_SHOW_RETRO;
		break;
		case CHILD_ISHOWARUDHA:
			istyle & IGRAPHIC_STYLE_SHOW_ARUDHAS ? istyle &= ~IGRAPHIC_STYLE_SHOW_ARUDHAS : istyle |= IGRAPHIC_STYLE_SHOW_ARUDHAS;
		break;
		case CHILD_ISHOWLAGNAS:
			istyle & IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS ? istyle &= ~IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS :
				istyle |= IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS;
		break;
		case CHILD_ISHOWSOUTH:
			istyle |= IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		case CHILD_ISHOWNORTH:
			istyle &= ~IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle |= IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		case CHILD_ISHOWEAST:
			istyle &= ~IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle |= IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		default:
			return false;
		break;
	}
	dump();
	return true;
}

/*****************************************************
**
**   DcHelper   ---   drawTextFormatted
**
******************************************************/
void DcHelper::drawTextFormatted( const wxRect &rect, const wxString t, const int& align )
{
  wxCoord w, h;
  dc->GetTextExtent( t, &w, &h );
  int xmitte, ymitte, x, y;

  ymitte = 2 * rect.y + rect.height;
  ymitte /= 2;
  if ( align & Align::VCenter ) y = ymitte - h /2;
  else if ( align & Align::Top ) y = rect.y;
  else y = rect.y + rect.height - h;

  xmitte = 2 * rect.x + rect.width;
  xmitte /= 2;
  if ( align & Align::HCenter ) x = xmitte - w /2;
  else if ( align & Align::Left ) x = rect.x;
  else x = rect.x + rect.width - w;
  dc->DrawText( t, x, y );
}

/*****************************************************
**
**   ExportableScrollWidget   ---   Constructor
**
******************************************************/
ExportableScrollWidget::ExportableScrollWidget( wxWindow *parent, wxWindowID id, const wxPoint& pos,
	const wxSize& size )
 : wxScrolledWindow( parent, id, pos, size )
{
  hborder = 10;
  vborder = 10;

	preferVedic = config->preferVedic;
	dragMode = false;
	scrollx = scrolly = 1;
	SetScrollRate( scrollx, scrolly );
	PushEventHandler( new MyEventPropagator() );
}

/*****************************************************
**
**   ExportableScrollWidget   ---   OnPaint
**
******************************************************/
void ExportableScrollWidget::OnPaint( wxPaintEvent &event )
{
	wxPaintDC context( this );
	PrepareDC( context );
	dc = &context;
	doPaint( dc );
}

/*****************************************************
**
**   ExportableScrollWidget   ---   drawTextFormatted
**
******************************************************/
void ExportableScrollWidget::drawTextFormatted( const wxRect &rect, const wxString t, const int& align )
{
	DcHelper( dc ).drawTextFormatted( rect, t, align );
}

/*****************************************************
**
**   ExportableScrollWidget   ---   drawSimpleText 
**
******************************************************/
void ExportableScrollWidget::drawSimpleText( const wxRect &r, const wxString &s )
{
	drawTextFormatted( r, s, Align::Center );
}

/*****************************************************
**
**   ExportableScrollWidget   ---   doExport
**
******************************************************/
void ExportableScrollWidget::doExport()
{
  int exporttype;
  wxString filename, s, e;
	int x, y;

	GetSize( &x, &y );
	ExportDialog dialog( this, x, y );
	moveWindow2Center( &dialog );
  if ( dialog.ShowModal() != wxID_OK ) return;
	dialog.getSizes( x, y );
   
  wxString filetypes = _( "PNG (*.png)|*.png|JPG (*.jpg)|*.jpg|Bitmap (*.bmp)|*.bmp|PCX (*pcx)|*pcx|PNM (*.pnm)|*.pnm|TIFF (*.tif)|*.tif|All files (*)| *.*" );

  wxFileDialog exportFileDialog( this, _("Export Chart" ), config->defExportPath, wxT( "out.png"  ),
    filetypes, wxSAVE|wxOVERWRITE_PROMPT, wxDefaultPosition );
  
  if ( exportFileDialog.ShowModal() == wxID_OK )
  {
    filename = exportFileDialog.GetDirectory() + wxFileName::GetPathSeparator() + exportFileDialog.GetFilename();
		e = exportFileDialog.GetFilename().Right(3).MakeLower();
		if ( e == wxT( "png" )) exporttype = wxBITMAP_TYPE_PNG;
		else if ( e == wxT( "jpg" )) exporttype = wxBITMAP_TYPE_JPEG;
		else if ( e == wxT( "jpeg" )) exporttype = wxBITMAP_TYPE_JPEG;
		else if ( e == wxT( "bmp" )) exporttype = wxBITMAP_TYPE_BMP;
		else if ( e == wxT( "pnm" )) exporttype = wxBITMAP_TYPE_PNM;
		else if ( e == wxT( "pcx" )) exporttype = wxBITMAP_TYPE_PCX;
		else if ( e == wxT( "tif" )) exporttype = wxBITMAP_TYPE_TIF;
		else
		{
			s.Printf( _( "Can't determine image handler for extension \"%s\", using default (PNG)" ), e.c_str() );
      doMessageBox( this, s );
			filename.Append( wxT ( ".png" ));
			exporttype = wxBITMAP_TYPE_PNG;
		}
    config->defExportPath = exportFileDialog.GetDirectory();
    wxBitmap bitmap( x, y );
    wxMemoryDC dc;
    dc.SelectObject( bitmap );
    doPaint( &dc );
    if ( bitmap.SaveFile( filename, (wxBitmapType)exporttype ))
    {
      s.sprintf( wxT( "%s %s" ), _("Chart exported to "), filename.c_str() );
      doMessageBox( this, s );
    }
  }
}

/**************************************************************
***
**   ExportableScrollWidget   ---   OnMouseMotion
***
***************************************************************/
void ExportableScrollWidget::OnMouseMotion( wxMouseEvent &event )
{
	int dx, dy, vx, vy;
	if ( dragMode )
	{
		dx = event.m_x - mousex;
		dy = event.m_y - mousey;
		GetViewStart( &vx, &vy );
		Scroll( vx - dx / scrollx, vy - dy / scrolly );
	}
	mousex = event.m_x;
	mousey = event.m_y;
}

/**************************************************************
***
**   ExportableScrollWidget   ---   OnMouseLeftDown
***
***************************************************************/
void ExportableScrollWidget::OnMouseLeftDown( wxMouseEvent& )
{
	dragMode = true;
}

/**************************************************************
***
**   ExportableScrollWidget   ---   OnMouseLeftUp
***
***************************************************************/
void ExportableScrollWidget::OnMouseLeftUp( wxMouseEvent& )
{
	dragMode = false;
}

/**************************************************************
***
**   ExportableScrollWidget   ---   OnMouseLeave
***
***************************************************************/
void ExportableScrollWidget::OnMouseLeave( wxMouseEvent& )
{
	dragMode = false;
}


