/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           ChildWindow.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "ChildWindow.h"
#endif

#include "ChildWindow.h"
#include "MView.h"

#include <wx/string.h>

#include <wx/app.h>
#include <wx/imaglist.h>
#include <wx/notebook.h>
#include <wx/log.h>
#include <wx/menu.h>
#include <wx/panel.h>
#include <wx/sizer.h>
#include <wx/splitter.h>

#include "Base.h"
#include "Conf.h"
#include "Document.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "Shortcuts.h"

#include "func.h"
#include "ViewBuilder.h"

extern Config *config;

IMPLEMENT_CLASS( ChildWindow, wxFrame )
BEGIN_EVENT_TABLE( ChildWindow, wxFrame )
	EVT_ACTIVATE( ChildWindow::OnActivate )
	EVT_MENU_RANGE( CHILD_CLOSE, CHILD_NEW_RASI+30, ChildWindow::OnCommand )
	EVT_COMMAND_RANGE( CHILD_CLOSE, CHILD_NEW_RASI+30, wxID_ANY, ChildWindow::OnCommand )
	EVT_CLOSE( ChildWindow::OnClose )
	EVT_KEY_DOWN( ChildWindow::OnKeyDown )
END_EVENT_TABLE()

IMPLEMENT_CLASS( SimpleChildWindow, ChildWindow )
BEGIN_EVENT_TABLE( SimpleChildWindow, ChildWindow )
	EVT_SIZE( SimpleChildWindow::OnSize )
	EVT_ACTIVATE( SimpleChildWindow::OnActivate )
END_EVENT_TABLE()

IMPLEMENT_CLASS( MultipleChildWindow, ChildWindow )
BEGIN_EVENT_TABLE( MultipleChildWindow, ChildWindow )
	EVT_ACTIVATE( MultipleChildWindow::OnActivate )
	EVT_SIZE( MultipleChildWindow::OnSize )
END_EVENT_TABLE()

DEFINE_EVENT_TYPE( CHILD_ADDTORECENT )
DEFINE_EVENT_TYPE( CHILD_SETACTIVE )
DEFINE_EVENT_TYPE( CHILD_CLOSED )
DEFINE_EVENT_TYPE( CHILD_ADDED )

/*****************************************************
**
**   ChildWindow   ---   Constructor 
**
******************************************************/
ChildWindow::ChildWindow( wxFrame *parent, Document *document, const wxSize &size, const bool ismain )
	: wxFrame( parent, -1, wxT( "childwindow" ), wxDefaultPosition, size ), doc( document )
{
	isvalid = true;
	ismainwindow = false;
	setMainWindow( ismain );
	if ( doc && ! ismain ) doc->addChildWindow( this );
	setTitle();
	SetMinSize( wxSize( 200, 200 ));
}

/*****************************************************
**
**   ChildWindow   ---   Destructor 
**
******************************************************/
ChildWindow::~ChildWindow()
{
	if ( ismainwindow )
	{
		delete doc;
		doc  = 0;
	}
}

/*****************************************************
**
**   ChildWindow   ---   OnKeyDown 
**
******************************************************/
void ChildWindow::OnKeyDown( wxKeyEvent &event )
{  
	ShortcutManager sm;
	int i;

	i = sm.lookupChildWindowKeyEvent( event );
	if ( i )
	{
		wxMenuEvent evt( wxEVT_COMMAND_MENU_SELECTED, i );
		GetParent()->ProcessEvent( evt );
	}
	else
	{
		i = sm.lookupAppWindowKeyEvent( event );
		if ( i )
		{
			wxMenuEvent evt( wxEVT_COMMAND_MENU_SELECTED, i );
			GetParent()->ProcessEvent( evt );
		}
		else event.Skip();
	}
}

/*****************************************************
**
**   ChildWindow   ---   OnDataChanged 
**
******************************************************/
void ChildWindow::OnDataChanged()
{
	setTitle();
}

/*****************************************************
**
**   ChildWindow   ---   dispatchCommand 
**
******************************************************/
bool ChildWindow::dispatchCommand( const int &command )
{
	wxLogMessage( wxT( "ChildWindow::dispatchCommand command %d classname %s" ), command, GetClassInfo()->GetClassName());
	if ( command >= CHILD_NEW_TEXT && command <= CHILD_NEW_RASI+30 )
	{
		//Command( command );
		return true;
	}

	switch( command )
	{
		case CHILD_CLOSE:
			Close();
		break;
		case CHILD_CLOSE_DOC:
			doc->getMainWindow()->Close();
		break;
		case CHILD_SAVE:
			if ( ! doc ) return false;
			doc->save();
		break;
		case CHILD_SAVEAS:
			if ( ! doc ) return false;
			doc->saveAs();
		break;
		case CHILD_QUICKPRINT:
			if ( ! doc ) return false;
			doc->doQuickPrint();
		break;
		case CHILD_PRINT:
			if ( ! doc ) return false;
			doc->doPrint();
		break;
		case CHILD_EDITDATA:
			if ( ! doc ) return false;
			doc->editData();
		break;
		case CHILD_ANIMATE:
			if ( ! doc ) return false;
			doc->StartTimer( ! doc->isAnimated());
		break;

		//case CHILD_COPY:
		//break;
		default:
			//if ( view ) { return( view->dispatchCommand( command )); }
			//else return false;
			return false;
		break;
	}
	return true;
}

/*****************************************************
**
**   ChildWindow   ---   OnClose 
**
******************************************************/
void ChildWindow::OnClose( wxCloseEvent &event )
{
	wxLogMessage( wxT( "ChildWindow::OnClose classname %s" ), GetClassInfo()->GetClassName());

	if ( ismainwindow )
	{
		if ( ! doc->queryClose())
		{
			event.Veto();
			return;
		}
		doc->closeDependenChildWindows();
	}
	wxCommandEvent e( CHILD_CLOSED, GetId() );
	e.SetEventObject( this );
	GetParent()->ProcessEvent( e );
	if ( doc ) doc->releaseChildWindow( this );
	Destroy();
	isvalid = false;
}

/*****************************************************
**
**   ChildWindow   ---   showContextMenu 
**
******************************************************/
void ChildWindow::showContextMenu( wxMenu *menu, const int &x, const int &y )
{
	PopupMenu( menu, x, y);
	delete menu;
}

/*****************************************************
**
**   ChildWindow   ---   OnActivate
**
******************************************************/
void ChildWindow::OnActivate( wxActivateEvent &event )
{
	wxCommandEvent e( CHILD_SETACTIVE, GetId() );
	e.SetEventObject( this );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   ChildWindow   ---   OnCommand 
**
******************************************************/
void ChildWindow::OnCommand( wxCommandEvent &event )
{
	if ( ! isvalid ) return;

	if ( event.GetId() >= CHILD_NEW_TEXT && event.GetId() <= CHILD_NEW_RASI+30 )
	{
		event.Skip();
		return;
	}
	dispatchCommand( event.GetId() );
}


/*****************************************************
**
**   ChildWindow   ---   setMainWindow 
**
******************************************************/
void ChildWindow::setMainWindow( const bool &b )
{
	ismainwindow = b;
	if ( b ) doc->setMainWindow( this );
}

/*****************************************************
**
**   SimpleChildWindow   ---   Constructor 
**
******************************************************/
SimpleChildWindow::SimpleChildWindow( wxFrame *parent, Document *doc, const wxSize &size, const bool ismain )
 : ChildWindow( parent, doc, size, ismain )
{
	view = 0;
}

/*****************************************************
**
**   SimpleChildWindow   ---   Destructor 
**
******************************************************/
SimpleChildWindow::~SimpleChildWindow()
{
}

/*****************************************************
**
**   SimpleChildWindow   ---   insertView 
**
******************************************************/
void SimpleChildWindow::insertView( MView *v )
{
	view = v;
	setTitle();
}

/*****************************************************
**
**   SimpleChildWindow   ---   postCreate
**
******************************************************/
void SimpleChildWindow::postCreate()
{
	assert( view );
	view->postCreate();
}

/*****************************************************
**
**   SimpleChildWindow   ---   getViewId
**
******************************************************/
int SimpleChildWindow::getViewId()
{
	assert( view );
	return view->getViewInfo()->viewId;
}

/*****************************************************
**
**   SimpleChildWindow   ---   OnDataChanged 
**
******************************************************/
void SimpleChildWindow::OnDataChanged()
{
	view->OnDataChanged();
	ChildWindow::OnDataChanged();
}

/*****************************************************
**
**   SimpleChildWindow   ---   OnActivate
**
******************************************************/
void SimpleChildWindow::OnActivate( wxActivateEvent &event )
{
	if ( view )	view->SetFocus();	
	ChildWindow::OnActivate( event );
}

/*****************************************************
**
**   SimpleChildWindow   ---   OnSize 
**
******************************************************/
void SimpleChildWindow::OnSize( wxSizeEvent &event )
{
	wxSize size = event.GetSize();
	assert( view );
	//printf( "SimpleChildWindow::OnSize x %d y %d view is %d\n", size.x, size.y, view->getViewInfo()->viewId );
	setConfigSizeForViewId( view->getViewInfo()->viewId, size,  view->getPreferVedic());
	event.Skip();
}

/**************************************************************
***
**   SimpleChildWindow   ---   getContextMenu
***
***************************************************************/
wxMenu *SimpleChildWindow::getContextMenu()
{
	assert( view );
	ContextMenuInfo menuinfo( view );
	view->SetContextMenuInfo( menuinfo );
	return ContextMenuProvider().getContextMenu( menuinfo );
}

/*****************************************************
**
**   SimpleChildWindow   ---   getPreferVedic 
**
******************************************************/
bool SimpleChildWindow::getPreferVedic()
{
	if ( ! view ) return true;
	else return view->getPreferVedic();
}

/*****************************************************
**
**   SimpleChildWindow   ---   supportsEWToggle 
**
******************************************************/
bool SimpleChildWindow::supportsEWToggle()
{
	if ( ! view ) return false;
	else return view->supportsEWToggle();
}

/*****************************************************
**
**   SimpleChildWindow   ---   setTitle 
**
******************************************************/
void SimpleChildWindow::setTitle()
{
	if ( view ) SetTitle( view->getTitle() );
	else SetTitle( _( "empty" ));
}

/**************************************************************
***
**   SimpleChildWindow   ---   getWindowLabel
***
***************************************************************/
const wxChar *SimpleChildWindow::getWindowLabel( const bool shortname )
{
	if ( ! view ) return wxT( "empty" );
	else return view->getWindowLabel( shortname );
}

/*****************************************************
**
**   MultipleChildWindow   ---   Constructor 
**
******************************************************/
MultipleChildWindow::MultipleChildWindow( wxFrame *parent, Document *doc, const wxSize &size, const bool ismain )
 : ChildWindow( parent, doc, size, ismain )
{
	setTitle();
	container = 0;
	imagelist = new wxImageList( 24, 24, true );
	imagelist->Add( IconProvider::getBitmap( BITMAP_DOCUMENT ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_CHART ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_YOGA ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_DASA ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_TEXT ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_SOLAR ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_SBC ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_GRAPHICALDASA ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_MIDPOINT ));
	imagelist->Add( IconProvider::getBitmap( BITMAP_TRANSIT ));
}

/*****************************************************
**
**   MultipleChildWindow   ---   Destructor 
**
******************************************************/
MultipleChildWindow::~MultipleChildWindow()
{
	for ( unsigned int i = 0; i < viewlist.size(); i++ ) delete viewlist[i];
	viewlist.clear();
	if ( container ) delete container;
	delete imagelist;
}

/*****************************************************
**
**   MultipleChildWindow   ---   OnDataChanged 
**
******************************************************/
void MultipleChildWindow::OnDataChanged()
{
	for ( unsigned int i = 0; i < viewlist.size(); i++ ) viewlist[i]->OnDataChanged();
}

/**************************************************************
***
**  MultipleChildWindow   ---   getContextMenu
***
***************************************************************/
wxMenu *MultipleChildWindow::getContextMenu()
{
	ContextMenuInfo menuinfo( 0 );
	return ContextMenuProvider().getContextMenu( menuinfo );
}

/*****************************************************
**
**   MultipleChildWindow   ---   getPreferVedic 
**
******************************************************/
bool MultipleChildWindow::getPreferVedic()
{
	return true;
}

/*****************************************************
**
**   MultipleChildWindow   ---   supportsEWToggle 
**
******************************************************/
bool MultipleChildWindow::supportsEWToggle()
{
	return false;
}

/*****************************************************
**
**   MultipleChildWindow   ---   OnActivate
**
******************************************************/
void MultipleChildWindow::OnActivate( wxActivateEvent &event )
{
	if ( viewlist.size() > 0 ) viewlist[0]->SetFocus();
	ChildWindow::OnActivate( event );
}

/*****************************************************
**
**   MultipleChildWindow   ---   setTitle 
**
******************************************************/
void MultipleChildWindow::setTitle()
{
	if ( doc ) SetTitle( doc->getHName() );
}


/**************************************************************
***
**   MultipleChildWindow   ---   insertNode
***
***************************************************************/
wxWindow *MultipleChildWindow::insertNode( wxWindow *parent, ViewNode *node )
{
	int id = 0, subtype = 0, tp = true;
	bool vedic;
	MView *view = 0;
	wxWindow *window1 = 0, *window2 = 0;
	ViewFactory f;
	wxSize size = wxDefaultSize;
	wxString pagename;
	Session *session = Session::get();

	// only used for getttext extraction
	const wxChar *dummy_titles[] = { _( "Main" ), _( "Varga (2)" ), _( "Graphical Dasa" ), _( "Tajaka/Solar" ) };

	if ( ! node ) return 0;
	if ( node && ! wxStrcmp ( node->getName(), wxT( "WidgetNode" )))
	{
		WidgetNode* wn = (WidgetNode*)node;
		tp = wn->toolpanel;
		subtype = wn->subtype;
		vedic = wn->vedic;

		switch( wn->classid )
		{
			case 0:  id = VIEW_PLANETLIST; break;
			case 1:  id = VIEW_GRAPHIC; break;
			case 3:  id = VIEW_DASA; break;
			case 4:  id = VIEW_TEXT; break;
			case 5:  id = VIEW_SOLAR; break;
			case 6:  id = VIEW_SBC; break;
			case 7:  id = VIEW_GRAPHICALDASA; break;
			case 8:  id = VIEW_MIDPOINT; break;
			case 9:  id = VIEW_TRANSIT; break;
			default: assert( 0 ); break;
		}
		MViewInfo info( parent, this, id, getDoc(), true );
		info.showtoolbar = tp;
		info.type = subtype;
		info.varga = subtype;
		info.vedic = vedic;
		view = f.createView( info );
		wn->widget = view;
		view->doLayout();
		viewlist.push_back( view );
		return view;
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "SplitterNode" )))
	{
		SplitterNode *sn = (SplitterNode*)node;
		MySplitter *splitter = new MySplitter( parent, -1, wxDefaultPosition, size, wxSP_LIVE_UPDATE|wxSP_NOBORDER );
		sn->widget = splitter;
		splitter->SetSashGravity( 0.5 );
		if ( sn->window1 ) { window1 = insertNode( splitter, sn->window1 ); }
		if ( sn->window2 ) { window2 =insertNode( splitter, sn->window2 ); }

		if ( sn->dir ) { splitter->SplitVertically( window1, window2 ); }
		else { splitter->SplitHorizontally( window1, window2 ); }
		return splitter;
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "NotebookNode" )))
	{
		NotebookNode *nn = (NotebookNode*)node;
		wxNotebook *notebook = new wxNotebook( parent, -1 );
		notebook->SetImageList( imagelist );
		nn->widget = notebook;
		for( unsigned int i = 0; i < nn->pages.size(); i++ )
		{
			PageNode *page = nn->pages[i];
			ViewNode *node = page->window;
			wxPanel *panel = new wxPanel( notebook, -1 );
			page->widget = panel;

			wxWindow *window = insertNode( panel, node );
			// Sizer stuff
			wxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
			sizer->Add( window, 1, wxEXPAND, 0);
			panel->SetAutoLayout(true);
			panel->SetSizer(sizer);
			sizer->Fit( panel);

			int image = 0;
			int count = 0;
			ViewNode *swindow = page->window;
			while( wxString( swindow->getName()) != wxT( "WidgetNode" ) && count++ < 20 )
			{
				if ( wxString( swindow->getName()) == wxT( "SplitterNode" )) swindow = ((SplitterNode*)swindow)->window1;
			}
			// TODO
			image = ((WidgetNode*)swindow)->classid;
			pagename = session->getLocale()->GetString( page->name.c_str() );
			notebook->AddPage( panel, pagename, false, image );
		}
		notebook->SetAutoLayout( true );
		return notebook;
	}
	return 0;
}

/*****************************************************
**
**   MultipleChildWindow   ---   initViews 
**
******************************************************/
void MultipleChildWindow::initViews( ViewContainer *cont )
{
	container = cont;
	assert( container->window );
	container->window->widget = insertNode( this, container->window );
}

/**************************************************************
***
**   MultipleChildWindow   ---   getWindowLabel
***
***************************************************************/
const wxChar *MultipleChildWindow::getWindowLabel( const bool shortname )
{
	if ( doc ) return doc->getHName();
	else return wxT( "dummy" );
}

/*****************************************************
**
**   MultipleChildWindow   ---   postCreateNode
**
******************************************************/
void MultipleChildWindow::postCreateNode( ViewNode *node )
{
	assert( node->widget );
	if ( node && ! wxStrcmp ( node->getName(), wxT( "SplitterNode" )))
	{
		SplitterNode *snode = (SplitterNode*)node;
		MySplitter *splitter = (MySplitter*)node->widget;
		wxSize size = splitter->GetSize();
		int sash = 100;
		if ( snode->sashpos_rel > 0 )
			{ sash = ( snode->dir ? size.GetX() : size.GetY() ) * snode->sashpos_rel / 100; }
		else if ( snode->sashpos > 0 )
			{ sash = snode->sashpos; }
		else
			{ sash = ( snode->dir ? size.GetX() : size.GetY() ) / 2; } // emulate 50% 
		splitter->SetSashPosition( sash );
		if ( snode->window1 ) postCreateNode( snode->window1 );
		if ( snode->window2 ) postCreateNode( snode->window2 );
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "NotebookNode" )))
	{
		NotebookNode *nnode = (NotebookNode*)node;
		for( unsigned int i = 0; i < nnode->pages.size(); i++ )
		{
			PageNode *pnode = nnode->pages[i];
			if ( pnode ) postCreateNode( pnode );
		}
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "ViewContainer" )))
	{
		ViewContainer *vnode = (ViewContainer*)node;
		if ( vnode ) postCreateNode( vnode->window );
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "PageNode" )))
	{
		PageNode *pnode = (PageNode*)node;
		if ( pnode ) postCreateNode( pnode->window );
	}
	else if ( node && ! wxStrcmp ( node->getName(), wxT( "WidgetNode" )))
	{
		// do nothing
		//WidgetNode *wnode = (WidgetNode*)node;
	}
	else assert( false );
}

/*****************************************************
**
**   MultipleChildWindow   ---   postCreate
**
******************************************************/
void MultipleChildWindow::postCreate()
{
	postCreateNode( container->window );
	for( unsigned int i = 0; i < viewlist.size(); i++ ) viewlist[i]->postCreate();
}

/*****************************************************
**
**   MultipleChildWindow   ---   OnSize 
**
******************************************************/
void MultipleChildWindow::OnSize( wxSizeEvent &event )
{
	//printf( "MultipleChildWindow ->>OnSize %d -- %d\n", event.GetSize().x, event.GetSize().y  );
	config->xMultipleView = event.GetSize().x;
	config->yMultipleView = event.GetSize().y;
	event.Skip();
}

/**************************************************************
***
**   ChildWindowFactory   ---   createPartnerWindow
***
***************************************************************/
ChildWindow *ChildWindowFactory::createPartnerWindow( wxFrame *parent, DocumentManager *manager )
{
	int viewid = VIEW_PARTNER;
	wxSize size = getSizeForViewId( viewid );
	SimpleChildWindow *child = new SimpleChildWindow( parent, 0, size , false );
	MViewInfo info( child, child, viewid );
	MView *mview = ViewFactory().createPartnerView( info, manager );
	mview->doLayout();
	child->insertView( mview );
	child->SetIcon( IconProvider::getIcon( BITMAP_PARTNER ));
	return child;
}

/**************************************************************
***
**   ChildWindowFactory   ---   createMainWindow
***
***************************************************************/
ChildWindow *ChildWindowFactory::createMainWindow( wxFrame *parent, Document *doc )
{
	ChildWindow *child;
	int viewid;

	if ( config->useMultipleViews )
	{
		viewid = VIEW_MULTIPLE;
		wxSize size = getSizeForViewId( viewid );
		child = new MultipleChildWindow( parent, doc, size, true );

		ViewContainer *container = new ViewContainer;
		ViewBuilder().getViewDefinition( container, config->defaultView );
		((MultipleChildWindow*)child)->initViews( container );
	}
	else
	{
		viewid = VIEW_PLANETLIST;
		wxSize size = getSizeForViewId( viewid );
		child = new SimpleChildWindow( parent, doc, size, true );
		MViewInfo info( child, child, viewid, doc );
		MView *mview = ViewFactory().createView( info );
		assert( mview );
		((SimpleChildWindow*)child)->insertView( mview );
		mview->doLayout();
	}
	doc->setMainWindow( child );
	child->SetIcon( IconProvider::getWindowIcon( VIEW_PLANETLIST ));
	return child;
}

/**************************************************************
***
**   ChildWindowFactory   ---   createChild
***
***************************************************************/
ChildWindow *ChildWindowFactory::createChild( wxFrame *parent, Document *doc, const int id, const bool &vedic )
{
	int subtype = -1, vd = config->preferVedic;

	int viewid = mapCommand2ViewId( id, vd, subtype ); // dirty: vd will be mapped for graphic widget
	wxSize size = getSizeForViewId( viewid, vd );
	SimpleChildWindow *child = new SimpleChildWindow( parent, doc, size, false );

	MViewInfo info( child, child, viewid, doc );
	info.vedic = vd;
	info.varga = subtype;
	info.type = subtype;
	MView *mview = ViewFactory().createView( info );
	child->insertView( mview );
	mview->doLayout();

	child->SetIcon( IconProvider::getWindowIcon( viewid ));
	return child;
}



