/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           DasaGraphicView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "DasaGraphicView.h"
#endif

#include "DasaGraphicView.h"

#include <wx/choicdlg.h>
#include <wx/choice.h>
#include <wx/dc.h>
#include <wx/dcmemory.h>
#include <wx/event.h>
#include <wx/gdicmn.h>
#include <wx/image.h>
#include <wx/settings.h>
#include <wx/sizer.h>
#include <wx/statbox.h>
#include <wx/stopwatch.h>
#include <wx/textdlg.h>

#include "func.h"
#include "Conf.h"
#include "Lang.h"
#include "Dasa.h"
#include "Session.h"

#include "Document.h"
#include "FontProvider.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "ToolPanel.h"

extern Config *config;

IMPLEMENT_CLASS( DasaGraphicView, MView )
BEGIN_EVENT_TABLE( DasaGraphicView, MView )
	EVT_MENU( TBB_DASA, DasaGraphicView::OnDasas )
	EVT_MENU( TBB_GOTO, DasaGraphicView::OnYear )
	EVT_MENU( TBB_NOW, DasaGraphicView::OnNow )
	EVT_CHOICE( TBS_ZOOM, DasaGraphicView::OnChoiceCommand )
	EVT_MOUSEWHEEL( DasaGraphicView::OnMouseWheelEvent )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DasaGraphicWidget, ExportableScrollWidget )
BEGIN_EVENT_TABLE( DasaGraphicWidget, ExportableScrollWidget )
END_EVENT_TABLE()

/*****************************************************
**
**   DasaGraphicView   ---   Constructor 
**
******************************************************/
DasaGraphicView::DasaGraphicView( MViewInfo info )
 : MView( info )
{
	supportEWToggle = false;
	preferVedic = true;

	initToolItems();
	swidget = new DasaGraphicWidget( this, doc, -1 );
	widget = swidget;
}

/*****************************************************
**
**   DasaGraphicView   ---   initToolItems
**
******************************************************/
void DasaGraphicView::initToolItems()
{
	if ( toolbar )
	{
		wxChoice *choice_zoom = (wxChoice*)toolbar->FindControl( TBS_ZOOM );
		if ( choice_zoom ) choice_zoom->SetSelection( 3 );
	}
}

/*****************************************************
**
**   DasaGraphicView   ---   OnNow 
**
******************************************************/
void DasaGraphicView::OnNow( wxCommandEvent &event )
{
	Formatter *formatter = Formatter::get();
	int month, day, year;

	formatter->getDateIntsFromJD( JDate().getJD(), day, month, year );
	swidget->setYear( year );
}

/*****************************************************
**
**   DasaGraphicView   ---   OnMouseWheelEvent
**
******************************************************/
void DasaGraphicView::OnMouseWheelEvent( wxMouseEvent &event )
{
	int x, y;
	const int offset = 30;
	swidget->GetViewStart( &x, &y );
	if ( event.GetWheelRotation() < 0 ) y += offset;
	else y -= offset;
	swidget->Scroll( x, y );
}

/*****************************************************
**
**   DasaGraphicView   ---   OnDataChanged 
**
******************************************************/
void DasaGraphicView::OnDataChanged()
{
	swidget->OnDataChanged();
	swidget->Refresh();
}

/*****************************************************
**
**   DasaGraphicView   ---   OnYear 
**
******************************************************/
void DasaGraphicView::OnYear( wxCommandEvent &event )
{
	Session *session = Session::get();
	wxString s, defvalue;
	JDate d;
	Formatter *formatter = Formatter::get();
	int month, day, year;
	long y;
	wxString errorMsg;

	int birthyear = swidget->getBirthYear();
	int totalyears = swidget->getTotalYears();
	formatter->getDateIntsFromJD( d.getJD(), day, month, year );

	s.sprintf( _("Please enter a year between %d and %d"), birthyear, birthyear + totalyears );
	defvalue.sprintf( wxT( "%d" ), year );

	wxTextEntryDialog dialog( this, s, session->getGuiAppName(), defvalue );
	if ( dialog.ShowModal() == wxID_OK )
	{
		wxString s = dialog.GetValue();
		s.ToLong( &y );
		if ( y < birthyear || y > birthyear + totalyears )
		{
			errorMsg.sprintf( _("The value \"%s\" is not valid."), (const wxChar*)s );
			doMessageBox( this, errorMsg, wxOK | wxICON_ERROR | wxCENTRE );
			return;
		}
		swidget->setYear( y );
	}
}

/*****************************************************
**
**   DasaGraphicView   ---   OnDasas 
**
******************************************************/
void DasaGraphicView::OnDasas( wxCommandEvent &event )
{
	Session *session = Session::get();
	int n = session->getNoOfDasaExperts();
	wxString l[n];

	for( int i = 0; i < n; i++ ) l[i] = session->getDasaExpert( i )->getName();
	wxMultiChoiceDialog dialog( this,
		_("Choose Dasas (Multiple selections with mouse click and Ctrl/Shift button)"), session->getGuiAppName(), n, l );

	dialog.SetSelections( swidget->dasas );
	if ( dialog.ShowModal() == wxID_OK )
	{
		swidget->dasas = dialog.GetSelections();
		swidget->OnDataChanged();
		swidget->setZoom();
		swidget->Refresh();
	}
}

/*****************************************************
**
**   DasaGraphicView   ---   getIStyle 
**
******************************************************/
int DasaGraphicView::getIStyle()
{
	return swidget->getIStyle();
}

/*****************************************************
**
**   DasaGraphicView   ---   OnToolbarCommand 
**
******************************************************/
void DasaGraphicView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_zoom = (wxChoice*)toolbar->FindControl( TBS_ZOOM );
		assert( choice_zoom );
		wxString combo = choice_zoom->GetStringSelection();
		if ( combo.Len() > 0 )
		{
			if ( combo.Last() == wxT( '%' )) combo = combo.Left( combo.Len() - 1 );
			long z2 = 100;
			combo.ToLong( &z2 );
			swidget->setZoom( (int)z2 );
		}
		swidget->Refresh();
	}
}

/*****************************************************
**
**   DasaGraphicView   ---   SetContextMenuInfo 
**
******************************************************/
void DasaGraphicView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
	MView::SetContextMenuInfo( menuinfo );
}

/**************************************************************
***
**   DasaGraphicView   ---   getWindowLabel
***
***************************************************************/
const wxChar *DasaGraphicView::getWindowLabel( const bool shortname )
{
	return shortname ? _( "Dasa" ) : _( "Dasa (Graphical)" );
}

/*****************************************************
**
**   DasaGraphicView   ---   dispatchCommand 
**
******************************************************/
bool DasaGraphicView::dispatchCommand( const int &command )
{
  switch( command )
	{
		case CHILD_EXPORT:
			doExport();
		break;
		default:
			return MView::dispatchCommand( command );
		break;
	}
	swidget->Refresh();

 return true;
}

/*****************************************************
**
**   DasaGraphicView   ---   doExport 
**
******************************************************/
void DasaGraphicView::doExport()
{
	// TODO
	//double ratio = ( preferVedic && ( getIStyle() & IGRAPHIC_STYLE_SOUTH_INDIAN ) ? 1.4 : 1.0 );
	//swidget->doExport( ratio );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   Constructor 
**
******************************************************/
DasaGraphicWidget::DasaGraphicWidget( wxWindow *parent, Horoscope *doc, wxWindowID id, const wxPoint& pos,
	const wxSize& size, long style ) 
 : ExportableScrollWidget( parent )
{
	this->doc = doc;
	istyle = config->iGraphicStyle;
	totalyears = 100;

	for( int i = 0; i < Session::get()->getNoOfDasaExperts(); i++ )
		{	if ( ( config->iDefaultDasaList >> i ) % 2 ) dasas.Add( i ); }

	OnDataChanged();
	setZoom( 100 );

	Formatter *formatter = Formatter::get();
	int month, day, year;

	formatter->getDateIntsFromJD( JDate().getJD(), day, month, year );
	//setYear( year );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   Destructor 
**
******************************************************/
DasaGraphicWidget::~DasaGraphicWidget()
{
	config->iDefaultDasaList = 0;
	for( unsigned int i = 0; i < dasas.size(); i++ ) config->iDefaultDasaList += 1 << dasas[i];
	clearDasaItems();
}

/*****************************************************
**
**   DasaGraphicWidget   ---   clearDasaItems 
**
******************************************************/
void DasaGraphicWidget::clearDasaItems()
{
	unsigned int i, j, k;
	GraphicalDasaItem *item, *item2;

	for( i = 0; i < dasaitems.size(); i++ )
	{
		item = dasaitems[i];

		for( j = 0; j < item->depitems.size(); j++ )  
		{
			item2 = item->depitems[j];
			// TODO nice memory leaks, delete will crash under windows
			//if ( item2->dasa ) delete item2->dasa;
			for( k = 0; k < item2->depitems.size(); k++ )  
			{
				//if ( item2->depitems[k]->dasa ) delete item2->depitems[k]->dasa;
				delete item2->depitems[k];
			}
			delete item2;
			item2->depitems.clear();
		}
		delete item;
	}
	dasaitems.clear();
}

/*****************************************************
**
**   DasaGraphicWidget   ---   OnDataChanged 
**
******************************************************/
void DasaGraphicWidget::OnDataChanged()
{
	unsigned int i, j, k;
	DasaExpert *expert;
	GraphicalDasaItem *item, *depitem;
	vector<Dasa*> w;

	clearDasaItems();
	for( i = 0; i < dasas.Count(); i++ )
	{
		expert = Session::get()->getDasaExpert( dasas.Item( i ));
		item = new GraphicalDasaItem( expert, 0 );
		vector<Dasa*> v = expert->getFirstLevel( doc, 0 );
		if ( dasas.Item( i ) == 2 && v[0]->getExtraText() ) // Kalachakra
		{
			item->name.sprintf( wxT( "%s (%s)" ), expert->getName(), v[0]->getExtraText().c_str() );
		}
		else { item->name = expert->getName(); }

		for( j = 0; j < v.size(); j++ )  
		{
			depitem = new GraphicalDasaItem( expert, v[j] );
			w = expert->getNextLevel( v[j] );
			for( k = 0; k < w.size(); k++ )  
			{
				depitem->depitems.push_back( new GraphicalDasaItem( expert, w[k] ) );
			}
			w.clear();
			item->depitems.push_back( depitem );
		}
		dasaitems.push_back( item );
	}
}

/*****************************************************
**
**   DasaGraphicWidget   ---   paintRuler 
**
******************************************************/
void DasaGraphicWidget::paintRuler()
{
	int y = ycursor + ybar * dasas.Count() * 3;
	int ysizey = 20;
	int ylength;
	int x;
	int xtext = 50;
	int ytext = 10;
	int i, j, xm;
	double steplength = ( xright - xleft ) / totalyears;
	double xmonth = steplength / 12;
	wxString s;
	JDate d;

	dc->SetBrush( wxNullBrush );
	dc->SetPen( *wxBLACK_PEN );
	dc->DrawLine( (int)xleft, y, (int)xright, y );

	for( i = 0; i <= totalyears; i++ )
	{
		x = (int)(xleft + i * steplength);
		dc->SetPen( *wxGREY_PEN );
		dc->DrawLine( x, (int)(ytop + 2 * hborder), x, y );
		dc->SetPen( *wxBLACK_PEN );
		dc->DrawLine( x, y - ysizey, x, y + ysizey );
		s.sprintf( wxT( "%d" ), birthyear + i );
		if ( i > 0 && i < totalyears )
			drawTextFormatted( wxRect( x-xtext, y + ysizey + ytext, 2*xtext, 2*ytext ), s, Align::Center );
		if ( i < totalyears )
		{
			for( j = 1; j < 12; j++ )
			{
				xm = (int)(x + j* xmonth);
				ylength = ysizey/( j % 3 ? 3 : 2 );
				if ( xmonth > 3 ) dc->DrawLine( xm, y - ylength, xm, y + ylength );
				if (( xmonth > 3 && ! ( j%3 )) || xmonth > 15 )
				{
					s.sprintf( wxT( "%d" ), j+1 );
					drawTextFormatted( wxRect( xm-xtext, y + ytext, 2*xtext, 2*ytext ), s, Align::Center );
				}
			}
		}
	}
	dc->SetPen( wxPen( *wxBLUE, 2, wxSOLID ));

	double actjd = d.getJD();
	d.setDate( 1, 1 , birthyear, 0 );
	double yy = ( actjd - d.getJD()  ) / 365.25; // age in years
	x = (int)(xleft + (xright - xleft ) * yy / totalyears );
	dc->DrawLine( x, (int)(ytop + hborder), x, y + 2 * ysizey );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   setYear 
**
******************************************************/
void DasaGraphicWidget::setYear( const int &year )
{
	Formatter *formatter = Formatter::get();
	int day, month;
	assert( doc );
	if ( birthyear == 0 )
	{
		formatter->getDateIntsFromJD( doc->getJD(), day, month, birthyear );
	}
	double portion = ( year - birthyear ) / (double)totalyears;
	double y = portion * ( xright - xleft );
	Scroll( (int)(y/scrolly), 0 );
	printf( "Portion %f scrolly %d total %d year %d birth %d\n", portion, scrolly, totalyears, year, birthyear );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   setZoom 
**
******************************************************/
void DasaGraphicWidget::setZoom( const int z )
{
	if ( z ) zoom = z;
	int width = 100 * zoom;
	int height = 100 + 100 * dasas.Count();
	SetVirtualSize( width, height );

	xleft = 2 * hborder;
	xright = width - 4 * hborder;
	ytop = 2 * vborder;
	ybottom = height - 4 * vborder;
}

/*****************************************************
**
**   DasaGraphicWidget   ---   getX4Jd 
**
******************************************************/
int DasaGraphicWidget::getX4Jd( const double &jd )
{
	double a = (xright - xleft)/(totalyears * 365.25); // pro tag
	return (int)(a * ( jd - startjd ) + 2 * hborder );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   getFgColour 
**
******************************************************/
wxColour DasaGraphicWidget::getFgColour( const int &i, const bool &planetdasa )
{
	static const unsigned char pdasafgcolor[9][3] = {
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 255, 255, 255 },  // Mars
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 255, 255, 255 },  // Saturn
		{ 255, 255, 255 },  // Rahu
		{ 0, 0, 0 }
	};
	static const unsigned char rdasafgcolor[12][3] = {
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 0, 0, 0 },
		{ 0, 0, 0 }, // Virgo, piebald -> black
		{ 255, 255, 255 },  // Libra
		{ 0, 0, 0 },
		{ 255, 255, 255 },  // Sag
		{ 0, 0, 0 },
		{ 255, 255, 255 },  // Aq
		{ 0, 0, 0 }
	};
	if ( planetdasa )
		return wxColour( pdasafgcolor[i][0], pdasafgcolor[i][1], pdasafgcolor[i][2] );
	else
		return wxColour( rdasafgcolor[i][0], rdasafgcolor[i][1], rdasafgcolor[i][2] );
}


/*****************************************************
**
**   DasaGraphicWidget   ---   getBgColour 
**
******************************************************/
wxColour DasaGraphicWidget::getBgColour( const int &i, const bool &planetdasa )
{
	// Jataka Parijata II, 21
	static const unsigned char pdasabgcolor[9][3] = {
		{ 255, 94, 73 }, // Sun red
		{ 255, 255, 255 }, // Moon white
		{ 204, 0, 0 },     // Mars red
		{ 102, 204, 102 },  // Mercury green
		{ 255, 255, 102 },  // Jupiter yellow
		{ 255, 102, 255 },   // Venus variegated
		{ 0, 0, 0 },  // Saturn black
		{ 153, 102, 0 }, // Rahu
		{ 132, 132, 132 }  // Ketu
	};

	// Jataka Parijata I, 23
	static const unsigned char rdasabgcolor[12][3] = {
		{ 255, 0, 0 }, // Aries red
		{ 255, 255, 255 }, // Taurus white
		{ 0, 204, 51 },  // Gemini green
		{ 255, 160, 255 },   // Cancer Pink
		{ 226, 255, 211 },  // Leo pale white
		{ 255, 255, 255 }, // Virgo piebald -> rainbow
		{ 0, 0, 0 },  // Libra black
		{ 229, 169, 104 },  // Scorpio golden
		{ 167, 123, 76 },  // Sagittarius yellowish or brown
		{ 255, 255, 255 }, // Capricorn variegated -> rainbow
		{ 104, 77, 47 },   // Aquarius deep-brown
		{ 255, 255, 255 } // Pisces white
	};
	if ( planetdasa )
		return wxColour( pdasabgcolor[i][0], pdasabgcolor[i][1], pdasabgcolor[i][2] );
	else 
		return wxColour( rdasabgcolor[i][0], rdasabgcolor[i][1], rdasabgcolor[i][2] );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   showDasaBar 
**
******************************************************/
void DasaGraphicWidget::showDasaBar( Dasa *d, DasaExpert *expert, double y )
{
	int x1 = getX4Jd( Max( d->getStartJD(), doc->getJD()));
	int x2 = getX4Jd( Min( d->getEndJD(), endjd )) - x1;
	int xstart, dummy, xwidth;
	GetViewStart( &xstart, &dummy );
	GetClientSize( &xwidth, &dummy );

	// Nothing to do because bar is outside visible area
	if ( x1 > xstart + xwidth || xstart > x2 + x1  ) return;

	wxString s;
	int format = TLARGE;
	wxColour bgcolor = getBgColour( d->getDasaLord(), expert->getType() == DASA_TYPE_PLANET );
	wxColour fgcolor = getFgColour( d->getDasaLord(), expert->getType() == DASA_TYPE_PLANET );
	dc->SetBrush( wxBrush( bgcolor, wxSOLID ));

	wxBrush b( bgcolor, wxSOLID );

	// Variegated colors are realized via rainbow or piebald Brush
	if (( expert->getType() == DASA_TYPE_PLANET && d->getDasaLord() == IVENUS )
		|| ( expert->getType() == DASA_TYPE_SIGN && d->getDasaLord() == CAPRICORNUS ))
	{
		wxBrush b( *wxWHITE_BRUSH );
		b.SetStipple( IconProvider::getBitmap( BITMAP_RAINBOW ));
		b.SetStyle( wxSTIPPLE );
		dc->SetBrush( b );
	}
	else if (  expert->getType() == DASA_TYPE_SIGN && d->getDasaLord() == VIRGO )
	{
		wxBrush b( *wxWHITE_BRUSH );
		b.SetStipple( IconProvider::getBitmap( BITMAP_PIEBALD ));
		b.SetStyle( wxSTIPPLE );
		dc->SetBrush( b );
	}
	dc->SetPen( *wxBLACK_PEN );
	dc->SetTextBackground( bgcolor );
	dc->SetTextForeground( fgcolor );

	dc->DrawRectangle( x1, (int)y, x2, (int)ybar );
	if ( x2 < 80 ) format = TMEDIUM;
	if ( x2 < 40 ) format = TSHORT;
	if ( expert->getType() == DASA_TYPE_PLANET )
	{
		s = Lang::get()->getObjectName(d->getDasaLord(), format);
	}
	else
	{
		s = Lang::get()->getSignName(d->getDasaLord(), format);
	}
	drawTextFormatted( wxRect( x1, (int)y, x2, (int)ybar ), s, Align::Center );
}

/*****************************************************
**
**   DasaGraphicWidget   ---   paintDasa 
**
******************************************************/
void DasaGraphicWidget::paintDasa( const int &whichdasa )
{
	unsigned int i, j;
	GraphicalDasaItem *item2, *item3, *dasaitem = dasaitems[whichdasa];
	wxString s;

	int x1 = getX4Jd( doc->getJD());
	int x2;
	for( i = 0; i < dasaitem->depitems.size(); i++ )  
	{
		dc->SetTextBackground( *wxWHITE );
		dc->SetTextForeground( *wxBLACK );
		s =  dasaitem->name;
		x2 = x1;
		while( x2 < xright - 200 )
		{
			drawTextFormatted( wxRect( x2, (int)(ycursor-ybar), 100, (int)ybar ), s, Align::Bottom + Align::Left );
			x2 += 800;
		}

		item2 = dasaitem->depitems[i];
		if ( item2->dasa->getStartJD() < endjd )
		{
			showDasaBar( item2->dasa, item2->expert, ycursor );
		}
		for( j = 0; j < item2->depitems.size(); j++ )  
		{
			item3 = item2->depitems[j];
			if (( doc->getJD() < item3->dasa->getEndJD() && endjd > item3->dasa->getStartJD() ))
				showDasaBar( item3->dasa, item3->expert, ycursor + ybar + 2 );
		}
	}
	ycursor += 3* ybar;
}

/*****************************************************
**
**   DasaGraphicWidget   ---   doPaint 
**
******************************************************/
void DasaGraphicWidget::doPaint( wxDC *p )
{
	Formatter *formatter = Formatter::get();
	JDate d;
	int month, day;
	wxCoord w, h;
	//wxStopWatch watch;
	//static int i = 1;

	//printf( "Start Paint Dasa %d x %d y %d \n", i++, GetSize().x, GetSize().y );

	SetFont( *FontProvider().getGraphicFont());
	assert( p );
	GetVirtualSize( &w, &h );

	formatter->getDateIntsFromJD( doc->getJD(), day, month, birthyear );
	d.setDate( 1, 1, birthyear, 0 );
	startjd = d.getJD();
	d.setDate( 1, 1, birthyear + totalyears, 0 );
	endjd = d.getJD();

	ybar = 30;
	ycursor = (int)(ybar * 1.5);
	
	dc->SetBrush( *wxGREY_BRUSH );
	dc->DrawRectangle( 0, 0, w, h );
	dc->SetFont( *FontProvider().getGraphicFont());

	// White page background
	dc->SetBrush( *wxWHITE_BRUSH );
	dc->SetPen( *wxWHITE_PEN );
	dc->DrawRectangle( hborder, vborder, (int)(xright + hborder), (int)(ybottom + vborder) );

	paintRuler();
	for( unsigned int i = 0; i < dasaitems.size(); i++ ) paintDasa( i );
	//printf( "Time for Dasa paint: %ld millisec\n", watch.Time() );
}

/**************************************************************
***
**   ViewFactory   ---   createDasaGraphicView
***
***************************************************************/
MView *ViewFactory::createGraphicalDasaView( MViewInfo info )
{
	return new DasaGraphicView( info );
}


