/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           DasaView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "DasaView.h"
#endif

#include "DasaView.h"

#include <wx/bmpbuttn.h>
#include <wx/imaglist.h>
#include <wx/sizer.h>
#include <wx/stattext.h>
#include <wx/textctrl.h>

#include "Base.h"
#include "ApplicationWindow.h"
#include "Document.h"
#include "FontProvider.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "ToolPanel.h"
#include "dialogs/DasaPanel.h"

#include "constants.h"
#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "Dasa.h"
#include "Lang.h"
#include "Session.h"
#include "Varga.h"

extern Config *config;

enum { DASAWINDOW_CTRL = wxID_HIGHEST + 1 };

IMPLEMENT_CLASS( DasaView, MView )
BEGIN_EVENT_TABLE( DasaView, MView )
	MYEVT_TREE_ITEM_EXPANDING( DASAWINDOW_CTRL, DasaView::expand )
	MYEVT_TREE_ITEM_COLLAPSING( DASAWINDOW_CTRL, DasaView::collapse )
	MYEVT_TREE_SEL_CHANGING( DASAWINDOW_CTRL, DasaView::OnSelChanging )
	EVT_BUTTON( DASA_CREATE_ENTRY_CHART, DasaView::OnEntryChart )
	EVT_MOUSEWHEEL( DasaView::OnMouseWheelEvent )
END_EVENT_TABLE()


enum { PIC_DASA = 0, PIC_FOLDER, PIC_DASAROOT, PIC_DASATREE1, PIC_DASATREE2, PIC_DASATREE3,
	PIC_DASATREE4, PIC_DASATREE5, PIC_FOLDERE, PIC_DASAROOTE, PIC_DASATREE1E, PIC_DASATREE2E,
	PIC_DASATREE3E, PIC_DASATREE4E, PIC_DASATREE5E };

/*****************************************************
**
**   DasaView   ---   Constructor 
**
******************************************************/
DasaView::DasaView( MViewInfo info )
 : MView( info )
{
	Session *session = Session::get();
	VargaExpert vexpert;
	DasaExpert *dexpert;
	wxString dasaname;
	int i;
	DasaMainListViewItem *item;
	wxTreeItemId id, root;

	currentjd = 0;
	treectrl = new MyTreeCtrl( this, DASAWINDOW_CTRL, wxDefaultPosition, wxDefaultSize,
		wxTR_HAS_BUTTONS|wxSUNKEN_BORDER );
	treectrl->SetFont( *FontProvider().getDefaultFont() );
	treectrl->SetIndent( 30 );
	treectrl->AssignImageList( IconProvider::createImageList( "bbbbbbbbbbbbbbb", BITMAP_DASA, BITMAP_DASA_FOLDER,
		BITMAP_MAHADASA, BITMAP_DASATREE1, BITMAP_DASATREE2, BITMAP_DASATREE3, BITMAP_DASATREE4, BITMAP_DASATREE5,
		BITMAP_DASA_FOLDERE, BITMAP_MAHADASAE, BITMAP_DASATREE1E, BITMAP_DASATREE2E, BITMAP_DASATREE3E,
		BITMAP_DASATREE4E, BITMAP_DASATREE5E ));

	widget = treectrl;
	showToolPanel( true );

	rootid = treectrl->AddRoot( _( "Dasas" ), PIC_DASA );
	treectrl->SetItemImage( rootid, PIC_DASA, wxTreeItemIcon_Expanded );
	baseid = treectrl->AppendItem( rootid, _( "Standard" ), PIC_FOLDER );
	treectrl->SetItemImage( baseid, PIC_FOLDERE, wxTreeItemIcon_Expanded );
	conditionalid = treectrl->AppendItem( rootid, _( "Conditional" ), PIC_FOLDER );
	treectrl->SetItemImage( conditionalid, PIC_FOLDERE, wxTreeItemIcon_Expanded );
	specialid = treectrl->AppendItem( rootid, _( "Special" ), PIC_FOLDER );
	treectrl->SetItemImage( specialid, PIC_FOLDERE, wxTreeItemIcon_Expanded );
  for( i = 0; i < session->getNoOfDasaExperts(); i++ )
  {
		if ( i == DASA_VIMSOTTARI || i == DASA_YOGINI || i == DASA_KALACHAKRA ) root = baseid;
		else if ( i >= DASA_ASHTOTTARI && i <= DASA_SHATTRIMSATSAMA ) root = conditionalid;
		else root = specialid;
    dexpert = session->getDasaExpert( i );
		dasaname.sprintf( wxT( "%s" ), dexpert->getName() );
		item = new DasaMainListViewItem( dexpert, doc, 0 );
		id = treectrl->AppendItem( root, dasaname,  PIC_DASAROOT, PIC_DASAROOT, item );
		treectrl->SetItemImage( id, PIC_DASAROOTE, wxTreeItemIcon_Expanded );
		treectrl->SetItemHasChildren( id );
		treectrl->SetItemBold( id, true );
  }
	treectrl->Expand( rootid );
	treectrl->Expand( baseid );
	treectrl->Show( true );
}

/*****************************************************
**
**   DasaView   ---   showToolPanel 
**
******************************************************/
void DasaView::showToolPanel( const bool b )
{
	DasaPanel *panel;
	wxFont *dfont;
	FontProvider f;
	if ( ! b )
	{
		if ( detailpanel ) delete detailpanel;
		detailpanel = 0;
	}
	else
	{
		panel = new DasaPanel( this, TOOLPANEL );
		detailpanel = panel;
		panel->label_t->SetFont( *f.getHeaderFont());
		dfont = FontProvider().getDefaultFont();
		panel->label_type->SetFont( *dfont );
		panel->label_startdate->SetFont( *dfont );
		panel->label_enddate->SetFont( *dfont );
		panel->label_duration->SetFont( *dfont );
	}
	updateDetailPanel( 0 );
}

/*****************************************************
**
**   DasaView   ---   collapse 
**
******************************************************/
void DasaView::collapse( MyTreeEvent& event )
{
	clearDetailPanel();
}

/*****************************************************
**
**   DasaView   ---   OnMouseWheelEvent
**
******************************************************/
void DasaView::OnMouseWheelEvent( wxMouseEvent &event )
{
	int x, y;
	const int offset = 30;
	treectrl->GetViewStart( &x, &y );
	if ( event.GetWheelRotation() < 0 ) y += offset;
	else y -= offset;
	treectrl->Scroll( x, y );
}

/*****************************************************
**
**   DasaView   ---   expand 
**
******************************************************/
void DasaView::expand( MyTreeEvent& event )
{
  unsigned int i;
	wxString s;
  vector<Dasa*> divector;
	wxTreeItemId id, id2;
	Dasa *dasa, *dasa2;
  DasaDependendViewItem *ditem;
	int level = 0;
	int pic = PIC_DASATREE1;
	int epic = PIC_DASATREE1E;

	id = event.GetItem();
	MyTreeItemData *data = treectrl->GetItemData( id );

	// Root item
	if ( ! data ) return;

	// Children already there
  if ( treectrl->GetChildrenCount( id ) > 0 ) return;

	DasaListViewItem *dlistitem = (DasaListViewItem*)data;

  DasaExpert *e = dlistitem->getExpert();
  if ( dlistitem->isRootItem())
	{
		divector = e->getFirstLevel( doc, ((DasaMainListViewItem*)dlistitem)->getVarga() );
	}
  else
	{
		dasa = ((DasaDependendViewItem*)dlistitem)->getDasa();
		divector = e->getNextLevel( dasa );
		dasa2 = dasa->getParent();
		while( dasa2 )
		{
			level++;
			dasa2 = dasa2->getParent();
		}
		switch( level )
		{
			case 0: pic = PIC_DASATREE2; epic = PIC_DASATREE2E; break;
			case 1: pic = PIC_DASATREE3; epic = PIC_DASATREE3E; break;
			case 2: pic = PIC_DASATREE4; epic = PIC_DASATREE4E; break;
			default: pic = PIC_DASATREE5; epic = PIC_DASATREE5E; break;
		}
	}
       
  for( i = 0; i < divector.size(); i++ )
  {  
    ditem = new DasaDependendViewItem( e, divector[i], doc );
		id2 = treectrl->AppendItem( id, ditem->writeContents(), pic, pic, ditem );
		treectrl->SetItemImage( id2, epic, wxTreeItemIcon_Expanded );
		treectrl->SetItemHasChildren( id2 );
		treectrl->SetItemHidden( id2, (divector[i]->getEndJD() < doc->getJD() ));
  }
}

/*****************************************************
**
**   DasaView   ---   OnSelChanging 
**
******************************************************/
void DasaView::OnSelChanging( MyTreeEvent& event )
{
	wxTreeItemId id = event.GetItem();
	updateDetailPanel( (DasaListViewItem*)treectrl->GetItemData( id ) );
}

/*****************************************************
**
**   DasaView   ---   clearDetailPanel 
**
******************************************************/
void DasaView::clearDetailPanel()
{
	if ( detailpanel )
	{
		DasaPanel *panel = (DasaPanel*)detailpanel;
		panel->label_t->SetLabel( wxT( "" ) );
		panel->label_type->SetLabel( wxT( "" ) );
		panel->label_startdate->SetLabel( wxT( "" ) );
		panel->label_enddate->SetLabel( wxT( "" ) );
		panel->label_duration->SetLabel( wxT( "" ) );
		panel->button_entry_chart->Enable( false );
	}
	currentjd = 0;
}

/*****************************************************
**
**   DasaView   ---   updateDetailPanel 
**
******************************************************/
void DasaView::updateDetailPanel( DasaListViewItem *item )
{
	if ( ! detailpanel ) return;
	Dasa *dasa, *dasa2;
	DasaDependendViewItem *depitem;
	Formatter *formatter = Formatter::get();
	wxChar tmp[256], buf[256], dasatype[256], dasalords[256];
	wxString s;
	double duration;
	int level = 0;
	const wxChar *k_dasaname[5] = { _( "Mahadasa" ), _( "Antardasa" ), _( "Pratyantardasa" ),
		_( "Sooksmantardasa" ), _( "Pranadasa" )};

	DasaPanel *panel = (DasaPanel*)detailpanel;
	if ( ! item )
	{
		clearDetailPanel();
		return;
	}
	DasaExpert *expert = item->getExpert();
	if ( item->isRootItem())
	{
		clearDetailPanel();
		wxSprintf( tmp, _( "%s Dasa" ), expert->getName() );
		panel->label_t->SetLabel( tmp );
	}
	else
	{
		wxStrcpy( dasalords, wxT( "" ) );
		depitem = (DasaDependendViewItem*)item;
		dasa = depitem->getDasa();
		dasa2 = dasa;
		while( dasa2 )
		{
			wxSprintf( tmp, wxT( "%s%s" ), ( wxStrlen( dasalords ) > 0 ?  wxT( " / " ) : wxT( "" )), dasalords );
			if ( expert->getType() == DASA_TYPE_SIGN )
				wxSprintf( dasalords, wxT( "%s%s" ), Lang::get()->getSignName( dasa2->getDasaLord(), TLARGE ).c_str(), tmp );
			else
				wxSprintf( dasalords, wxT( "%s%s" ), Lang::get()->getObjectName( dasa2->getDasaLord(), TLARGE ).c_str(), tmp );
			level++;
			dasa2 = dasa2->getParent();
		}
		if ( level <= 5 )
			{ wxStrcpy( dasatype, k_dasaname[level-1] ); }
		else
			{ wxSprintf( dasatype, _( "Antardasa Level %d" ), level ); }

		double t = doc->getLocation()->getTimeZone() + doc->getLocation()->getDST();
		t /= 24;
		currentjd = dasa->getStartJD();
		duration = dasa->getEndJD() - Max( doc->getJD(), dasa->getStartJD() );
		wxSprintf( tmp, _( "%s of %s Dasa" ), dasatype, expert->getName() );
		panel->label_t->SetLabel( tmp );
		wxSprintf( tmp, _( "Lord: %s" ), dasalords );
		panel->label_type->SetLabel( tmp );
		wxSprintf( tmp, _( "Start: %s" ), formatter->getFullDateStringFromJD( dasa->getStartJD()+t ).c_str() );
		panel->label_startdate->SetLabel( tmp );
		wxSprintf( tmp, _( "End: %s" ), formatter->getFullDateStringFromJD( dasa->getEndJD()+t ).c_str() );
		panel->label_enddate->SetLabel( tmp );
		getDasaDuration( s, duration );
		wxSprintf( buf, _( "Duration: %s" ), s.c_str() );
		panel->label_duration->SetLabel( buf );
		panel->button_entry_chart->Enable( true );
	}
}

/*****************************************************
**
**   DasaView   ---   OnDataChanged 
**
******************************************************/
void DasaView::OnDataChanged()
{
	updateDasa( rootid );
	updateDetailPanel( 0 );
}

/*****************************************************
**
**   DasaView   ---   updateDasa 
**
******************************************************/
void DasaView::updateDasa( wxTreeItemId masterid )
{
	wxTreeItemId id;
	MyTreeItemData *data;
	DasaListViewItem *item;
	DasaDependendViewItem *depitem;
	vector<Dasa*> vdasa;
	int i = 0;
	long cook;

  if ( treectrl->GetChildrenCount( masterid ) == 0 ) return;
	data = treectrl->GetItemData( masterid );
	if ( ! data )
	{
		id = treectrl->GetFirstChild( masterid, cook );
		while( id.IsOk() )
		{
			updateDasa( id );
			id = treectrl->GetNextSibling( id );
		}
		return;
	}
	item = (DasaListViewItem*)data;
	DasaExpert *expert = item->getExpert();

	if ( item->isRootItem() )
		vdasa = expert->getFirstLevel( item->getHoroscope(), ((DasaMainListViewItem*)item)->getVarga() );
	else
		vdasa = expert->getNextLevel( ((DasaDependendViewItem*)item)->getDasa() ); 

	id = treectrl->GetFirstChild( masterid, cook );
	while( id.IsOk() )
	{
		depitem = (DasaDependendViewItem*)treectrl->GetItemData( id );
		assert( depitem );
		delete depitem->getDasa();
		depitem->setDasa( vdasa[i] );
		treectrl->SetItemText( id, depitem->writeContents() );
		treectrl->SetItemHidden( id, vdasa[i]->getEndJD() < doc->getJD() );
		updateDasa( id );

		assert( i < 100 && i >= 0 );
		i++;
		id = treectrl->GetNextSibling( id );
	}
}

/*****************************************************
**
**   DasaView   ---   OnEntryChart
**
******************************************************/
void DasaView::OnEntryChart( wxCommandEvent &event )
{
	wxChar tmp[256];
	if ( currentjd == 0 )
	{
		doMessageBox( this, wxT( "Error: can't determine date for entry chart." ));
	}
	else
	{
		Document *d = new Document;
		d->setLocation( *doc->getLocation() );
		d->setDate( currentjd );
		wxSprintf( tmp, wxT( "%s: %s" ), doc->getHName(), _( "Entry Chart" ));
		d->setHName( tmp );

		wxCommandEvent event( CREATE_ENTRY_CHART, GetParent()->GetId() );
		event.SetEventObject( (wxObject*)d );
		wxPostEvent( GetParent(), event );
	}
}

/**************************************************************
***
**   DasaView   ---   getWindowLabel
***
***************************************************************/
const wxChar *DasaView::getWindowLabel( const bool shortname )
{
	return shortname ? _( "Dasa" ) : _( "Dasa (Tree)" );
	//return _( "Dasa" );
}

/*****************************************************
**
**   DasaListViewItem   ---   Constructor 
**
******************************************************/
DasaListViewItem::DasaListViewItem( DasaExpert *e, Horoscope *h )
 : expert( e )
{
	horoscope = h;
}

/*****************************************************
**
**   DasaListViewItem   ---   Destructor 
**
******************************************************/
DasaListViewItem::~DasaListViewItem()
{
}

/*****************************************************
**
**   DasaMainListViewItem   ---   Constructor 
**
******************************************************/
DasaMainListViewItem::DasaMainListViewItem( DasaExpert *expert, Horoscope *h, int v )
	: DasaListViewItem( expert, h )
{
	varga = v;
}

/*****************************************************
**
**   DasaDependendViewItem   ---   Constructor 
**
******************************************************/
DasaDependendViewItem::DasaDependendViewItem( DasaExpert *expert, Dasa *d, Horoscope *h )
	: DasaListViewItem( expert, h ), dasa( d )
{
}

/*****************************************************
**
**   DasaDependendViewItem   ---   writeContents 
**
******************************************************/
wxString DasaDependendViewItem::writeContents()
{
	wxString s, t;
	Formatter *formatter = Formatter::get();
	Location *loc = horoscope->getDataSet()->getLocation();
	double timediff = loc->getTimeZone() + loc->getDST();
	timediff /= 24;

	double jd = horoscope->getJD() + timediff;

	if ( expert->getType() == DASA_TYPE_SIGN )
		{ t = Lang::get()->getSignName( dasa->getDasaLord()); }
	else
		{ t = Lang::get()->getObjectName( dasa->getDasaLord(), TLARGE ); }
	if ( dasa->getExtraText() &&  ! wxStrcmp( dasa->getExtraText(), wxT( "" )) )
		s << formatter->getDateStringFromJD( Max( jd, dasa->getStartJD() )) << wxT( " " ) << t;
	else
		s.sprintf( wxT( "%s %s (%s)" ), formatter->getDateStringFromJD( Max( jd, dasa->getStartJD() )).c_str(),
			t.c_str(), dasa->getExtraText().c_str() );
	
	return s;
}

/**************************************************************
***
**   ViewFactory   ---   createDasaView
***
***************************************************************/
MView *ViewFactory::createDasaView( MViewInfo info )
{
	info.showtoolbar = false;
	return new DasaView( info );
}


