/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Document.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Document.h"
#endif

#include "Document.h"

#include <wx/app.h>
#include <wx/choice.h>
#include <wx/filedlg.h>
#include <wx/filename.h>
#include <wx/icon.h>

#include "Base.h"
#include "IconProvider.h"
#include "ChildWindow.h"
#include "dialogs/DataDialog.h"
#include "dialogs/PrintConfigurationDialog.h"
#include "Printer.h"

#include "constants.h"
#include "func.h"
#include "Conf.h"

extern Config *config;

enum { DOCUMENT_TIMER = wxID_HIGHEST + 1 };

IMPLEMENT_CLASS( Document, wxEvtHandler )
BEGIN_EVENT_TABLE( Document, wxEvtHandler )
  EVT_TIMER( DOCUMENT_TIMER, Document::OnTimer )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DocumentListener, wxObject )

DEFINE_EVENT_TYPE( DOC_SAVED )
DEFINE_EVENT_TYPE( DOC_UPDATED )
DEFINE_EVENT_TYPE( CREATE_ENTRY_CHART )

/*****************************************************
**
**   Document   ---   Constructor 
**
******************************************************/
Document::Document()
: Horoscope(), wxEvtHandler()
{
	dirty = false;
	mainwindow = 0;
	timer = new wxTimer( this, DOCUMENT_TIMER );
  StartTimer( false );
}

#ifdef _WX_WANT_NEW_SIZET_WXCHAR_INT
void *Document::operator new ( size_t size, const wxChar *fileName, int lineNum )
{
	return wxDebugAlloc(size, (wxChar*) fileName, lineNum, true);
}
#endif

#ifdef _WX_WANT_DELETE_VOID
void Document::operator delete ( void *buf )
{
	wxDebugFree(buf);
}
#endif

/*****************************************************
**
**   Document   ---   Destructor 
**
******************************************************/
Document::~Document()
{
	if ( timer->IsRunning() ) timer->Stop();
}

/*****************************************************
**
**   Document   ---   addChildWindow 
**
******************************************************/
void Document::addChildWindow( ChildWindow *child )
{
	if ( child != mainwindow ) childlist.Append( child );
}

/*****************************************************
**
**   Document   ---   releaseChildWindow 
**
******************************************************/
void Document::releaseChildWindow( ChildWindow *child )
{
	if ( child == mainwindow ) return;

	for( ChildWindowList::Node *node = childlist.GetFirst(); node; node = node->GetNext() )
	{
		if ( node->GetData() == child )
		{
			childlist.DeleteNode( node );
			return;
		}
	}
	assert( 0 );
}

/*****************************************************
**
**   Document   ---   setMainWindow 
**
******************************************************/
void Document::setMainWindow( ChildWindow *child )
{
	mainwindow = child;
}

/*****************************************************
**
**   Document   ---   queryClose 
**
******************************************************/
bool Document::queryClose()
{
	if ( ! dirty ) return true;

	wxString s;
	s.sprintf( wxT( "%s: %s ?" ), _( "Do you want to save the file" ), getHName());
	int i = doMessageBox( mainwindow, s, wxYES_NO | wxCANCEL | wxCENTRE );
	if ( i == wxID_CANCEL ) return false;
	else if ( i == wxID_YES )
	{
		return save();
	}
	else
	{
		dirty = false;
		return true;
	}
}

/*****************************************************
**
**   Document   ---   closeDependenChildWindows 
**
******************************************************/
void Document::closeDependenChildWindows()
{
	ChildWindowList::Node *node = childlist.GetFirst();

	while( node )
	{
		node->GetData()->Close();
		childlist.DeleteNode( node );
		node = childlist.GetFirst();
	}
}

/*****************************************************
**
**   Document   ---   updateAllChildWindows 
**
******************************************************/
void Document::updateAllChildWindows()
{
	mainwindow->OnDataChanged();

	for( ChildWindowList::Node *node = childlist.GetFirst(); node; node = node->GetNext() )
	{
		(node->GetData())->OnDataChanged();
	}
}

/*****************************************************
**
**   Document   ---   save 
**
******************************************************/
bool Document::save()
{
	wxChar buf[256];

	if ( ! wxStrcmp( dataSet->getFilename(), wxT( "" ))) return saveAs();
	if ( ! Horoscope::save() )
	{
		wxSprintf( buf, wxT( "%s %s" ), _( "Cannot save file" ), dataSet->getFilename().c_str() );
		doMessageBox( mainwindow, buf, wxOK | wxICON_ERROR | wxCENTRE );
		return false;
	}
	dirty = false;
	wxCommandEvent event( DOC_SAVED, mainwindow->GetId() );
	event.SetEventObject( this );
	wxPostEvent( mainwindow->GetParent(), event );
	return true;
}

/*****************************************************
**
**   Document   ---   saveAs 
**
******************************************************/
bool Document::saveAs()
{
	wxChar buf[256];
	wxString filename = dataSet->getName();
	if ( filename == wxT( "" ))
	{
		filename.sprintf( wxT( "%s.mtx" ), getHName() );
	}
	wxString filetypes = _( "Maitreya text (*.mtx)|*.mtx| All files (*.*)|*.*" );

	wxFileDialog saveFileDialog( mainwindow, _("Save file as" ), config->defSavePath, filename,
		filetypes, wxSAVE, wxDefaultPosition );
						
	if ( saveFileDialog.ShowModal() == wxID_OK )
	{
		filename.Clear();
		filename << saveFileDialog.GetDirectory().c_str() << wxFileName::GetPathSeparator()
			<< saveFileDialog.GetFilename().c_str();
		if ( ! Horoscope::saveAs( filename.c_str() ) )
		{
			wxSprintf( buf, wxT( "%s %s" ), _( "Cannot save file" ), filename.c_str());
			doMessageBox( mainwindow, buf, wxOK | wxICON_ERROR | wxCENTRE );
			return false;
		}
		dirty = false;
		wxCommandEvent event( DOC_SAVED, mainwindow->GetId() );
		event.SetEventObject( this );
		wxPostEvent( mainwindow->GetParent(), event );
		config->defSavePath = saveFileDialog.GetDirectory();
	}
	else
		{ return false; }
	return true;
}

/*****************************************************
**
**   Document   ---   editData 
**
******************************************************/
void Document::editData()
{
	DataDialog dialog( mainwindow, this );
	dialog.ShowModal();
	wxCommandEvent event( DOC_UPDATED, mainwindow->GetId() );
	event.SetEventObject( this );
	wxPostEvent( mainwindow->GetParent(), event );
}

/*****************************************************
**
**   Document   ---   doQuickPrint 
**
******************************************************/
void Document::doQuickPrint()
{
	Printer( this ).print( config->printDefaultPrintout, wxEmptyString, true );
}

/*****************************************************
**
**   Document   ---   doPrint 
**
******************************************************/
void Document::doPrint()
{
	PrintConfigurationDialog dialog( mainwindow, this );
	moveWindow2Center( &dialog );
	if ( dialog.ShowModal() == wxID_OK )
		Printer( this ).print( dialog.getSelectedPrintout(), dialog.getFilename(), false );
}

/*****************************************************
**
**   Document   ---   StartTimer 
**
******************************************************/
void Document::StartTimer( bool b )
{
	if ( b )
		{ if ( ! timer->IsRunning() ) timer->Start( config->animateInterval * 1000 ); }
	else
	{
		if ( timer->IsRunning() ) timer->Stop();
	}
}

/*****************************************************
**
**   Document   ---   OnTimer 
**
******************************************************/
void Document::OnTimer( wxTimerEvent &event )
{
	double sw;
	if ( config->animateMode )
	{
		sw = config->animateStepLength;
		switch( config->animateStepDimension )
		{
			case 0:  // seconds
				sw /= 86400;
			break;
			case 1:  // minutes
				sw /= 1400;
			break;
			case 2:  // hours
				sw /= 24;
			break;
			case 3:  // days
			break;
			case 4:  // months
				sw *= 30.6;
			break;
			case 5:  // years
				sw *= 365.25;
			break;
			default:  // seconds
				sw /= 86400;
			break;
		}
		setDate( getJD() + sw );
	}
	else
		{ setActualDate(); }
	update();
	updateAllChildWindows();
}


/*****************************************************
**
**   DocumentManager   ---   addDoc 
**
******************************************************/
void DocumentManager::addDoc( Document *doc )
{
	docs.Append( doc );
	for ( unsigned int i = 0; i < clients.size(); i++ ) clients[i]->documentListChanged();
}

/*****************************************************
**
**   DocumentManager   ---   deleteDoc 
**
******************************************************/
void DocumentManager::deleteDoc( Document *doc )
{
	docs.DeleteObject( doc );
	for ( unsigned int i = 0; i < clients.size(); i++ ) clients[i]->documentListChanged();
}

/*****************************************************
**
**   DocumentManager   ---   updateAllDocs 
**
******************************************************/
void DocumentManager::updateAllDocs()
{
	Document *doc;
	for ( Documents::Node *n = docs.GetFirst(); n; n = n->GetNext() )
	{
		doc = n->GetData();
		doc->update();
	}
}

/*****************************************************
**
**   DocumentManager   ---   addClient 
**
******************************************************/
void DocumentManager::addClient( DocumentListener *l )
{
	clients.push_back( l );
}

/*****************************************************
**
**   DocumentManager   ---   deleteClient 
**
******************************************************/
void DocumentManager::deleteClient( DocumentListener *l )
{
	vector<DocumentListener*>::iterator iter;
	for( iter = clients.begin(); iter != clients.end(); iter++ )
	{
		if ( (*iter) == l )
		{
			clients.erase( iter );
			return;
		}
	}
	// OTTO
	//delete l;
	assert( 0 );
}

/*****************************************************
**
**   DocumentManager   ---   getNbDocuments 
**
******************************************************/
int DocumentManager::getNbDocuments()
{
	return docs.GetCount();
}

/**************************************************************
***
**  DocumentManager   ---   getDocument
***
***************************************************************/
Document *DocumentManager::getDocument( const int &i )
{
	return docs.Item( i )->GetData();
}


