/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           EphemView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "EphemView.h"
#endif

#include "EphemView.h"

#include <wx/choice.h>
#include <wx/dc.h>
#include <wx/dcmemory.h>
#include <wx/log.h>
#include <wx/msgdlg.h>
#include <wx/spinctrl.h>
#include <wx/statbox.h>
#include <wx/string.h>

#include "Base.h"
#include "FontProvider.h"
#include "MenuProvider.h"
#include "ToolPanel.h"

#include "Conf.h"
#include "Ephemeris.h"
#include "func.h"
#include "Lang.h"
#include "Writer.h"

extern Config *config;

IMPLEMENT_CLASS( EphemView, MView )
BEGIN_EVENT_TABLE( EphemView, MView )
	EVT_MENU( TBB_NOW, EphemView::OnNow )
  EVT_SPINCTRL( TBS_YEAR, EphemView::OnYearSpin  )
  EVT_SPINCTRL( TBS_MONTH, EphemView::OnMonthSpin )
  EVT_CHOICE( TBS_EPHEMPLANETS, EphemView::OnChoiceCommand )
  EVT_CHOICE( TBS_EPHEMTZ, EphemView::OnChoiceTZ )
  EVT_CHOICE( TBS_DASA, EphemView::OnChoiceCommand )
  EVT_CHOICE( TBS_EPHEMMODE, EphemView::OnChoiceCommand )
  EVT_CHOICE( TBS_EPHEMDEG, EphemView::OnChoiceCommand )
	EVT_IDLE( EphemView::OnIdle )
	EVT_MOUSEWHEEL( EphemView::OnMouseWheelEvent )
END_EVENT_TABLE()

IMPLEMENT_CLASS( EphemWidget, ExportableScrollWidget )
BEGIN_EVENT_TABLE( EphemWidget, ExportableScrollWidget )
END_EVENT_TABLE()


/*****************************************************
**
**   EphemView   ---   Constructor 
**
******************************************************/
EphemView::EphemView( MViewInfo info )
 : MView( info )
{
	supportEWToggle = true;
  int dummy;
  Formatter::get()->getDateIntsFromJD( JDate().getJD(), dummy, month, year );
	expert = new EphemExpert();

  oldmonth = month;
	dasa = 0;
  planets = 1;
  mode = config->ephemMode;
  gmode = ( mode == 1 );
	theText = 0;
	ewidget = 0;
	tz = config->ephemTimezone;
	expert->setPreferVedic( preferVedic );
	expert->prepareMonth( month, year, tz );

	initToolItems();
	initClientView();
	SetSize( 600, 450 );
	updateView = true;
}

/*****************************************************
**
**   EphemView   ---   Destructor 
**
******************************************************/
EphemView::~EphemView()
{
	config->ephemTimezone = tz;
	config->ephemMode = mode;
	delete expert;
}

/*****************************************************
**
**   EphemView   ---   initToolItems
**
******************************************************/
void EphemView::initToolItems()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if( spin_year ) spin_year->SetValue( year );

		wxSpinCtrl *spin_month = (wxSpinCtrl*)toolbar->FindControl( TBS_MONTH );
		if( spin_month ) spin_month->SetValue( month );

		wxChoice *choice_planets = (wxChoice*)toolbar->FindControl( TBS_EPHEMPLANETS );
		if ( choice_planets )
		{
			choice_planets->SetSelection( 1 );
			choice_planets->Enable( ! preferVedic && mode <= 1 );
		}

		wxChoice *choice_ephem_mode = (wxChoice*)toolbar->FindControl( TBS_EPHEMMODE );
		if( choice_ephem_mode ) choice_ephem_mode->SetSelection( mode );

		wxChoice *choice_ephemtz = (wxChoice*)toolbar->FindControl( TBS_EPHEMTZ );
		if( choice_ephemtz ) choice_ephemtz->SetSelection( tz );

		wxChoice *choice_ephemdeg = (wxChoice*)toolbar->FindControl( TBS_EPHEMDEG );
		if ( choice_ephemdeg ) choice_ephemdeg->Enable( mode == 1 );

		wxChoice *choice_dasa = (wxChoice*)toolbar->FindControl( TBS_DASA );
		if ( choice_dasa ) choice_dasa->Enable( mode == 5 );
	}
}

/*****************************************************
**
**   EphemView   ---   OnMouseWheelEvent
**
******************************************************/
void EphemView::OnMouseWheelEvent( wxMouseEvent &event )
{
	int x, y;
	const int offset = 30;
	if ( ewidget )
	{
		ewidget->GetViewStart( &x, &y );
		if ( event.GetWheelRotation() < 0 ) y += offset;
		else y -= offset;
		ewidget->Scroll( x, y );
	}
	else event.Skip();
}

/*****************************************************
**
**   EphemView   ---   initClientView
**
******************************************************/
void EphemView::initClientView()
{
	if ( ewidget ) delete ewidget;
	ewidget = 0;
	if ( theText ) delete theText;
	theText = 0;
	if ( mode == 1 )
	{
		ewidget = new EphemWidget( this, expert );
		ewidget->setPreferVedic( preferVedic );
		widget = ewidget;
	}
	else
	{
		theText = new TextWidget(this, config->preferHtml2Text, CHILD_TEXTCTRL, 
			wxDefaultPosition, wxSize( 500, 400 ));
		widget = theText;
	}
	doLayout();
}

/*****************************************************
**
**   EphemView   ---   OnPreferVedic 
**
******************************************************/
void EphemView::OnPreferVedic( bool b )
{
	preferVedic = b;
	expert->setPreferVedic( b );
	expert->prepareMonth( month, year, tz );
	if ( ewidget ) ewidget->setPreferVedic( b );
	MView::OnPreferVedic( b );
	OnToolbarCommand();
	updateView = true;
}

/*****************************************************
**
**   EphemView   ---   updateText
**
******************************************************/
void EphemView::updateText()
{
	Writer *writer = theText->getWriter();
	writer->beginWriting();
	switch ( mode )
	{
		case 1:
			assert( false );
		break;
		case 2:
			expert->writeDetails( writer );
		break;
		case 3:
			expert->writeIngress( writer, ( preferVedic ? false : planets ) );
		break;
		case 4:
			expert->writeLunar( writer );
		break;
		case 5:
			expert->writeKp( writer, dasa );
		break;
		default:
			expert->writeDefaultEphemeris( writer, ( preferVedic ? false : planets ) );
		break;
	}
	writer->endWriting();
	theText->setContents( writer->getContents());
}

/*****************************************************
**
**   EphemView   ---   OnChoiceTZ 
**
******************************************************/
void EphemView::OnChoiceTZ( wxCommandEvent& )
{
	if ( toolbar )
	{
		wxChoice *choice_ephemtz = (wxChoice*)toolbar->FindControl( TBS_EPHEMTZ );
		if ( choice_ephemtz ) tz = choice_ephemtz->GetSelection();
		config->ephemTimezone = tz;
		expert->prepareMonth( month, year, tz );
		updateView = true;
	}
}

/*****************************************************
**
**   EphemView   ---   OnYearSpin 
**
******************************************************/
void EphemView::OnYearSpin( wxSpinEvent &event )
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if ( spin_year ) year = spin_year->GetValue();
		expert->prepareMonth( month, year, tz );
		updateView = true;
	}
}

/*****************************************************
**
**   EphemView   ---   OnMonthSpin 
**
******************************************************/
void EphemView::OnMonthSpin( wxSpinEvent &event )
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		wxSpinCtrl *spin_month = (wxSpinCtrl*)toolbar->FindControl( TBS_MONTH );
		if ( spin_year && spin_month )
		{
			if ( oldmonth == 12 && spin_month->GetValue() == 1 )
				spin_year->SetValue( spin_year->GetValue() + 1 );
			if ( oldmonth == 1 && spin_month->GetValue() == 12 )
				spin_year->SetValue( spin_year->GetValue() - 1 );

			oldmonth = spin_month->GetValue();
			year = spin_year->GetValue();
			month = spin_month->GetValue();
		}
		expert->prepareMonth( month, year, tz );
		updateView = true;
	}
}
      
/*****************************************************
**
**   EphemView   ---   OnToolbarCommand 
**
******************************************************/
void EphemView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_EPHEMMODE );
		if ( choice_mode )
		{
			if ( mode != choice_mode->GetSelection() )
			{
				mode = choice_mode->GetSelection();
				config->ephemMode = mode;
				bool newgmode = ( mode == 1 );
				if ( gmode != newgmode )  initClientView();
				gmode = newgmode;
			}
		}
		wxChoice *choice_planets = (wxChoice*)toolbar->FindControl( TBS_EPHEMPLANETS );
		if ( choice_planets )
		{
			planets = choice_planets->GetSelection();
			choice_planets->Enable( ! preferVedic && mode <= 1 );
		}

		wxChoice *choice_ephemdeg = (wxChoice*)toolbar->FindControl( TBS_EPHEMDEG );
		if ( choice_ephemdeg )
		{
			setMaxDeg( choice_ephemdeg->GetSelection());
			choice_ephemdeg->Enable( mode == 1 );
		}

		wxChoice *choice_dasa = (wxChoice*)toolbar->FindControl( TBS_DASA );
		if ( choice_dasa )
		{
			dasa = choice_dasa->GetSelection();
			choice_dasa->Enable( mode == 5 );
		}
		updateView = true;
	}
}
   
/*****************************************************
**
**   EphemView   ---   setMaxDeg
**
******************************************************/
void EphemView::setMaxDeg( const int &sel )
{
	double max_deg;
	switch( sel )
	{
		case 0 : max_deg =   360; break;
		case 1 : max_deg =   180; break;
		case 2 : max_deg =    90; break;
		case 3 : max_deg =    45; break;
		case 4 : max_deg =    30; break;
		case 5 : max_deg =  22.5; break;
		case 6 : max_deg = 13.333333333; break;
		case 7 : max_deg = 11.25; break;
		case 8 : max_deg = 5.125; break;
		default: max_deg =   360; break;
	}
	if ( ewidget ) ewidget->setMaxDeg( max_deg );
}

/*****************************************************
**
**   EphemView   ---   OnNow 
**
******************************************************/
void EphemView::OnNow( wxCommandEvent &event )
{
	int dummy;
	if ( toolbar )
	{
		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		wxSpinCtrl *spin_month = (wxSpinCtrl*)toolbar->FindControl( TBS_MONTH );
		if ( spin_year && spin_month )
		{
			Formatter::get()->getDateIntsFromJD( JDate().getJD(), dummy, month, year );
			spin_year->SetValue( year );
			spin_month->SetValue( month );
			oldmonth = month;
		}
		expert->prepareMonth( month, year, tz );
		updateView = true;
	}
}   


/**************************************************************
***
**   EphemView   ---   OnIdle
***
***************************************************************/
void EphemView::OnIdle( wxIdleEvent &event )
{
	if ( ! updateView ) return;
	if ( gmode )
	{
		expert->calcMonth();
		assert( ewidget );
		ewidget->setFilter( planets );
		ewidget->Refresh();
	}
	else
	{
		assert( theText );
		updateText();
	}
	updateView = false;
}

/**************************************************************
***
**   EphemView   ---   getWindowLabel
***
***************************************************************/
const wxChar *EphemView::getWindowLabel( const bool shortname )
{
	return _( "Ephemeris" );
}

/**************************************************************
***
**   EphemWidget   ---   Constructor
***
***************************************************************/
EphemWidget::EphemWidget( wxWindow *parent, EphemExpert *e )
 : ExportableScrollWidget( parent ), expert( e )
{
	max_deg = 360;
	setZoom( 10 );
}

/**************************************************************
***
**   EphemWidget   ---   Destructor
***
***************************************************************/
EphemWidget::~EphemWidget()
{
}

/*****************************************************
**
**   EphemWidget   ---   setZoom 
**
******************************************************/
void EphemWidget::setZoom( const int z )
{
  if ( z ) zoom = z;
  int width = 100 * zoom;
  int height = 500;
  SetVirtualSize( width, height );

  xleft = 4 * hborder;
  xright = width - xleft - 4 * hborder;
  ytop = 2 * vborder;
  ybottom = height - 4 * vborder;
}


/*****************************************************
**
**   EphemWidget   ---   paintRuler
**
******************************************************/
void EphemWidget::paintRuler()
{
	wxString s;
	wxPen *greypen;
	int i;
	int xtext = 50;
	int ytext = 10;
	int ysizey = 20;
	int y = (int)ybottom-30;
	int x = (int)xleft;
	double ystep;

	dc->SetBrush( wxNullBrush );
	dc->SetPen( *wxBLACK_PEN ); 
	dc->DrawRectangle( (int)xleft, (int)ybottom, (int)(xright - xleft), (int)(ytop - ybottom) );
	int mlen = expert->getNumberOfDays();
	double dstep = ( xright - xleft ) / mlen;
	greypen = wxThePenList->FindOrCreatePen( *wxLIGHT_GREY, 1, wxSHORT_DASH );
	for( i = 0; i < mlen; i++ )
	{
		dc->SetPen( *wxBLACK_PEN );
		x =  (int)(xleft+i*dstep);
		dc->DrawLine( x, (int)ybottom, x, (int)ybottom+5 );
    s.sprintf( wxT( "%d" ), i+1 );
		drawTextFormatted( wxRect( x-xtext, y + ysizey + ytext, 2*xtext, 2*ytext ), s, Align::Center );
		if ( ! ( ( i + 1 ) % 5 ) && i > 0 )
		{
			dc->SetPen( *greypen );
			dc->DrawLine( (int)x, (int)ybottom, (int)x, (int)ytop );
		}
	}

	ystep = (ybottom - ytop) / max_deg;
	for( i = 1; i < max_deg; i++ )
	{
		dc->SetPen( *wxBLACK_PEN );
		if ( max_deg >= 180 && ( i % 30 ) ) continue;
		if ( max_deg >= 90 && ( i % 10 ) ) continue;
		if ( max_deg >= 22 && ( i % 5 ) ) continue;
		y = (int)(ybottom - i * ystep);
		dc->DrawLine( (int)xleft-5, (int)y, (int)xleft, (int)y );
    s.sprintf( wxT( "%d" ), i );
		drawTextFormatted( wxRect( (int)xleft-xtext-5, y - ysizey, xtext, 2 * ysizey ), s, Align::Right+Align::VCenter );
		dc->SetPen( *greypen );
		dc->DrawLine( (int)xleft, (int)y, (int)xright, (int)y );
	}
}

/*****************************************************
**
**   EphemWidget   ---   doPaint 
**
******************************************************/
void EphemWidget::doPaint( wxDC *p )
{
  wxCoord w, h;

  assert( p ); 
  dc = p;
  GetVirtualSize( &w, &h );
  dc->SetBrush( *wxGREY_BRUSH );
  dc->DrawRectangle( 0, 0, w, h );

  // White page background
  dc->SetBrush( *wxWHITE_BRUSH );
  dc->SetPen( *wxWHITE_PEN );
	dc->SetFont( *FontProvider().getGraphicFont());
  dc->DrawRectangle( hborder, vborder, (int)(w - 2 * hborder), (int)(h - 2 * vborder) );

  paintRuler();
	paintPlanets();
}

/*****************************************************
**
**   EphemWidget   ---   paintPlanets
**
******************************************************/
void EphemWidget::paintPlanets()
{
	int d, p;            // loop indices fpr planets and days
	int x1, x2, y1, y2;  // daily positions in y and y dimension
	int xp;              // x position for jumps
	int startp, endp;    // planetary indices to start and end with
	double l1, l2;       // length of planets
	double s1, s2;       // daily speed
	double yp, yp2, ydiff;      // length values for jumps
	int ytotal =  (int)(ybottom - ytop);   // total height of interior
	int mlen = expert->getNumberOfDays();  // length of month
	double dstep = ( xright - xleft ) / mlen;  // daily step in x dimension
	int lstep = 20;                 // step height for legend entries
	int legend = (int)ytop + lstep; // Legend on the right side
	wxString s;
	Lang *lang = Lang::get();

	if ( preferVedic )
	{
		startp = ISUN;
		endp = IKETU;
	}
	else
	{
		startp = ( filter == 2 ? WCUPIDO : WSUN );
		endp = ( filter == 1 ?  WLUNARNODE : WPOSEIDON );
	}

  for( p = startp; p <= endp; p++ )
  //for( p = IMOON; p <= IMOON; p++ )
	{
		// loop for AS and MC
		if ( p > WLUNARNODE && p < WCUPIDO ) continue;

		// loop if daily motion of moon is smaller than max_deg
		if ( ( max_deg < 14 ) && ( p == IMOON || p == WMOON )) continue;

		setLineStyle( p );
		for( d = 0; d < mlen; d++ )
		{
			x1 =  (int)(xleft+d*dstep);
			x2 =  (int)(x1+dstep);
			l1 = a_red( expert->getLength( p, d ), max_deg );
			s1 = expert->getSpeed( p, d );
			y1 = (int)(ybottom - l1 * ytotal / max_deg);
			l2 = a_red( expert->getLength( p, d+1 ), max_deg );
			s2 = expert->getSpeed( p, d+1 );
			y2 = (int)(ybottom - l2 * ytotal / max_deg);
			if ( expert->getRetro( p, d ))
			{
				if (( l1 < l2 ) && expert->getRetro( p, d+1 )) // handle jumps
				{
					yp = l1;
					yp2 = max_deg - l2;
					ydiff = yp / ( yp + yp2 );
					xp = (int)( x1 + ydiff * dstep );
					//printf( "1 planet %d yp %f yp2 %f x1 %d x2 %d xp %d l1 %f l2 %f ydiff %f\n", p, yp, yp2, x1, x2, xp, l1, l2, ydiff );
					dc->DrawLine( x1, y1, xp, (int)ybottom );
					dc->DrawLine( xp, (int)ytop, x2, y2 );
				}
				else { dc->DrawLine( x1, y1, x2, y2 ); } // that's normal
			}
			else
			{
				if (( l1 > l2 ) && ! expert->getRetro( p, d+1 )) // handle jumps
				{
					yp = max_deg - l1;
					yp2 = l2;
					ydiff = yp / ( yp + yp2 );
					xp = (int)( x1 + ydiff * dstep );
					//printf( "2 planet %d yp %f yp2 %f x1 %d x2 %d xp %d l1 %f l2 %f ydiff %f\n", p, yp, yp2, x1, x2, xp, l1, l2, ydiff );
					dc->DrawLine( x1, y1, xp, (int)ytop );
					dc->DrawLine( xp, (int)ybottom, x2, y2 );
				}
				else { dc->DrawLine( x1, y1, x2, y2 ); } // that happens normally
			}
		}
		dc->DrawLine( (int)xright+10, legend, (int)xright+25, legend );
		drawTextFormatted( wxRect( (int)xright+30, legend-20, 60, 40 ), lang->getObjectName( p, TMEDIUM ),
			Align::Left+Align::VCenter );
		legend += lstep;
	}
}

/*****************************************************
**
**   EphemWidget   ---   setLineStyle
**
******************************************************/
void EphemWidget::setLineStyle( const int &planet )
{
	int oi = expert->getPlanetaryIndex( planet );
	wxPen *pen = wxBLACK_PEN;
	const int lw = 2;
	switch( oi )
	{
		case OSUN:
			pen = wxThePenList->FindOrCreatePen( wxColour( 187, 23, 23 ), lw, wxSOLID );
		break;
		case OMOON:
			pen = wxThePenList->FindOrCreatePen( wxColour( 57, 153, 255 ), lw, wxSOLID );
		break;
		case OMERCURY:
			pen = wxThePenList->FindOrCreatePen( wxColour( 102, 204, 102 ), lw, wxSOLID );
		break;
		case OVENUS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 255, 23, 243 ), lw, wxSOLID );
		break;
		case OMARS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 255, 0, 0 ), lw, wxSOLID );
		break;
		case OJUPITER:
			pen = wxThePenList->FindOrCreatePen( wxColour( 130, 130, 204 ), lw, wxSOLID );
		break;
		case OSATURN:
			pen = wxThePenList->FindOrCreatePen( wxColour( 24, 35, 104 ), lw, wxSOLID );
		break;
		case OURANUS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 234, 183, 44 ), lw, wxSOLID );
		break;
		case ONEPTUNE:
			pen = wxThePenList->FindOrCreatePen( wxColour( 107, 234, 52 ), lw, wxSOLID );
		break;
		case OPLUTO:
			pen = wxThePenList->FindOrCreatePen( wxColour( 127, 234, 232 ), lw, wxSOLID );
		break;
		case OMEANNODE:
			pen = wxThePenList->FindOrCreatePen( wxColour( 146, 132, 117 ), lw, wxSOLID );
		break;
		case OTRUENODE:
			pen = wxThePenList->FindOrCreatePen( wxColour( 146, 132, 117 ), lw, wxSOLID );
		break;
		case OCUPIDO:
			pen = wxThePenList->FindOrCreatePen( wxColour( 255, 23, 243 ), lw, wxSHORT_DASH );
		break;
		case OHADES:
			pen = wxThePenList->FindOrCreatePen( wxColour( 24, 35, 104 ), lw, wxSHORT_DASH );
		break;
		case OZEUS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 130, 130, 204 ), lw, wxSHORT_DASH );
		break;
		case OKRONOS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 255, 0, 0 ), lw, wxSHORT_DASH );
		break;
		case OAPOLLON:
			pen = wxThePenList->FindOrCreatePen( wxColour( 102, 204, 102 ), lw, wxSHORT_DASH );
		break;
		case OADMETOS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 127, 234, 232 ), lw, wxSHORT_DASH );
		break;
		case OVULKANUS:
			pen = wxThePenList->FindOrCreatePen( wxColour( 234, 183, 44 ), lw, wxSHORT_DASH );
		break;
		case OPOSEIDON:
			pen = wxThePenList->FindOrCreatePen( wxColour( 107, 234, 52 ), lw, wxSHORT_DASH );
		break;
		default:
			assert( false );
		break;
	}
	dc->SetPen( *pen);
}

/**************************************************************
***
**   ViewFactory   ---   createEphemView
***
***************************************************************/
MView *ViewFactory::createEphemView( MViewInfo info )
{
	return new EphemView( info );
}

