/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           GraphicView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "GraphicView.h"
#endif

#include "GraphicView.h"

#include <wx/app.h>
#include <wx/dc.h>
#include <wx/dcmemory.h>
#include <wx/event.h>
#include <wx/filedlg.h>
#include <wx/sizer.h>
#include <wx/statbox.h>
#include <wx/textctrl.h>

#include "ColorProvider.h"
#include "constants.h"
#include "func.h"
#include "Conf.h"
#include "Lang.h"
#include "Session.h"
#include "Varga.h"

#include "Document.h"
#include "FontProvider.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "Painter.h"
#include "VedicGraphic.h"
#include "WesternGraphic.h"

extern Config *config;

IMPLEMENT_CLASS( GraphicView, MView )
BEGIN_EVENT_TABLE( GraphicView, MView )
	//EVT_SIZE( GraphicView::OnSize )
END_EVENT_TABLE()

IMPLEMENT_CLASS( GraphicWidget, ExportableScrollWidget )
BEGIN_EVENT_TABLE( GraphicWidget, ExportableScrollWidget )
END_EVENT_TABLE()

/*****************************************************
**
**   GraphicView   ---   Constructor 
**
******************************************************/
GraphicView::GraphicView( MViewInfo info )
 : MView( info )
{
	supportEWToggle = true;
	gwidget = new GraphicWidget( this, doc, preferVedic, viewinfo.varga, -1 ); 
	widget = gwidget;
}

/*****************************************************
**
**   GraphicView   ---   Destructor 
**
******************************************************/
GraphicView::~GraphicView()
{
}

/*****************************************************
**
**   GraphicView   ---   getIStyle 
**
******************************************************/
int GraphicView::getIStyle()
{
	return gwidget->getIStyle();
}

/*****************************************************
**
**   GraphicView   ---   getWStyle 
**
******************************************************/
int GraphicView::getWStyle()
{
	return gwidget->getWStyle();
}

/*****************************************************
**
**   GraphicView   ---   OnSize 
**
******************************************************/
void GraphicView::OnSize( wxSizeEvent &event )
{
	//printf( "GraphicView::OnSize x %d y %d\n", event.GetSize().x, event.GetSize().y );
	gwidget->SetSize( event.GetSize() );
	gwidget->Refresh();
	MView::OnSize( event );
}

/*****************************************************
**
**   GraphicView   ---   SetContextMenuInfo 
**
******************************************************/
void GraphicView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
	menuinfo.graphic = ( preferVedic ? getIStyle() : getWStyle() );
	MView::SetContextMenuInfo( menuinfo );
}

/*****************************************************
**
**   GraphicView   ---   OnPreferVedic 
**
******************************************************/
void GraphicView::OnPreferVedic( bool f )
{
	gwidget->preferVedic = f;
	MView::OnPreferVedic( f );
}


/**************************************************************
***
**   GraphicView   ---   getWindowLabel
***
***************************************************************/
const wxChar *GraphicView::getWindowLabel( const bool shortname )
{
	static wxChar tmp[64];
	if ( preferVedic )
	{
		if ( shortname )
		{
			if ( gwidget->getVarga() < NB_VARGAS-1 )
				wxSprintf( tmp, wxT( "D%d" ), VargaExpert().getVargaDivision( gwidget->getVarga() ) );
			else
				wxSprintf( tmp, wxT( "%s" ), VargaExpert().getVargaName( gwidget->getVarga()));
		}
		else
		{
			if ( gwidget->getVarga() < NB_VARGAS-1 )
			{
				wxSprintf( tmp, wxT( "%s (D%d)" ), VargaExpert().getVargaName( gwidget->getVarga() ),
					VargaExpert().getVargaDivision( gwidget->getVarga() ) );
			}
			else
			{
				wxSprintf( tmp, wxT( "%s" ), VargaExpert().getVargaName( gwidget->getVarga()));
			}
		}
	}
	else
	{
		return shortname ? _( "Graphic" ) : _( "Western Graphic" );
	}
	return tmp;
}

/*****************************************************
**
**   GraphicView   ---   dispatchCommand 
**
******************************************************/
bool GraphicView::dispatchCommand( const int &command )
{
	int istyle = gwidget->getIStyle();
	int wstyle = gwidget->getWStyle();

  switch( command )
	{
		case CHILD_WSHOWASPECTS:
		case CHILD_WSHOWRETRO:
		case CHILD_WSHOWHOUSES:
		case CHILD_WSHOWURANIAN:
		case CHILD_ISHOWRETRO:
		case CHILD_ISHOWARUDHA:
		case CHILD_ISHOWLAGNAS:
		case CHILD_ISHOWSOUTH:
		case CHILD_ISHOWNORTH:
		case CHILD_ISHOWEAST:
		case CHILD_EXPORT:
			return gwidget->dispatchCommand( command );
		break;
		default:
			return MView::dispatchCommand( command );
		break;
	}
	gwidget->setIStyle( istyle ); 
	gwidget->setWStyle( wstyle ); 
	gwidget->Refresh();

 return true;
}

/*****************************************************
**
**   GraphicView   ---   doExport 
**
******************************************************/
void GraphicView::doExport()
{
	gwidget->doExport();
}

/*****************************************************
**
**   GraphicWidget   ---   Constructor 
**
******************************************************/
GraphicWidget::GraphicWidget( wxWindow *parent, Horoscope *doc, bool indian, int varga, wxWindowID id,
	const wxPoint& pos, const wxSize& size )
 : ExportableScrollWidget( parent, id, pos, size )
{
	this->doc = doc;
	this->varga = varga;
	this->preferVedic = indian;
	istyle = config->iGraphicStyle;
	wstyle = config->wGraphicStyle;
	isBlank = false;
	PushEventHandler( new MyEventPropagator() );
}

/*****************************************************
**
**   GraphicWidget   ---   dispatchCommand
**
******************************************************/
bool GraphicWidget::dispatchCommand( const int &command )
{
  switch( command )
	{
		case CHILD_WSHOWASPECTS:
			wstyle & WGRAPHIC_STYLE_SHOW_ASPECTS ? wstyle &= ~WGRAPHIC_STYLE_SHOW_ASPECTS
				: wstyle |= WGRAPHIC_STYLE_SHOW_ASPECTS;
		break;
		case CHILD_WSHOWRETRO:
			 wstyle & WGRAPHIC_STYLE_SHOW_RETRO ? wstyle &= ~WGRAPHIC_STYLE_SHOW_RETRO
			 	: wstyle |= WGRAPHIC_STYLE_SHOW_RETRO;
		break;
		case CHILD_WSHOWHOUSES:
			wstyle & WGRAPHIC_STYLE_SHOW_HOUSES ? wstyle &= ~WGRAPHIC_STYLE_SHOW_HOUSES
				: wstyle |= WGRAPHIC_STYLE_SHOW_HOUSES;
		break;
		case CHILD_WSHOWURANIAN:
			wstyle & WGRAPHIC_STYLE_SHOW_URANIAN ? wstyle &= ~WGRAPHIC_STYLE_SHOW_URANIAN
				: wstyle |= WGRAPHIC_STYLE_SHOW_URANIAN;
		break;
		case CHILD_ISHOWRETRO:
			istyle & IGRAPHIC_STYLE_SHOW_RETRO ? istyle &= ~IGRAPHIC_STYLE_SHOW_RETRO
				: istyle |= IGRAPHIC_STYLE_SHOW_RETRO;
		break;
		case CHILD_ISHOWARUDHA:
			istyle & IGRAPHIC_STYLE_SHOW_ARUDHAS ? istyle &= ~IGRAPHIC_STYLE_SHOW_ARUDHAS
				: istyle |= IGRAPHIC_STYLE_SHOW_ARUDHAS;
		break;
		case CHILD_ISHOWLAGNAS:
			istyle & IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS ? istyle &= ~IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS :
				istyle |= IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS;
		break;
		case CHILD_ISHOWSOUTH:
			istyle |= IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		case CHILD_ISHOWNORTH:
			istyle &= ~IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle |= IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		case CHILD_ISHOWEAST:
			istyle &= ~IGRAPHIC_STYLE_SOUTH_INDIAN;
			istyle &= ~IGRAPHIC_STYLE_NORTH_INDIAN;
			istyle |= IGRAPHIC_STYLE_EAST_INDIAN;
		break;
		case CHILD_EXPORT:
			doExport();
		break;
	}
	Refresh();
	return true;
}

/*****************************************************
**
**   GraphicWidget   ---   doPaint 
**
******************************************************/
void GraphicWidget::doPaint( wxDC *p )
{
	int w, h;
	ColorProvider cp;
	assert( p );
	p->GetSize( &w, &h  );

	wxColour bgcolor = *cp.getBgColor();
	wxColour fgcolor = *cp.getFgColor();
	p->SetBackground( wxBrush( bgcolor, wxTRANSPARENT ));

	// flood fill
	p->SetBrush( wxBrush( bgcolor, wxSOLID ));
	p->SetPen( wxPen( bgcolor, 1, wxSOLID ));
	p->DrawRectangle( 0, 0, w, h );

	p->SetBrush( wxBrush( bgcolor, wxTRANSPARENT ));
	p->SetBackgroundMode( wxTRANSPARENT );	
	p->SetPen( wxPen( fgcolor, 1, wxSOLID ));
	p->SetTextBackground( bgcolor );
	p->SetTextForeground( fgcolor );
	p->SetFont( *FontProvider().getGraphicFont() );
	//p->SetFont( *FontProvider().getSymbolFont() );
	
	Painter *painter = new DcPainter( p );
	if ( ! isBlank )
	{
	preferVedic ?  VedicGraphic( painter ).paint( doc, istyle, varga, wxRect( 0, 0, w, h ))
		: WesternGraphic( painter ).paint( doc, wxRect( 0, 0, w, h ), wstyle );
	}
	delete painter;
}

/**************************************************************
***
**   ViewFactory   ---   createGraphicView
***
***************************************************************/
MView *ViewFactory::createGraphicView( MViewInfo info )
{
	info.showtoolbar = false;
	return new GraphicView( info );
}


