/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           MidpointView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "MidpointView.h"
#endif

#include "MidpointView.h"

#include <wx/string.h>

#include <wx/choice.h>
#include <wx/icon.h>
#include <wx/sizer.h>
#include <wx/spinctrl.h>
#include <wx/statbox.h>
#include <wx/textctrl.h>

#include "Base.h"
#include "Document.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "ToolPanel.h"
#include "dialogs/DialogElements.h"

#include "constants.h"
#include "Conf.h"
#include "DataSet.h"
#include "func.h"
#include "Lang.h"
#include "Midpoint.h"
#include "Session.h"

extern Config *config;

IMPLEMENT_CLASS( MidpointBaseView, TextViewBase )

IMPLEMENT_CLASS( MidpointView, MidpointBaseView )
BEGIN_EVENT_TABLE( MidpointView, MidpointBaseView )
  EVT_CHOICE( TBS_MIDPOINTMODE, MidpointView::OnChoiceCommand )
  EVT_CHOICE( TBS_SORT, MidpointView::OnChoiceCommand )
  EVT_CHOICE( TBS_FILTER, MidpointView::OnChoiceCommand )
  EVT_SPINCTRL( TBS_ORBIS, MidpointView::OnSpinCommand )
  EVT_TEXT_ENTER( TBS_ORBIS, MidpointView::OnChoiceCommand )
  EVT_CHOICE( TBS_CIRCLEDEGREES, MidpointView::OnChoiceCommand )
  EVT_CHOICE( TBS_YEARLENGTH, MidpointView::OnChoiceCommand )
  EVT_SPINCTRL( TBS_YEAR, MidpointView::OnSpinCommand )
  EVT_TEXT_ENTER( TBS_YEAR, MidpointView::OnChoiceCommand )
END_EVENT_TABLE()

IMPLEMENT_CLASS( TransitView, MidpointBaseView )
BEGIN_EVENT_TABLE( TransitView, MidpointBaseView )
	EVT_COMMAND( TBS_DATE, COMMAND_DATE_SPIN_CHANGED, TransitView::OnGeneralCommand )
	EVT_COMMAND( TBS_TIME, COMMAND_TIME_SPIN_CHANGED, TransitView::OnGeneralCommand )
	EVT_MENU( TBB_NOW, TransitView::OnActual )
	EVT_CHOICE( TBS_TRANSITMODE, TransitView::OnChoiceCommand )
	EVT_CHOICE( TBS_YEARLENGTH, TransitView::OnChoiceCommand )
	EVT_SPINCTRL( TBS_ORBIS, TransitView::OnSpinCommand )
	EVT_TEXT_ENTER( TBS_ORBIS, TransitView::OnGeneralCommand )
	EVT_CHOICE( TBS_SORT, TransitView::OnChoiceCommand )
	EVT_CHOICE( TBS_CIRCLEDEGREES, TransitView::OnGeneralCommand )
END_EVENT_TABLE()

/*****************************************************
**
**   MidpointBaseView   ---   Constructor 
**
******************************************************/
MidpointBaseView::MidpointBaseView( MViewInfo info )
 : TextViewBase( info )
{
	hsSystem = config->hsSystem;
	supportEWToggle = false;
	preferVedic = false;
}

/*****************************************************
**
**   MidpointBaseView   ---   dispatchCommand 
**
******************************************************/
bool MidpointBaseView::dispatchCommand( const int &command )
{
	if ( command == CHILD_HSEBERTIN )
	{
		onChangeHsSystem( false );
		return true;
	}
	else if ( command == CHILD_HSWITTE )
	{
		onChangeHsSystem( true );
		return true;
	}
	else return TextViewBase::dispatchCommand( command );
}

/*****************************************************
**
**   MidpointBaseView   ---   SetContextMenuInfo 
**
******************************************************/
void MidpointBaseView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
  menuinfo.midpoint = hsSystem;
  TextViewBase::SetContextMenuInfo( menuinfo );
} 

/*****************************************************
**
**   MidpointBaseView   ---   onChangeHsSystem 
**
******************************************************/
void MidpointBaseView::onChangeHsSystem( const int &hsSystem )
{
	this->hsSystem = hsSystem;
	writeContents();
}

/*****************************************************
**
**   MidpointView   ---   Constructor 
**
******************************************************/
MidpointView::MidpointView( MViewInfo info )
 : MidpointBaseView( info )
{
	int day, month;

	if ( viewinfo.mode != -1 ) mode = viewinfo.mode;
	else mode = 0;

	orbis = config->wOrbisRadix;
	circle = config->hsGradkreisRadix;
	Formatter::get()->getDateIntsFromJD( JDate().getJD(), day, month, year );
	yl = config->hsYearLength;
	yearlength = calcYearLength( yl, false );
	sortorder = 0;
	filter = 0;

	initToolItems();
	OnToolbarCommand(); // does it all, refresh and hide-and-seek for tool items
}

/*****************************************************
**
**   MidpointView   ---   Destructor 
**
******************************************************/
MidpointView::~MidpointView()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis  && config->hsRememberOrbis ) config->wOrbisRadix = spin_orbis->GetValue();

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if( choice_circle && config->hsRememberGradkreis ) config->hsGradkreisRadix = choice_circle->GetSelection();
	}
}

/**************************************************************
***
**   MidpointView   ---   initToolItems
***
***************************************************************/
void MidpointView::initToolItems()
{
	if ( toolbar )
	{
		wxChoice *choice_viewmode = (wxChoice*)toolbar->FindControl( TBS_MIDPOINTMODE );
		if( choice_viewmode ) choice_viewmode->SetSelection( mode );

		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis ) spin_orbis->SetValue( orbis );

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle ) choice_circle->SetSelection( circle );

		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if( spin_year ) spin_year->SetValue( year );

		YearLengthChoice *choice_yl = (YearLengthChoice*)toolbar->FindControl( TBS_YEARLENGTH );
		if( choice_yl ) choice_yl->SetSelection( yl );

		SortChoice *choice_sort = (SortChoice*)toolbar->FindControl( TBS_SORT );
		if( choice_sort ) choice_sort->SetSelection( sortorder );

		FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
		if ( choice_filter ) choice_filter->SetSelection( filter );
	}
}

/*****************************************************
**
**   MidpointView   ---   onChangeHsSystem 
**
******************************************************/
void MidpointView::onChangeHsSystem( const int &hsSystem )
{
	if ( toolbar )
	{
		FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
		assert( choice_filter );
		choice_filter->recreate( hsSystem );
	}
	MidpointBaseView::onChangeHsSystem( hsSystem );
}

/*****************************************************
**
**   MidpointView   ---   write
**
******************************************************/
void MidpointView::write( Writer *writer )
{
  MidpointExpert expert( hsSystem );
	switch( mode )
	{
		case 0:
			expert.writeRadixAnalysis( writer, doc, orbis, circle, sortorder, filter );
		break;
		case 1:
			expert.writeChainAnalysis( writer, doc, orbis, circle, sortorder, 0 );
		break;
		case 2:
		expert.writeChainAnalysis( writer, doc, orbis, circle, sortorder, 1 );
			break;
		case 3:
			expert.writeYearlyPreviewAnalysis( writer, doc->getDataSet(), yearlength, circle, year );
		break;
		default:
			printf( "Error: wrong view mode %d in midpoint window\n", mode );
		break;
	}
}

/*****************************************************
**
**   MidpointView   ---   OnToolbarCommand 
**
******************************************************/
void MidpointView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_viewmode = (wxChoice*)toolbar->FindControl( TBS_MIDPOINTMODE );
		if( choice_viewmode ) mode = choice_viewmode->GetSelection();

		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if( spin_orbis )
		{
			spin_orbis->Enable( mode != 3 );
			orbis = spin_orbis->GetValue();
		}

		wxSpinCtrl *spin_year = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if( spin_year )
		{
			spin_year->Enable( mode == 3 );
			year = spin_year->GetValue();
		}

		YearLengthChoice *choice_yl = (YearLengthChoice*)toolbar->FindControl( TBS_YEARLENGTH );
		if( choice_yl )
		{
			choice_yl->Enable( mode == 3 );
			yearlength = choice_yl->getYearLength();
		}

		SortChoice *choice_sort = (SortChoice*)toolbar->FindControl( TBS_SORT );
		if( choice_sort )
		{
			choice_sort->Enable( mode == 0 );
			sortorder = choice_sort->GetSelection();
		}

		FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
		if( choice_filter )
		{
			choice_filter->Enable( mode == 0 );
			filter = choice_filter->getFilter();
		}

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle )
		{
			circle = choice_circle->GetSelection();
		}
	}
	writeContents();
}


/**************************************************************
***
**   MidpointView   ---   getWindowLabel
***
***************************************************************/
const wxChar *MidpointView::getWindowLabel( const bool shortname )
{
	//return shortname ? _( "Midpoint" ) : _( "Midpoint Window" );
	return _( "Midpoint" );
}


/*****************************************************
**
**   TransitView   ---   Constructor 
**
******************************************************/
TransitView::TransitView( MViewInfo info )
 : MidpointBaseView( info )
{
	dataset.setLocation( *getDoc()->getLocation() ); // use birth place instead of standard location

	if ( viewinfo.mode != -1 ) mode = viewinfo.mode;
	else mode = config->hsTransitModus;

	yl = config->hsYearLength;
	yearlength = calcYearLength( yl, false );
	orbis = config->wOrbisTransit;
	circle = config->hsGradkreisTransit;
	sortorder = 0;

	initToolItems();
	OnToolbarCommand(); // does it all, refresh and hide-and-seek for tool items
}

/*****************************************************
**
**   TransitView   ---   Destructor 
**
******************************************************/
TransitView::~TransitView()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis  && config->hsRememberOrbis ) config->wOrbisTransit = spin_orbis->GetValue();

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle  && config->hsRememberGradkreis ) config->hsGradkreisTransit = choice_circle->GetSelection();

		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_TRANSITMODE );
		if ( choice_mode  && config->hsRememberTransitModus ) config->hsTransitModus = choice_mode->GetSelection();
	}
}

/**************************************************************
***
**   TransitView   ---   initToolItems
***
***************************************************************/
void TransitView::initToolItems()
{
  if ( toolbar )
  {
		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_TRANSITMODE );
		if ( choice_mode ) choice_mode->SetSelection( config->hsTransitModus );

		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if( spin_date && spin_time )
		{
			dataset.setDate( spin_date->getJD() - .5 + spin_time->getTime()/24 );
		}
		else dataset.setDate( JDate().getJD());

		YearLengthChoice *choice_yl = (YearLengthChoice*)toolbar->FindControl( TBS_YEARLENGTH );
		if( choice_yl )
		{
			choice_yl->SetSelection( yl );
		}

		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if( spin_orbis ) spin_orbis->SetValue( orbis );

		CircleDegreeChoice *choice_degree = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if( choice_degree ) choice_degree->SetSelection( circle );

		SortChoice *choice_sort = (SortChoice*)toolbar->FindControl( TBS_SORT );
		if( choice_sort ) choice_sort->SetSelection( sortorder );
	}
}

/*****************************************************
**
**   TransitView   ---   OnToolbarCommand 
**
******************************************************/
void TransitView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_TRANSITMODE );
		if ( choice_mode ) mode = choice_mode->GetSelection();
					 
		YearLengthChoice *choice_yl = (YearLengthChoice*)toolbar->FindControl( TBS_YEARLENGTH );
		if ( choice_yl )
		{
			choice_yl->Enable( choice_mode->GetSelection() != 1 );
			yl = choice_yl->GetSelection();
			yearlength = choice_yl->getYearLength();
		}

		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if ( spin_date && spin_time )
		{
			dataset.setDate( spin_date->getJD() - .5 + spin_time->getTime()/24 );
		}

		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis ) orbis = spin_orbis->GetValue();

		CircleDegreeChoice *choice_degree = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_degree ) circle = choice_degree->GetSelection();

		SortChoice *choice_sort = (SortChoice*)toolbar->FindControl( TBS_SORT );
		if ( choice_sort ) sortorder = choice_sort->GetSelection();
	}
	writeContents();
}

/*****************************************************
**
**   TransitView   ---   OnActual 
**
******************************************************/
void TransitView::OnActual( wxCommandEvent &event )
{
	if ( toolbar )
	{
		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if ( spin_date && spin_time )
		{
			spin_date->setActual();
			spin_time->setActual();
		}
		else dataset.setDate( JDate().getJD());
	}
}

/*****************************************************
**
**   TransitView   ---   write
**
******************************************************/
void TransitView::write( Writer *writer )
{
  MidpointExpert expert( hsSystem );
	if ( toolbar )
	{
		switch( mode )
		{
			case 0:
				expert.writeSolarArcAnalysis( writer, doc->getDataSet(), &dataset, yearlength, orbis, circle, sortorder );
			break;
			case 1:
				expert.writeTransitAnalysis( writer, doc->getDataSet(), &dataset, orbis, circle, sortorder );
			break;
			case 2:
				expert.writeDirectionAnalysis( writer, doc->getDataSet(), &dataset, yearlength, orbis, circle, sortorder );
			break;
			case 3:
				expert.writeConstantArcAnalysis( writer, doc->getDataSet(), &dataset, yearlength, orbis, circle, sortorder );
			break;
			case 4:
				expert.writePlanetaryArcAnalysis( writer, OMOON, doc->getDataSet(), &dataset, yearlength, orbis, circle,
					sortorder );
			break;
			default:
				printf( "Error: wrong view mode %d in transit window\n", mode );
			break;
		}
	}
}

/**************************************************************
***
**   TransitView   ---   getWindowLabel
***
***************************************************************/
const wxChar *TransitView::getWindowLabel( const bool shortname )
{
	//return shortname ? _( "Transit" ) : _( "Transit Window" );
	return _( "Transit" );
}

/**************************************************************
***
**   ViewFactory   ---   createMidpointView
***
***************************************************************/
MView *ViewFactory::createMidpointView( MViewInfo info )
{
	return new MidpointView( info );
}

/**************************************************************
***
**   ViewFactory   ---   createTransitView
***
***************************************************************/
MView *ViewFactory::createTransitView( MViewInfo info )
{
	return new TransitView( info );
}

