/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           PartnerView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "PartnerView.h"
#endif

#include "PartnerView.h"

#include <wx/string.h>

#include <wx/choice.h>
#include <wx/sizer.h>
#include <wx/spinctrl.h>
#include <wx/splitter.h>
#include <wx/statbox.h>

#include "Base.h"
#include "constants.h"
#include "Conf.h"
#include "DataSet.h"
#include "Document.h"
#include "func.h"
#include "GraphicView.h"
#include "IconProvider.h"
#include "Lang.h"
#include "MenuProvider.h"
#include "Midpoint.h"
#include "Pair.h"
#include "Session.h"
#include "ToolPanel.h"
#include "Writer.h"
#include "dialogs/DialogElements.h"

extern Config *config;

IMPLEMENT_CLASS( PartnerTextWidget, TextWidget )
BEGIN_EVENT_TABLE( PartnerTextWidget, TextWidget )
END_EVENT_TABLE()

IMPLEMENT_CLASS( PartnerView, MView )
BEGIN_EVENT_TABLE( PartnerView, MView )
	EVT_CHOICE( TBS_PARTNER1, PartnerView::OnPartnerChoice )
	EVT_CHOICE( TBS_PARTNER2, PartnerView::OnPartnerChoice )
	EVT_SPINCTRL( TBS_ORBIS, PartnerView::OnSpinCommand )
	EVT_CHOICE( TBS_SORT, PartnerView::OnChoiceCommand )
	EVT_CHOICE( TBS_FILTER, PartnerView::OnChoiceCommand )
	EVT_CHOICE( TBS_CIRCLEDEGREES, PartnerView::OnChoiceCommand )
END_EVENT_TABLE()
const int tb_partnerwindow[] = { TBS_PARTNER1 , TBS_PARTNER2, TBS_ORBIS, TBS_SORT, TBS_FILTER, TBS_CIRCLEDEGREES, -1 };

IMPLEMENT_CLASS( PartnerViewDocListener, DocumentListener )

/*****************************************************
**
**   PartnerTextWidget   ---   Constructor 
**
******************************************************/
PartnerTextWidget::PartnerTextWidget( wxWindow *parent, ChildWindow *frame, wxWindowID id,
	const wxPoint& pos, const wxSize& size )
 : TextWidget( parent, config->preferHtml2Text, id, pos, size )
{
}

/*****************************************************
**
**   PartnerView   ---   Constructor 
**
******************************************************/
PartnerView::PartnerView( MViewInfo info, DocumentManager *manager )
 : MView( info )
{
	docmanager = manager;
	mode = config->partnerStyle;
	textctrl1 = 0;
	pptext = 0;
	splitter = 0;
	gwidget = 0;
	ch = 0;
	supportEWToggle = true;

	hsSystem = config->hsSystem;
	istyle = config->iGraphicStyle;
	wstyle = config->wGraphicStyle;
	orbis = config->wOrbisPartner;
	circle = config->hsGradkreisPartner;
	sortorder = 0;
	filter = 0;

	listener = new PartnerViewDocListener( manager, this );

	updatePartnerChoices();
	initToolItems();
	updateToolItems();
	initClientView();
	onDocChanged();
	SetSize( 600, 450 );
}

/*****************************************************
**
**   PartnerView   ---   Destructor 
**
******************************************************/
PartnerView::~PartnerView()
{
	config->partnerStyle = mode;
	if ( toolbar )
	{
		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis  && config->hsRememberOrbis ) config->wOrbisPartner = spin_orbis->GetValue();

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle  && config->hsRememberGradkreis ) config->hsGradkreisPartner = choice_circle->GetSelection();
	}
	docmanager->deleteClient( listener );
	delete ch;
}

/*****************************************************
**
**   PartnerView   ---   postCreate
**
******************************************************/
void PartnerView::postCreate()
{
	if ( mode == 1 )
	{
		assert( splitter );
		splitter->SetSashPosition( 200 );
	}
}

/*****************************************************
**
**   PartnerView   ---   initToolItems
**
******************************************************/
void PartnerView::initToolItems()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis ) spin_orbis->SetValue( orbis );

		wxChoice *choice_sort = (wxChoice*)toolbar->FindControl( TBS_SORT );
		if ( choice_sort ) choice_sort->SetSelection( sortorder );

		FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
		if ( choice_filter ) choice_filter->SetSelection( filter );

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle ) choice_circle->SetSelection( circle );
	}
}

/*****************************************************
**
**   PartnerView   ---   OnToolbarCommand 
**
******************************************************/
void PartnerView::OnToolbarCommand()
{
	if ( toolbar )
	{
		setupCharts();
		wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
		if ( spin_orbis ) orbis = spin_orbis->GetValue();

		wxChoice *choice_sort = (wxChoice*)toolbar->FindControl( TBS_SORT );
		if ( choice_sort ) sortorder = choice_sort->GetSelection();

		FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
		if ( choice_filter ) filter = choice_filter->GetSelection();

		CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
		if ( choice_circle ) circle = choice_circle->GetSelection();
	}
	onDocChanged();
}

/*****************************************************
**
**   PartnerView   ---   updateToolItems 
**
******************************************************/
void PartnerView::updateToolItems()
{
	if ( ! toolbar ) return;
	bool showExtraItems = ( ! preferVedic && ! mode );

	wxSpinCtrl *spin_orbis = (wxSpinCtrl*)toolbar->FindControl( TBS_ORBIS );
	if ( spin_orbis ) spin_orbis->Enable( showExtraItems );

	wxChoice *choice_sort = (wxChoice*)toolbar->FindControl( TBS_SORT );
	if ( choice_sort ) choice_sort->Enable( showExtraItems );

	FilterChoice *choice_filter = (FilterChoice*)toolbar->FindControl( TBS_FILTER );
	if ( choice_filter ) choice_filter->Enable( showExtraItems );

	CircleDegreeChoice *choice_circle = (CircleDegreeChoice*)toolbar->FindControl( TBS_CIRCLEDEGREES );
	if ( choice_circle ) choice_circle->Enable( showExtraItems );
}

/*****************************************************
**
**   PartnerView   ---   initClientView 
**
******************************************************/
void PartnerView::initClientView()
{
	if ( widget ) delete widget;
	widget = 0;
	gwidget = 0;
	pptext = 0;
	splitter = 0;
	textctrl1 = 0;
	if ( ch ) delete ch;
	ch = 0;
	if ( mode == 0 )
	{
		textctrl1 = new TextWidget(this, config->preferHtml2Text, CHILD_TEXTCTRL, wxDefaultPosition, wxSize( 600, 400 ));
		widget = textctrl1;
	}
	else
	{
		if ( ! ch ) ch = new CompositHoroscope();
		splitter = new MySplitter( this );

		//pptext = new PlanetaryPositionTextCtrl( splitter, ch );
		pptext = new PlanetaryPositionTextCtrl( splitter, ch, config->preferHtml2Text, -1,
			wxDefaultPosition, wxSize( 200, 400 ) );
		pptext->setPrefix( _( "Composite Chart" ) );
		gwidget = new GraphicWidget( splitter, ch, preferVedic );
		splitter->SplitVertically( pptext, gwidget, 200 );
		widget = splitter;   
	}
	doLayout();
	updateToolItems();
}

/**************************************************************
***
** 
***
***************************************************************/
const wxChar *PartnerView::getWindowLabel( const bool shortname )
{
  return shortname ? _( "Partner" ) : _( "Partner Window" );
}

/*****************************************************
**
**   PartnerView   ---   onDocChanged 
**
******************************************************/
void PartnerView::onDocChanged()
{
	wxString s;
	if ( mode == 0 )
	{
		if ( textctrl1 ) writeStream( s );
	}
	else
	{
		if ( ch ) ch->update( h1, h2 );
		if ( pptext )
		{
			pptext->setEmpty( ! h1 || ! h2 || ( h1 == h2 ));
			pptext->setEmptyMessage( ( h1 == h2 && h1 && h2 ) ? _( "Charts identical" ) : _( "No Chart loaded" ) );
			pptext->dump();
		}
		if ( gwidget )
		{
			gwidget->setBlank( ! h1 || ! h2 || ( h1 == h2 ));
			gwidget->Refresh();
		}
	}
}

/*****************************************************
**
**   PartnerView   ---   setupCharts 
**
******************************************************/
void PartnerView::setupCharts()
{
	int sel1 = -1, sel2 = -1;

	wxChoice *choice_partner1 = (wxChoice*)toolbar->FindControl( TBS_PARTNER1 );
	if ( choice_partner1 ) sel1 = choice_partner1->GetSelection();

	wxChoice *choice_partner2 = (wxChoice*)toolbar->FindControl( TBS_PARTNER2 );
	if ( choice_partner2 ) sel2 = choice_partner2->GetSelection();
	if ( sel1 > -1 ) h1 = docmanager->getDocument( sel1 );
	else h1 = 0;
	if ( sel2 > -1 ) h2 = docmanager->getDocument( sel2 );
	else h2 = 0;
}

/*****************************************************
**
**   PartnerView   ---   OnPartnerChoice 
**
******************************************************/
void PartnerView::OnPartnerChoice( wxCommandEvent &event )
{
	setupCharts();
	onDocChanged();
}

/*****************************************************
**
**   PartnerView   ---   dispatchCommand 
**
******************************************************/
bool PartnerView::dispatchCommand( const int &command )
{
	wxSize mySize = GetSize();
	switch( command )
	{
		case CHILD_HSEBERTIN:
			onChangeHsSystem( false );
		break;
		case CHILD_HSWITTE:
			onChangeHsSystem( true );
		break;
		case CHILD_PARTNER_TEXT:
			mode = 0;
			initClientView();
			SetSize( mySize );
			onDocChanged();
		break;
		case CHILD_PARTNER_COMPOSIT:
			mode = 1;
			initClientView();
			SetSize( mySize );
			onDocChanged();
		break;
		case CHILD_WSHOWASPECTS:
		case CHILD_WSHOWRETRO:
		case CHILD_WSHOWHOUSES:
		case CHILD_WSHOWURANIAN:
		case CHILD_ISHOWRETRO:
		case CHILD_ISHOWARUDHA:
		case CHILD_ISHOWLAGNAS:
		case CHILD_ISHOWSOUTH:
		case CHILD_ISHOWNORTH:
		case CHILD_ISHOWEAST:
		case CHILD_EXPORT:
			if ( gwidget ) gwidget->dispatchCommand( command );
			if ( pptext ) pptext->dispatchCommand( command );
		break;
		default:
			return MView::dispatchCommand( command );
		break;
	}
	return true;
}

/*****************************************************
**
**   PartnerView   ---   onChangeHsSystem 
**
******************************************************/
void PartnerView::onChangeHsSystem( const int &hsSystem )
{
	this->hsSystem = hsSystem;
	onDocChanged();
}

/*****************************************************
**
**   PartnerView   ---   updatePartnerChoices 
**
******************************************************/
void PartnerView::updatePartnerChoices()
{
	if ( ! toolbar ) return;
	int sel1 = -1, sel2 = -1;
	wxString docname;

	int docnum = docmanager->getNbDocuments();
	wxChoice *choice_partner1 = (wxChoice*)toolbar->FindControl( TBS_PARTNER1 );
	if ( choice_partner1 ) sel1 = choice_partner1->GetSelection();

	wxChoice *choice_partner2 = (wxChoice*)toolbar->FindControl( TBS_PARTNER2 );
	if ( choice_partner2 ) sel2 = choice_partner2->GetSelection();
	if ( sel2 >= docnum ) sel2 = sel1;

	if ( sel1 == -1 && docnum > 0 ) sel1 = 0;
	if ( ( sel2 == sel1 || sel2 == -1 ) && docnum > 0 ) sel2 = docnum - 1;

	choice_partner1->Clear();
	choice_partner2->Clear();

	int i = 1;
	for ( i = 0; i < docnum; i++ )
	{
		docname = docmanager->getDocument( i )->getHName();
		choice_partner1->Append( docname );
		choice_partner2->Append( docname );
	}
	choice_partner1->SetSelection( sel1 );
	choice_partner2->SetSelection( sel2 );
	if ( sel1 != -1 && docnum > 0 ) h1 = docmanager->getDocument( Min( sel1, docnum ));
	else h1 = 0;
	if ( sel2 != -1 && docnum > 0 ) h2 = docmanager->getDocument( Min( sel2, docnum ));
	else h2 = 0;
}

/*****************************************************
**
**   PartnerView   ---   OnPreferVedic 
**
******************************************************/
void PartnerView::OnPreferVedic( bool f )
{
	preferVedic = f;
	updateToolItems();
	if ( gwidget ) gwidget->preferVedic = f;
	if ( pptext ) pptext->setPreferVedic( f );
	onDocChanged();
}

/*****************************************************
**
**   PartnerView   ---   updateClientView 
**
******************************************************/
void PartnerView::updateClientView()
{
	int sel1 = -1, sel2 = -1;
	Horoscope *h1, *h2;

	wxChoice *choice_partner1 = (wxChoice*)toolbar->FindControl( TBS_PARTNER1 );
	if ( choice_partner1 ) sel1 = choice_partner1->GetSelection();
	wxChoice *choice_partner2 = (wxChoice*)toolbar->FindControl( TBS_PARTNER2 );
	if ( choice_partner2 ) sel2 = choice_partner2->GetSelection();
	h1 = docmanager->getDocument( sel1 );
	h2 = docmanager->getDocument( sel2 );
}

/*****************************************************
**
**   PartnerView   ---   SetContextMenuInfo 
**
******************************************************/
void PartnerView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
	menuinfo.partnerStyle = mode;
	menuinfo.midpoint = hsSystem;
	if ( gwidget ) 
	{
		menuinfo.graphic = ( preferVedic ? gwidget->getIStyle() : gwidget->getWStyle() );
	}
	MView::SetContextMenuInfo( menuinfo );
}


/*****************************************************
**
**   PartnerView   ---   writeStream 
**
******************************************************/
void PartnerView::writeStream( wxString &s )
{
	MidpointExpert expert( hsSystem );

	Writer *writer = textctrl1->getWriter();
	writer->beginWriting();
	
	if ( ! h1 || ! h2 )
		{ writer->writeLine( wxT( "No horoscope loaded" )); }
	else if ( h1 == h2 )
			{ writer->writeLine( wxT( "Horoscopes identical" )); }
	else
	{
		if ( ! preferVedic )
			{ expert.writePartnerAnalysis( writer, h1, h2, orbis, circle, sortorder, filter ); }
		else
		{
			Pair pair( h1, h2 );
			pair.update();
			pair.write( writer );
		}
	}
	writer->endWriting();
	textctrl1->setContents( writer->getContents());
}

/*****************************************************
**
**   PartnerView   ---   documentListChanged 
**
******************************************************/
void PartnerView::documentListChanged()
{
	updatePartnerChoices();
	onDocChanged();
}

/*****************************************************
**
**   PartnerView   ---   updateAllDocs 
**
******************************************************/
void PartnerView::updateAllDocs()
{
}

/*****************************************************
**
**   PartnerViewDocListener   ---   Constructor 
**
******************************************************/
PartnerViewDocListener::PartnerViewDocListener( DocumentManager *manager, PartnerView *view )
  :  DocumentListener( manager )
{
	this->manager = manager;
	this->view = view;
	manager->addClient( this );
}

/*****************************************************
**
**   PartnerViewDocListener   ---   Destructor 
**
******************************************************/
PartnerViewDocListener::~PartnerViewDocListener()
{
	manager->deleteClient( this );
}

/*****************************************************
**
**   PartnerViewDocListener   ---   updateAllDocs 
**
******************************************************/
void PartnerViewDocListener::updateAllDocs()
{
	view->updateAllDocs();
}

/*****************************************************
**
**   PartnerViewDocListener   ---   documentListChanged 
**
******************************************************/
void PartnerViewDocListener::documentListChanged()
{
	view->documentListChanged();
}

/**************************************************************
***
**    ViewFactory   ---   createPartnerView
***
***************************************************************/
MView *ViewFactory::createPartnerView( MViewInfo info, DocumentManager *manager )
{
	return new PartnerView( info, manager );
}


