/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           PdfWriter.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "PdfWriter.h"
#endif

#include "PdfWriter.h"

#include "Ashtakavarga.h"
#include "Conf.h"
#include "config.h"
#include "Dasa.h"
#include "func.h"
#include "Horoscope.h"
#include "Midpoint.h"
#include "PrintoutHelper.h"
#include "PrintoutManager.h"
#include "SbcView.h"
#include "Session.h"
#include "Varga.h"
#include "VedicGraphic.h"
#include "WesternGraphic.h"
  
#include <wx/defs.h>
#include <wx/filename.h>
#include <wx/font.h>
#ifdef __WXMSW__
#include <wx/mimetype.h>
#endif
  
extern Config *config;

/*****************************************************
**
**   PdfWriter   ---   Constructor
**
******************************************************/
PdfWriter::PdfWriter( Horoscope *h, const int papersize )
 //: wxPdfDocument( wxPORTRAIT, wxString(_T("mm")), papersize )
 : wxPdfDocument( wxPORTRAIT, wxString(_T("mm")), (wxPaperSize)papersize )
 //: wxPdfDocument()
{
	horoscope = h;
	info = 0;
	type = WRITER_PDF;
	Session *session = Session::get();
	
	/*
	int l = session->getLocaleCode();
	if ( l == wxLANGUAGE_TELUGU )
	{
		AddFont( wxT( "Akshar" ));
		header1FontName = wxT( "Akshar" );
		header2FontName = wxT( "Akshar" );
		header3FontName = wxT( "Akshar" );
		defaultFontName = wxT( "Akshar" );
		tinyFontName = wxT( "Akshar" );
	}
	else 
	{
	*/
		header1FontName = wxT( "Arial" );
		header2FontName = wxT( "Arial" );
		header3FontName = wxT( "Arial" );
		defaultFontName = wxT( "Times" );
		tinyFontName = wxT( "Times" );
	//}
	header1FontStyle = wxT( "B" );
	header2FontStyle = wxT( "IB" );
	header3FontStyle = wxT( "U" );
	defaultFontStyle = wxT ( "" );
	tinyFontStyle = wxT ( "I" );
	header1FontSize = 14;
	header2FontSize = 12;
	header3FontSize = 10;
	defaultFontSize = 10;
	tinyFontSize = 8;

	lsSolid = new wxPdfLineStyle();
  wxPdfArrayDouble dash1;
  dash1.Add(10.);
  dash1.Add(20.);
  dash1.Add(5.);
  dash1.Add(10.);
  lsDot = new wxPdfLineStyle( -1, wxPDF_LINECAP_BUTT, wxPDF_LINEJOIN_MITER, dash1, 10.);

  //wxPdfArrayDouble dash2;
  lsLongDash = new wxPdfLineStyle( -1, wxPDF_LINECAP_BUTT, wxPDF_LINEJOIN_MITER, dash1, 0.);

  wxPdfArrayDouble dash3;
  dash3.Add(2.);
  dash3.Add(10.);
  lsDotDash = new wxPdfLineStyle( -1, wxPDF_LINECAP_ROUND, wxPDF_LINEJOIN_ROUND, dash3, 0.);

	setDefaultFont();
}

/*****************************************************
**
**   PdfWriter   ---   Destructor
**
******************************************************/
PdfWriter::~PdfWriter()
{
	delete lsSolid;
	delete lsDot;
	delete lsLongDash;
	delete lsDotDash;
}

/*****************************************************
**
**   PdfWriter   ---   setDefaultFont
**
******************************************************/
void PdfWriter::setDefaultFont()
{
	SetFont( defaultFontName, defaultFontStyle, defaultFontSize );
}

/*****************************************************
**
**   PdfWriter   ---   Footer
**
******************************************************/
void PdfWriter::Footer()
{
	if ( PageNo() == 1 ) return;
	SetY(-15);
	SetFont( tinyFontName, tinyFontStyle, tinyFontSize );
	Cell(0,10,wxString::Format(_T("Page %d/{nb}"),PageNo()),0,0,wxPDF_ALIGN_CENTER);
	setDefaultFont();
}

/*****************************************************
**
**   PdfWriter   ---   setPrintInfo
**
******************************************************/
void PdfWriter::setPrintInfo( const int &p )
{
	info = PrintoutManager::get()->getPrintInfoById( p );
}

/*****************************************************
**
**   PdfWriter   ---   printItem
**
******************************************************/
void PdfWriter::printItem( PrintItem *item )
{
	Session *session = Session::get();
  VargaView view( horoscope );
  AshtakavargaExpert aexpert( &view );
	switch( item->type )
	{
		case PI_TITLE:
			writeTitle();
		break;
		case PI_BASEDATA:
			PrintoutHelper( horoscope, this ).writePrintoutSummary( item->style );
			Ln();
		break;
		case PI_VEDICGRAPHICPAIR:
			writeVedicPair( item->p1, item->p2, item->style );
		break;
		case PI_SBC:
			writeSbc();
		break;
		case PI_MIDPOINT:
			writeMidpoint();
		break;
		case PI_GRAPHIC:
			writeWesternChart( item->style );
		break;
		case PI_ASHTAKAVARGA:
			aexpert.update(); aexpert.write( this );
		break;
		case PI_DASA:
			session->getDasaExpert( item->style )->write( this, horoscope );
		break;
		default:
			printf( "ERROR, unknown type %d\n", item->type );
		break;
	}
}

/*****************************************************
**
**   PdfWriter   ---   print
**
******************************************************/
void PdfWriter::print()
{
	for( unsigned int i = 0; i < info->items.size(); i++ ) printItem( info->items[i] );
}

/*****************************************************
**
**   PdfWriter   ---   printPageFrame
**
******************************************************/
void PdfWriter::printPageFrame()
{
	//if ( config->printShowPageFrame )
		//dc->DrawRoundedRectangle( (int)xleft, (int)ytop, (int)xw, (int)yh, (int)hline );
}

/*****************************************************
**
**   PdfWriter   ---   beginWriting
**
******************************************************/
void PdfWriter::beginWriting()
{
  AliasNbPages(); 
  wxString s;
	Session *session = Session::get();
	s << session->getGuiAppName() << wxT( " " ) << _( "Version" ) << wxT( " " ) << session->getVersion();
  SetAuthor( s );
	s.Clear();
	s << wxT( "Chart of " ) << horoscope->getHName();
  SetTitle( s );
	SetCreator( wxGetUserName() );
  AddPage();
}

/*****************************************************
**
**   PdfWriter   ---   endWriting
**
******************************************************/
void PdfWriter::endWriting()
{
	wxString command, buf;

  SaveAsFile( filename );

	if ( config->printLaunchPdfViewer )
	{
#ifdef __WXMSW__
		wxFileType *filetype = wxMimeTypesManager().GetFileTypeFromExtension( wxT( "pdf" ));
		if ( filetype != 0 )
		{
			command = filetype->GetOpenCommand( filename );
		}
		else
		{
			doMessageBox( 0, _( "Cannot launch pdf viewer. Do you have a pdf viewer (e.g. Acrobat Reader) installed?" ),
				wxOK | wxICON_ERROR | wxCENTRE );
			return;
		}
#else
		if ( config->printPdfViewerCommand == wxEmptyString )
		{
			config->printPdfViewerCommand = locateFile( wxT( "xpdf" ));
			if ( config->printPdfViewerCommand == wxEmptyString )
				config->printPdfViewerCommand = locateFile( wxT( "kghostview" ));
			if ( config->printPdfViewerCommand == wxEmptyString )
				config->printPdfViewerCommand = locateFile( wxT( "acroread" ));
		}
		if ( config->printPdfViewerCommand == wxEmptyString )
		{
			doMessageBox( 0, _( "Cannot launch pdf viewer. Please configure the pdf viewer in the configuration dialog." ),
				wxOK | wxICON_ERROR | wxCENTRE );
			return;
		}
		command << config->printPdfViewerCommand << wxT( " \"" ) << filename << wxT( "\"" );
#endif
		long r = wxExecute( command );
		if ( r == -1 )
		{
			buf.Printf( _( "Can't launch pdf viewer, command is \"%s\" (error code %ld)" ), command.c_str(), r );
			doMessageBox( 0, buf, wxOK | wxICON_ERROR | wxCENTRE );
		}
	}
	else
	{
		s.Printf( _( "Chart printed to file %s." ), filename.c_str() );
		doMessageBox( 0, s, wxOK | wxCENTRE );
	}
}

/*****************************************************
**
**   PdfWriter   ---   writeTitle
**
******************************************************/
void PdfWriter::writeTitle()
{
  writeHeader1( horoscope->getHName() );
}

/*****************************************************
**
**   PdfWriter   ---   writeMidpoint
**
******************************************************/
void PdfWriter::writeMidpoint()
{
	MidpointExpert( config->hsSystem ).writeRadixAnalysis( this, horoscope,
		config->wOrbisRadix, config->hsGradkreisRadix );
}

/*****************************************************
**
**   PdfWriter   ---   writeSbc
**
******************************************************/
void PdfWriter::writeSbc()
{
	/*
	double height = xw/2;
	wxColourDatabase db;
	SbcWriter writer( horoscope, horoscope );

  writer.bgcolor = *wxWHITE;
  writer.fgcolor = *wxBLACK;
  writer.pcolor = *wxBLACK;
  writer.mcolor = *wxLIGHT_GREY;
  writer.a1color = db.Find( wxT( "DARK SLATE GREY" ));
  writer.a1color = db.Find( wxT( "DIM GREY" ));
  writer.a2color = *wxLIGHT_GREY;
	//writer.doPaint( dc, wxRect( (int)xleft, (int)ycursor, (int)xw, (int)height ));
	ycursor += height;
	*/
}

/*****************************************************
**
**   PdfWriter   ---   writeWesternChart
**
******************************************************/
void PdfWriter::writeWesternChart( int style )
{
	double totalwidth = GetPageWidth() - GetLeftMargin() - GetRightMargin();
	double width = totalwidth;
	double height = width;
	if ( GetY() + height > GetPageHeight() ) AddPage();

	if ( style == -1 ) style = config->wGraphicStyle;
  WesternGraphic graphic( this );
	graphic.paint( horoscope, wxRect( (int)GetLeftMargin(), (int)GetY(), (int)width, (int)height ), style );
	SetY( GetY() + height );
}

/*****************************************************
**
**   PdfWriter   ---   writeVedicPair
**
******************************************************/
void PdfWriter::writeVedicPair( const int &varga1, const int &varga2, int style )
{
	double totalwidth = GetPageWidth() - GetLeftMargin() - GetRightMargin();
	double width = totalwidth/2;
	double height = width;
  if ( style & IGRAPHIC_STYLE_SOUTH_INDIAN ) height *= .85;
	if ( GetY() + height > GetPageHeight() ) AddPage();

	if ( style == -1 ) style = config->iGraphicStyle;
  VedicGraphic graphic( this );
	graphic.paint( horoscope, style, varga1,
		wxRect( (int)GetLeftMargin(), (int)GetY(), (int)width, (int)height ));
	graphic.paint( horoscope, style, varga2,
		wxRect( (int)(GetLeftMargin() + width), (int)GetY(), (int)width, (int)height ));
	SetY( GetY() + height+5 );
}

/*****************************************************
**
**   PdfWriter   ---   writeHeader1
**
******************************************************/
void PdfWriter::writeHeader1( const wxString &s1 )
{
	SetFont( header1FontName, header1FontStyle, header1FontSize );
	MultiCell( 0, 7, s1, wxPDF_BORDER_NONE, wxPDF_ALIGN_CENTER );
	setDefaultFont();
	//Ln();
}

/*****************************************************
**
**   PdfWriter   ---   writeHeader2
**
******************************************************/
void PdfWriter::writeHeader2( const wxString &s1 )
{
	SetFont( header2FontName, header2FontStyle, header2FontSize );
	MultiCell(0,5,s1);
	setDefaultFont();
}

/*****************************************************
**
**   PdfWriter   ---   writeHeader3
**
******************************************************/
void PdfWriter::writeHeader3( const wxString &s1 )
{
	SetFont( header3FontName, header3FontStyle, header3FontSize );
	MultiCell(0,4,s1);
	setDefaultFont();
}

/*****************************************************
**
**   PdfWriter   ---   writeParagraph
**
******************************************************/
void PdfWriter::writeParagraph( const wxString &s1 )
{
	MultiCell(0,4,s1);
}

/*****************************************************
**
**   PdfWriter   ---   writeLine
**
******************************************************/
void PdfWriter::writeLine( const wxString &s1 )
{
	MultiCell(0,4,s1);
}

/*****************************************************
**
**   PdfWriter   ---   writeTableHeaderEntry
**
******************************************************/
void PdfWriter::writeTableHeaderEntry( const wxString &s1 )
{
}   


/*****************************************************
**
**   PdfWriter   ---   writeTable
**
******************************************************/
void PdfWriter::writeTable( Table &t )
{
	int c, r;
	double d;
	TableEntry entry;
	Row row;
	int align, frame;
	double total_width = 0, left_p;

	for( r = 0; r < (int)t.nb_rows; r++ )
	{
		row = t.contents[r];
		for( c = 0; c < (int)t.nb_cols; c++ )
		{
			entry = row.value[c];
			d = GetStringWidth( entry.value );
			t.contents[r].value[c].width = (int)d+3;
			t.col_width[c] = Max( (int)d+3, t.col_width[c] );
		}
	}
	for ( c = 0; c < (int)t.nb_cols; c++ )
	{
		total_width += t.col_width[c];
	}
	double pw = GetPageWidth() - GetLeftMargin() - GetRightMargin();
	left_p = GetLeftMargin() + ( pw - total_width ) / 2;
	SetX( left_p );
	for( r = 0; r < (int)t.nb_rows; r++ )
	{
		row = t.contents[r];
		for( c = 0; c < (int)t.nb_cols; c++ )
		{
			frame = wxPDF_BORDER_NONE;
			if ( c == 0 ) frame = wxPDF_BORDER_LEFT;
			if ( c == (int)t.nb_cols-1 ) frame = wxPDF_BORDER_RIGHT;
			if ( r == 0 || r == 1 ) frame |= wxPDF_BORDER_TOP;
			if ( r == (int)t.nb_rows-1 ) frame |= wxPDF_BORDER_BOTTOM;
			if ( t.col_line[c] ) frame |= wxPDF_BORDER_RIGHT;

			entry = row.value[c];
			if ( entry.isHeader ) SetFont( defaultFontName, wxT( "B" ), defaultFontSize );
			else SetFont( defaultFontName, defaultFontStyle, defaultFontSize );
			switch( t.col_alignment[c] )
			{
				case TABLE_ALIGN_LEFT: align = wxPDF_ALIGN_LEFT; break;
				case TABLE_ALIGN_RIGHT: align = wxPDF_ALIGN_RIGHT; break;
				case TABLE_ALIGN_CENTER: align = wxPDF_ALIGN_CENTER; break;
				default: align = wxPDF_ALIGN_LEFT; break;
			}
			Cell(t.col_width[c],5, entry.value, frame );
		}
		Ln();
		SetX( left_p );
	}
	SetX( GetLeftMargin() );
	SetFont( defaultFontName, defaultFontStyle, defaultFontSize );
}

/*****************************************************
**
**   PdfWriter   ---  writeListInColumns
**
******************************************************/
void PdfWriter::writeListInColumns( const vector<wxString> v, const int &colnr )
{
	unsigned int i, j, index;
	unsigned int ePerCol = (int)v.size() / colnr;
	if ( (int)v.size() % colnr ) ePerCol++;

	Table table( colnr, ePerCol );
	for( i = 0; i < (unsigned int)colnr; i++ )
	{
		table.col_alignment[i] = TABLE_ALIGN_LEFT;
		for( j = 0; j < ePerCol; j++ )
		{
			index = i*ePerCol + j;
			if ( index < v.size() ) table.setEntry( i, j, v[index] );
		}
		s << Endl;
	}
	writeTable( table );
}

/*****************************************************
**
**   PdfWriter   ---  writeMidpointDiagram
**
******************************************************/
void PdfWriter::writeMidpointDiagram( vector<MidpointDiagram> mp )
{
}

/*****************************************************
**
**   PdfWriter   ---  drawRectangle
**
******************************************************/
void PdfWriter::drawRectangle( const double &x, const double &y, const double &w, const double &h )
{
	Rect( x, y, w, h );
}

/*****************************************************
**
**   PdfWriter   ---  drawLine
**
******************************************************/
void PdfWriter::drawLine( const double &x1, const double &y1, const double &x2, const double &y2 )
{
	Line( x1, y1, x2, y2 );
}

/*****************************************************
**
**   PdfWriter   ---  drawTextFormatted
**
******************************************************/
void PdfWriter::drawTextFormatted( const double &x, const double &y, const double &w, const double &h,
	const wxString &t, const int& align )
{
	double oldx = GetX();
	double oldy = GetY();
	SetXY( x, y );
	int a = wxPDF_ALIGN_LEFT;
	if ( align & Align::HCenter ) a = wxPDF_ALIGN_CENTER;
	else if ( align & Align::Right ) a = wxPDF_ALIGN_RIGHT;

	//Cell( w, h, t, wxPDF_BORDER_FRAME, 1, a );
	Cell( w, h, t, wxPDF_BORDER_NONE, 1, a );
	SetXY( oldx, oldy );
}

/*****************************************************
**
**   PdfWriter   ---  getTextExtent
**
******************************************************/
void PdfWriter::getTextExtent( const wxString &s, double *w, double *h )
{
	*w = GetStringWidth( s ) + 2;
	*h = defaultFontSize/2.6;
}

/*****************************************************
**
**   PdfWriter   ---   drawEllipse
**
******************************************************/
void PdfWriter::drawEllipse( const double &x0, const double &y0, const double &xmax, const double &ymax )
{
	Ellipse( x0 + xmax/2 , y0 + ymax/2, xmax/2, ymax/2 );
}

/*****************************************************
**
**   PdfWriter   ---   drawArc 
**
******************************************************/
void PdfWriter::drawArc( const double &x0, const double &y0, const double &xmax, const double &ymax,
	const double &w1, const double &w2 )
{
	Ellipse( x0 + xmax/2 , y0 + ymax/2, xmax/2, ymax/2, 0, w1, w2 );
	//Ellipse( x0, y0, xmax, ymax, w1, w2 );
}

/*****************************************************
**
**   PdfWriter   ---   drawPoint 
**
******************************************************/
void PdfWriter::drawPoint( const double &x, const double &y )
{
	Line( x, y, x, y );
}

/*****************************************************
**
**   PdfWriter   ---   setTransparentBrush
**
******************************************************/
void PdfWriter::setTransparentBrush()
{
	setBrushColor( *wxWHITE );
}

/*****************************************************
**
**   PdfWriter   ---   setBrushColor
**
******************************************************/
void PdfWriter::setBrushColor( const wxColour &c )
{
	SetFillColor( c );
}

/*****************************************************
**
**   PdfWriter   ---   getBrushColor
**
******************************************************/
wxColour PdfWriter::getBrushColor()
{
	return *wxWHITE;
}

/*****************************************************
**
**   PdfWriter   ---   setTransparentPen
**
******************************************************/
void PdfWriter::setTransparentPen()
{
	SetDrawColor( *wxWHITE );
}

/*****************************************************
**
**   PdfWriter   ---   setPenColor
**
******************************************************/
void PdfWriter::setPenColor( const wxColour &c )
{
	SetDrawColor( c );
}

/*****************************************************
**
**   PdfWriter   ---   getPenColor
**
******************************************************/
wxColour PdfWriter::getPenColor()
{
	return *wxBLACK;
}

/*****************************************************
**
**   PdfWriter   ---   setLineStyle
**
******************************************************/
void PdfWriter::setLineStyle( const int &s )
{
	if ( s == wxDOT_DASH ) SetLineStyle( *lsDotDash );
	else if ( s == wxDOT ) SetLineStyle( *lsDot );
	else if ( s == wxLONG_DASH ) SetLineStyle( *lsLongDash );
	else SetLineStyle( *lsSolid );
}



