/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           SbcView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "SbcView.h"
#endif

#include "SbcView.h"
#include "SbcView_defs.h"

#include <stdio.h>

#include <wx/button.h>
#include <wx/dc.h>
#include <wx/filedlg.h>
#include <wx/filename.h>
#include <wx/sizer.h>
#include <wx/splitter.h>
#include <wx/statbox.h>

#include "func.h"
#include "Base.h"
#include "ColorProvider.h"
#include "Document.h"
#include "FontProvider.h"
#include "IconProvider.h"
#include "MenuProvider.h"
#include "ToolPanel.h"
#include "Conf.h"
#include "Lang.h"
#include "Writer.h"
#include "WriterFactory.h"
#include "dialogs/DialogElements.h"

extern Config *config;

IMPLEMENT_CLASS( SbcView, MView )
BEGIN_EVENT_TABLE( SbcView, MView )
  EVT_COMMAND( TBS_DATE, COMMAND_DATE_SPIN_CHANGED, SbcView::OnChoiceCommand )
  EVT_COMMAND( TBS_TIME, COMMAND_TIME_SPIN_CHANGED, SbcView::OnChoiceCommand )
  EVT_MENU( TBB_NOW, SbcView::OnActual )
  EVT_CHOICE( TBS_SBCMODE, SbcView::OnChoiceCommand )
  EVT_CHOICE( TBS_SBCTEXT, SbcView::OnChoiceCommand )
	//EVT_SIZE( SbcView::OnSize )
END_EVENT_TABLE()

IMPLEMENT_CLASS( SbcWidget, ExportableScrollWidget )
BEGIN_EVENT_TABLE( SbcWidget, ExportableScrollWidget )
	EVT_LEFT_DOWN( SbcWidget::OnMouseLeft )
END_EVENT_TABLE()

const int nak_aspects[28][4] = {
	{ ASWINI, ROHINI, PPHALGUNI, JYESHTA },
	{ BHARANI, KRITTIKA, MAKHA, ANURADHA },
	{ KRITTIKA, VISAKHA, SRAVANA, BHARANI },
	{ ROHINI, SWATI, ABHIJIT, ASWINI },
	{ MRIGASIRA, CHITTA, USHADHA, REVATI },
	{ ARUDRA, HASTHA, PSHADHA, UBHADRA },
	{ PUNARVASU, UPHALGUNI, MULA, PBHADRA },
	{ PUSHYAMI, PPHALGUNI, JYESHTA, SATHABISHAK },
	{ ASLESHA, MAKHA, ANURADHA, DHANISHTA },
	{ MAKHA, SRAVANA, BHARANI, ASLESHA },
	{ PPHALGUNI, ABHIJIT, ASWINI, PUSHYAMI },
	{ UPHALGUNI, USHADHA, REVATI, PUNARVASU },
	{ HASTHA, PSHADHA, UBHADRA, ARUDRA },
	{ CHITTA, MULA, PBHADRA, MRIGASIRA },
	{ SWATI, JYESHTA, SATHABISHAK, ROHINI },
	{ VISAKHA, ANURADHA, DHANISHTA, KRITTIKA },
	{ ANURADHA, BHARANI, ASLESHA, VISAKHA },
	{ JYESHTA, ASWINI, PUSHYAMI, SWATI },
	{ MULA, REVATI, PUNARVASU, CHITTA },
	{ PSHADHA, UBHADRA, ARUDRA, HASTHA },
	{ USHADHA, PBHADRA, MRIGASIRA, UPHALGUNI },
	{ ABHIJIT, SATHABISHAK, ROHINI, PPHALGUNI },
	{ SRAVANA, MAKHA, KRITTIKA, DHANISHTA },
	{ DHANISHTA, ASLESHA, VISAKHA, SRAVANA },
	{ SATHABISHAK, PUSHYAMI, SWATI, ABHIJIT },
	{ PBHADRA, PUNARVASU, CHITTA, USHADHA },
	{ UBHADRA, ARUDRA, HASTHA, PSHADHA },
	{ REVATI, MRIGASIRA, UPHALGUNI, MULA }
};

const bool isbenefic[10] = { false, true, false, true, true, true, false, false, false, true };

/*****************************************************
**
**   SbcView   ---   Constructor 
**
******************************************************/
SbcView::SbcView( MViewInfo info )
 : MView( info )
{
	supportEWToggle = false;
	preferVedic = true;
	th = new Horoscope();
	th->update();

	splitter = new MySplitter( this, -1, wxDefaultPosition, wxSize( 800, 400 ) );
	theText = new TextWidget( splitter, config->preferHtml2Text );//, -1, wxDefaultPosition, wxSize( 250, 400 ) );
	gwidget = new SbcWidget( splitter, doc, th, -1, wxDefaultPosition, wxSize( 550, 400 ));
	splitter->SplitVertically( theText, gwidget ); //, 250 );
	widget = splitter;
	textmode = gocharamode = 0;

	initToolItems();
	OnDataChanged();
}

/*****************************************************
**
**   SbcView   ---   Destructor 
**
******************************************************/
SbcView::~SbcView()
{
	delete th;
}

/*****************************************************
**
**   SbcView   ---   OnPaint 
**
******************************************************/
void SbcView::OnPaint(wxPaintEvent& event)
{
	writeContents();
}

/*****************************************************
**
**   SbcView   ---   postCreate
**
******************************************************/
void SbcView::postCreate()
{
	wxSplitterWindow *splitter = (wxSplitterWindow*)widget;
	splitter->SetSashPosition( 250 );
}

/*****************************************************
**
**   SbcView   ---   SetContextMenuInfo 
**
******************************************************/
void SbcView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
	menuinfo.isSbc = true;
	MView::SetContextMenuInfo( menuinfo );
}

/*****************************************************
**
**   SbcView   ---   initToolItems
**
******************************************************/
void SbcView::initToolItems()
{
	if ( toolbar )
	{
		wxChoice *choice_textmode = (wxChoice*)toolbar->FindControl( TBS_SBCTEXT );
		if ( choice_textmode ) choice_textmode->SetSelection( textmode );

		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_SBCMODE );
		if( choice_mode ) choice_mode->SetSelection( gocharamode );

		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		if ( spin_date ) spin_date->Enable( gocharamode );

		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if ( spin_time ) spin_time->Enable( gocharamode );

		toolbar->EnableTool( TBB_NOW, gocharamode );
	}
}

/*****************************************************
**
**   SbcView   ---   OnToolbarCommand 
**
******************************************************/
void SbcView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_textmode = (wxChoice*)toolbar->FindControl( TBS_SBCTEXT );
		if ( choice_textmode ) textmode = choice_textmode->GetSelection();

		wxChoice *choice_mode = (wxChoice*)toolbar->FindControl( TBS_SBCMODE );
		if( choice_mode ) gocharamode = choice_mode->GetSelection();

		gwidget->setGocharaMode( gocharamode == 1 );
		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if ( spin_date &&  spin_time )
		{
			double jd = spin_date->getJD() - .5 + spin_time->getTime()/24;
			th->setDate( jd + ( config->defaultLocation.getTimeZone() + config->defaultLocation.getDST()) / 24 );
			th->update();
			spin_date->Enable( gocharamode );
			spin_time->Enable( gocharamode );
		}	
		toolbar->EnableTool( TBB_NOW, gocharamode );
	}
	OnDataChanged();
}

/*****************************************************
**
**   SbcView   ---   OnActual 
**
******************************************************/
void SbcView::OnActual( wxCommandEvent &event )
{
	if ( toolbar )
	{
		DateSpin *spin_date = (DateSpin*)toolbar->FindControl( TBS_DATE );
		TimeSpin *spin_time = (TimeSpin*)toolbar->FindControl( TBS_TIME );
		if ( spin_date && spin_time )
		{
			spin_date->setActual();
			spin_time->setActual();
		}
		OnToolbarCommand();
	}
}  

/*****************************************************
**
**   SbcView   ---   OnDataChanged 
**
******************************************************/
void SbcView::OnDataChanged()
{
	writeContents();
	gwidget->OnDataChanged();
	gwidget->Refresh();
}

/*****************************************************
**
**   SbcView   ---   getIStyle 
**
******************************************************/
int SbcView::getIStyle()
{
	//return gwidget->getIStyle();
	return 0;
}

/*****************************************************
**
**   SbcView   ---   dispatchCommand 
**
******************************************************/
bool SbcView::dispatchCommand( const int &command )
{
	wxString st;
  switch( command )
	{
		case CHILD_EXPORT:
			gwidget->doExport();
		break;
		// TODO
		/* do this one day ... menus are not there
  	case CHILD_SELECT_ALL:
			//textctrl->SetSelection( -1, -1 );
		break;
  	case CHILD_COPY:
			//textctrl->doCopy();
		break;
		case CHILD_SAVETEXT:
			doSaveText();
			//updateTextWindow( st );
			//textctrl->doExport( st );
		break;
		*/
	}
	return MView::dispatchCommand( command );
}

/*****************************************************
**
**   SbcView   ---   doSaveText
**
******************************************************/
void SbcView::doSaveText()
{
  wxString filename, s;
  wxString filetypes = _( "Text Files (*.txt)|*.txt|Html Files (*.html)|*.html| All Files|*.*" );
  
  wxFileDialog dialog( this, _("Save Text" ), config->defExportPath, wxT( "out.txt" ), filetypes,
    wxSAVE|wxOVERWRITE_PROMPT, wxDefaultPosition );

  if ( dialog.ShowModal() == wxID_OK )
  {
		Writer *w = WriterFactory().getWriter( dialog.GetFilterIndex() == 0 ? WRITER_TEXT : WRITER_HTML );
    filename = dialog.GetDirectory() + wxFileName::GetPathSeparator() + dialog.GetFilename();
		wxFile file;
		file.Create( filename, true );
		w->beginWriting();
		updateTextWindow( w );
		w->endWriting();
		file.Write(  w->getContents());
    config->defExportPath = dialog.GetDirectory();
		delete w;
  }
}

/*****************************************************
**
**   SbcView   ---   updateTextWindow 
**
******************************************************/
void SbcView::updateTextWindow( Writer *writer )
{
	switch( textmode )
	{
		case 1:
			updateTextWindowAfflictions( writer );
		break;
		default:
			updateTextWindowPositions( writer );
		break;
	}
}

/*****************************************************
**
**   SbcView   ---   writeContents 
**
******************************************************/
void SbcView::writeContents()
{
	Writer *writer = theText->getWriter();
	writer->beginWriting();
	updateTextWindow( writer );
	writer->endWriting();
	theText->setContents( writer->getContents());
}

/*****************************************************
**
**   SbcView   ---   updateTextWindowAfflictions 
**
******************************************************/
void SbcView::updateTextWindowAfflictions( Writer *writer )
{
	Lang *lang = Lang::get();
	int i, j, k, l, diff, star, rnak, nak, nak_index;
	const int special_nak[9] = { 0, 9, 15, 17, 18, 22, 24, 25, 26 };
	const wxChar *special_nak_name[9] = { _( "Janma" ), _( "Karmaksha" ), _( "Sanghatika" ),
		_( "Samudayika" ), _( "Adhana" ), _( "Vinasana" ), _( "Jati" ), _( "Desa" ), _( "Abhisheka" ) };
	int vedas[4];
	wxString s1;

	Horoscope *hh = ( gocharamode ? th : doc );
	int nakmode = config->iNakshatraMode28;

	writer->writeHeader1( _( "Afflictions" ));
	star = getNakshatra( doc->getObjectLength( IMOON ), nakmode );
	for( i = ISUN; i <= IKETU; i++ )
	{
		nak = getNakshatra( hh->getObjectLength( i ), nakmode );
		s1.Printf( wxT( "%s (%s)" ), lang->getObjectName(i, TLARGE).c_str(),
			lang->getNakshatraName( nak, nakmode, TMEDIUM ).c_str());
		writer->writeHeader3( s1 );
		//writer->writeLine( s1 );
		vedas[0] = nak;
		nak_index = nak;
		if ( nakmode == N27 && nak > ABHIJIT ) nak_index++;
		vedas[1] = nak_aspects[nak_index][1];
		vedas[2] = nak_aspects[nak_index][2];
		vedas[3] = nak_aspects[nak_index][3];
		if ( nakmode == N27 )
		{
			for( j = 1; j < 4; j++ )
			{
				if ( vedas[j] == ABHIJIT ) vedas[j] = -1;
				else if ( vedas[j] > ABHIJIT ) vedas[j]--;
			}
		}
		for( j = 0; j < 4; j++ )
		{
			if ( vedas[j] == -1 ) continue;

			// affliction of special nakshatras
			if ( nakmode == N27 )
			{
				for( l = 0; l < 9; l++ )
				{
					diff = (int)a_red( star + special_nak[l] - vedas[j], 27 );
					if ( diff == 0 )
					{
						s1.Printf( wxT( "%6s: %s" ), lang->getNakshatraName( vedas[j], nakmode, TMEDIUM ).c_str(), special_nak_name[l] );
						writer->writeLine( s1 );
					}
				}
			}

			// Radix Nakshatra afflictions
			for( k = ISUN; k <= IKETU; k++ )
			{
				rnak = getNakshatra( doc->getObjectLength( k ), nakmode );
				if ( rnak != vedas[j] ) continue;
				if ( ! gocharamode && i == k ) continue;
				s1.Printf( wxT( "%6s: %s" ), lang->getNakshatraName( vedas[j], nakmode, TMEDIUM ).c_str(),
					lang->getObjectName(k, TLARGE).c_str());
				writer->writeLine( s1 );
			}
		}
	}
}

/*****************************************************
**
**   SbcView   ---   updateTextWindowPositions 
**
******************************************************/
void SbcView::updateTextWindowPositions( Writer *writer )
{
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	int i;
	double len;
	int nak_mode = ( config->iNakshatraMode28 ? N28 : N27 );

	Horoscope *hh = ( gocharamode == 1 ? th : doc );
	if ( gocharamode == 0 ) writer->writeHeader1( _( "Birth Chart" ));
	else writer->writeHeader1( _( "Gochara" ));

	Table table( 3, 11 );
	table.setHeader( 0,  _( "Planet" ));
	table.setHeader( 1,  _( "Length" ));
	table.setHeader( 2,  _( "Nakshatra" ));
	int line = 1;
	for( i = ISUN; i <= IASCENDANT; i++ )
	{
		table.setEntry( 0, line,  lang->getObjectName( i, TSHORT ));
		len =  hh->getObjectLength(i);
		table.setEntry( 1, line,  formatter->getPosFormatted( len, hh->isObjectRetrograde(i)));
		table.setEntry( 2, line,  lang->getNakshatraName( getNakshatra( len, nak_mode ), nak_mode, TSHORT ) );
		line++;
	}
	writer->writeTable( table ); 
}

/**************************************************************
***
**   SbcView   ---   getWindowLabel
***
***************************************************************/
const wxChar *SbcView::getWindowLabel( const bool shortname )
{
	return shortname ? _( "Sbc" ) : _( "Sarvatobhadra" );
}


/*****************************************************
**
**   SbcWidget   ---   Constructor 
**
******************************************************/
SbcWidget::SbcWidget( wxWindow *parent, Horoscope* h1, Horoscope *h2, wxWindowID id, const wxPoint& pos,
	const wxSize& size )
 : ExportableScrollWidget( parent, id, pos, size )
{
	ColorProvider cp;
	bgcolor = *cp.getBgColor();
	fgcolor = *cp.getFgColor();

	SetMinSize( wxSize( 150, 150 ));
	writer = new SbcWriter( h1, h2 );
	writer->bgcolor = *cp.getBgColor();
	writer->fgcolor = *cp.getFgColor();
	writer->pcolor = *cp.getFg2Color();
	writer->mcolor = *cp.getMarkedColor();
	writer->a1color = *cp.getBeneficColor();
	writer->a2color = *cp.getMaleficColor();
	SetBackgroundColour( bgcolor );
	SetForegroundColour( fgcolor );
}

/*****************************************************
**
**   SbcWidget   ---   Destructor 
**
******************************************************/
SbcWidget::~SbcWidget()
{
	delete writer;
}

/*****************************************************
**
**   SbcWidget   ---   OnMouseLeft 
**
******************************************************/
void SbcWidget::OnMouseLeft( wxMouseEvent& event)
{
	int x = event.m_x;
	int y = event.m_y;
	if ( writer->getFieldIndex( x, y ))
	{
		writer->toggleMark( x, y );
		//writer->markField( x, y );
		Refresh();
	}
}

/*****************************************************
**
**   SbcWidget   ---   OnDataChanged 
**
******************************************************/
void SbcWidget::OnDataChanged()
{
	writer->writeNakshatras();
}


/*****************************************************
**
**   SbcWidget   ---   doPaint 
**
******************************************************/
void SbcWidget::doPaint( wxDC *p )
{
	wxCoord xx, yy;
	p->GetSize( &xx, &yy  );

	wxCoord w, h;
	p->GetSize( &w, &h  );
	p->SetFont( *FontProvider().getGraphicFont());

	// flood fill
	p->SetBrush( wxBrush( bgcolor, wxSOLID ));
	p->SetPen( wxPen( bgcolor, 1, wxSOLID ));
	p->DrawRectangle( 0, 0, w, h );

	writer->doPaint( p, wxRect( 0, 0, xx, yy ));
}

/**************************************************************
***
**   ViewFactory   ---   createSbcView
***
***************************************************************/
MView *ViewFactory::createSbcView( MViewInfo info )
{
	return new SbcView( info );
}

/*****************************************************
**
**   SbcWriter   ---   Constructor 
**
******************************************************/
SbcWriter::SbcWriter( Horoscope* h1, Horoscope *h2 )
{
	h = h1;
	th = h2;
	int x, y;

	gochara = false;
	for( int i = 0; i < 9; i++ )
	{
		for( int j = 0; j < 9; j++ )
		{
			mark[i][j] = 0;
		}
	}
	getNakshatraIndex( getNakshatra( h->getObjectLength( IMOON ), N28 ), x, y ) ;
	mark[x][y] = 1;
	getRasiIndex( getRasi( h->getObjectLength( IASCENDANT )), x, y );
	mark[x][y] = 1;
	getWeekdayIndex( h->getDataSet()->getDate()->getWeekDay(), x, y );
	mark[x][y] = 1;
}

/*****************************************************
**
**   SbcWriter   ---   OnMouseLeft 
**
******************************************************/
void SbcWriter::markField( const int &x1, const int &y1 )
{
	wxCoord x = x1;
	wxCoord y = y1;
	if ( getFieldIndex( x, y ))
	{
		mark[x][y] = ! mark[x][y];
		//Refresh();
		//RefreshRect( getFieldCoord( x, y ));
	}
}

/*****************************************************
**
**   SbcWriter   ---   writeNakshatras 
**
******************************************************/
void SbcWriter::writeNakshatras()
{
  int i, j;
  Lang *lang = Lang::get();
	Horoscope *horoscope = ( gochara ? th : h );
 
  for( i = 0; i < 28; i++ )
	{
		contents[i].nb = 0;
		contents[i].lines = 0;
		contents[i].s.Clear();
		contents[i].planets.clear();
	}

  int imax = gochara ? IKETU : IASCENDANT;
  for( i = ISUN; i <= imax; i++ )
  {
		if ( config->iNakshatraMode28 )
		{
			j = getNakshatra( horoscope->getObjectLength( i ), N28 );
		}
		else
		{
			j = getNakshatra( horoscope->getObjectLength( i ), N27 );
			if ( j >= ABHIJIT ) j++;
		}
    if ( contents[j].nb != 0 ) contents[j].s << wxT( " " );

		// Caution: reference counting not okay, so copy string
		wxString a = lang->getObjectName(i, TSHORT);
    contents[j].s << a;
    contents[j].nb++;
    contents[j].planets.push_back( i );
	}
}

/*****************************************************
**
**   SbcWriter   ---   paintChart 
**
******************************************************/
void SbcWriter::paintChart()
{
	Lang *lang = Lang::get();
	int i, j;
	wxString s;
	FontProvider fp;

	// mark special fields
	dc->SetPen( wxPen( mcolor, 1, wxSOLID ));
	dc->SetBrush( wxBrush( mcolor, wxSOLID ));
	for( i = 0; i < 9; i++ )
	{
		for( j = 0; j < 9; j++ )
		{
			if ( mark[i][j] ) dc->DrawRectangle( getFieldCoord( i, j ));
		}
	}
	dc->SetPen( wxPen( fgcolor, 1, wxSOLID ));
	dc->SetBrush( wxBrush( bgcolor, wxTRANSPARENT ));

	// draw grid
  dc->DrawRectangle( (int)(x0-xAussen), (int)(y0-yAussen), (int)(2* xAussen), (int)(2*yAussen) );
  dc->DrawRectangle( (int)(x0-xAussen - 2), (int)(y0-yAussen - 2), (int)(2* xAussen + 4), (int)(2*yAussen + 4) );
	for( i = 1; i <= 8; i++ )
	{
		dc->DrawLine( (int)(x0-xAussen + i*xstep), (int)(y0-yAussen), (int)(x0-xAussen + i*xstep), (int)(y0+yAussen  ));
		dc->DrawLine( (int)(x0-xAussen), (int)(y0-yAussen + i*ystep), (int)(x0+xAussen), (int)(y0-yAussen+i*ystep ));
	}

	drawAspects();

	// Draw Nakshatra Names
	dc->SetTextForeground( fgcolor );
	for( i = 0; i < 28; i++ )
	{
		drawTextFormatted( getNakshatraCoord( i ), lang->getNakshatraName( i, N28, TSHORT ),
			Align::HCenter + Align::Bottom );
	}
	// Static text portions
	drawTextFormatted( getFieldCoord( 0, 0 ), sanskrit_ee, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 8, 0 ), sanskrit_a, Align::HCenter + Align::Bottom );

	// Line 2
	drawTextFormatted( getFieldCoord( 1, 1 ), sanskrit_ri, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 1 ), sanskrit_g, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 1 ), sanskrit_s, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 1 ), sanskrit_d, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 1 ), sanskrit_ch, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 1 ), sanskrit_l, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 1 ), sanskrit_u, Align::HCenter + Align::Bottom );

	// Line 3
	drawTextFormatted( getFieldCoord( 1, 2 ), sanskrit_kh, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 2 ), sanskrit_ai, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 2 ), lang->getSignName( AQUARIUS, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 2 ), lang->getSignName( PISCES, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 2 ), lang->getSignName( ARIES, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 2 ), sanskrit_li, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 2 ), sanskrit_a, Align::HCenter + Align::Bottom );

	//Line 4
	drawTextFormatted( getFieldCoord( 1, 3 ), sanskrit_j, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 3 ), lang->getSignName( CAPRICORNUS, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 3 ), sanskrit_ah, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 3 ), wxString( lang->getWeekdayName( 5 )).Left(3), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 3 ), sanskrit_o, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 3 ), lang->getSignName( TAURUS, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 3 ), sanskrit_v, Align::HCenter + Align::Bottom );
	
	//Line 5
	drawTextFormatted( getFieldCoord( 1, 4 ), sanskrit_bh, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 4 ), lang->getSignName( SAGITTARIUS, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 4 ), wxString( lang->getWeekdayName( 4 )).Left(3), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 4 ), wxString( lang->getWeekdayName( 6 )).Left(3), Align::HCenter + Align::Bottom );
	s = wxString( lang->getWeekdayName( 0 )).Left(3) + wxT( " " ) + wxString( lang->getWeekdayName( 2 )).Left(3);
	drawTextFormatted( getFieldCoord( 5, 4 ), s, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 4 ), lang->getSignName( GEMINI, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 4 ), sanskrit_k, Align::HCenter + Align::Bottom );

	//Line 6
	drawTextFormatted( getFieldCoord( 1, 5 ), sanskrit_y, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 5 ), lang->getSignName( SCORPIO, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 5 ), sanskrit_am, Align::HCenter + Align::Bottom );
	s = wxString( lang->getWeekdayName( 1 )).Left(3) + wxT( " " ) + wxString( lang->getWeekdayName( 3 )).Left(3);
	drawTextFormatted( getFieldCoord( 4, 5 ), s, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 5 ), sanskrit_au, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 5 ), lang->getSignName( CANCER, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 5 ), sanskrit_h, Align::HCenter + Align::Bottom );

	// Line 7
	drawTextFormatted( getFieldCoord( 1, 6 ), sanskrit_n, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 6 ), sanskrit_e, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 6 ), lang->getSignName( LIBRA, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 6 ), lang->getSignName( VIRGO, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 6 ), lang->getSignName( LEO, TMEDIUM ), Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 6 ), sanskrit_li, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 6 ), sanskrit_dd, Align::HCenter + Align::Bottom );

	// Line 8
	drawTextFormatted( getFieldCoord( 1, 7 ), sanskrit_ri, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 2, 7 ), sanskrit_t, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 3, 7 ), sanskrit_r, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 4, 7 ), sanskrit_p, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 5, 7 ), sanskrit_th, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 6, 7), sanskrit_m, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 7, 7 ), sanskrit_uu, Align::HCenter + Align::Bottom );

	drawTextFormatted( getFieldCoord( 0, 8 ), sanskrit_i, Align::HCenter + Align::Bottom );
	drawTextFormatted( getFieldCoord( 8, 8 ), sanskrit_aa, Align::HCenter + Align::Bottom );

	// Draw planets
	dc->SetTextForeground( pcolor );
	for( i = 0; i < 28; i++ )
	{
		drawTextFormatted( getNakshatraCoord( i ), contents[i].s, Align::HCenter + Align::Top );
	}
}

/*****************************************************
**
**   SbcWriter   ---   getFieldIndex 
**
******************************************************/
bool SbcWriter::getFieldIndex( int &x, int& y )
{
	if ( x < x0-xAussen ) return false;
	if ( x > x0+xAussen ) return false;
	if ( y < y0-yAussen ) return false;
	if ( y > y0+yAussen ) return false;

	x = (int)(( x - x0 + xAussen ) / xstep);
	y = (int)(( y - y0 + yAussen ) / ystep);
	return true;
}

/*****************************************************
**
**   SbcWriter   ---   getFieldCoord 
**
******************************************************/
wxRect SbcWriter::getFieldCoord( const int &x, const int &y )
{
	return wxRect( (int)(x0-xAussen+x*xstep), (int)(y0-yAussen+y*ystep), (int)xstep, (int)ystep );
}

/*****************************************************
**
**   SbcWriter   ---   getWeekdayIndex 
**
******************************************************/
void SbcWriter::getWeekdayIndex( const int &w, int &x, int &y )
{
	assert( w >= 0 && w < 7 );
	switch( w )
	{
		case 0:
		case 2: x = 5; y = 4; break;
		case 1:
		case 3: x = 4; y = 5; break;
		case 4: x = 3; y = 4; break;
		case 5: x = 4; y = 3; break;
		default: x = 4; y = 4; break;
	}
}

/*****************************************************
**
**   SbcWriter   ---   getWeekdayCoord 
**
******************************************************/
wxRect SbcWriter::getWeekdayCoord( const int &w )
{
	int x, y;
	getWeekdayIndex( w, x, y );
	return getFieldCoord( x, y );
}

/*****************************************************
**
**   SbcWriter   ---   getRasiIndex 
**
******************************************************/
void SbcWriter::getRasiIndex( const int &r, int &x, int &y )
{
	assert( r >= 0 && r < 12 );
	switch( r )
	{
		case ARIES: x = 5; y = 2; break;
		case TAURUS: x = 6; y = 3; break;
		case GEMINI: x = 6; y = 4; break;
		case CANCER: x = 6; y = 5; break;
		case LEO: x = 5; y = 6; break;
		case VIRGO: x = 4; y = 6; break;
		case LIBRA: x = 3; y = 6; break;
		case SCORPIO: x = 2; y = 5; break;
		case SAGITTARIUS: x = 2; y = 4; break;
		case CAPRICORNUS: x = 2; y = 3; break;
		case AQUARIUS: x = 3; y = 2; break;
		case PISCES:
		default: x = 4; y = 2; break;
	}
}

/*****************************************************
**
**   SbcWriter   ---   getRasiCoord 
**
******************************************************/
wxRect SbcWriter::getRasiCoord( const int &r )
{
	int x, y;
	getRasiIndex( r, x, y );
	return getFieldCoord( x, y );
}

/*****************************************************
**
**   SbcWriter   ---   getNakshatraIndex 
**
******************************************************/
void SbcWriter::getNakshatraIndex( const int &i, int &x, int &y )
{
	assert( i >= 0 && i <= 27 );
	if ( i < 2 )
		{ x = i + 6; y = 0; }
	else if ( i >= 2 && i <= 8 )
		{ x = 8; y = i-1; }
	else if ( i >= 9 && i <= 15 )
		{ x = 16 - i; y = 8; }
	else if ( i >= 16 && i <= 22 )
		{ x = 0; y =  8 - ( i - 15 ); }
	else
		{ x = i - 22; y = 0; }
}

/*****************************************************
**
**   SbcWriter   ---   getNakshatraCoord 
**
******************************************************/
wxRect SbcWriter::getNakshatraCoord( const int &i )
{
	int x, y;
	assert( i >= 0 && i <= 27 );
	getNakshatraIndex( i, x, y );
	return wxRect( (int)(x0-xAussen+(x*xstep)), (int)(y0-yAussen+(y*ystep)), (int)xstep, (int)ystep );
}

/*****************************************************
**
**   SbcWriter   ---   drawAspects 
**
******************************************************/
void SbcWriter::drawAspects()
{
	int i, j, xr1, yr1, xr2, yr2, sw;
	unsigned int k;
	wxRect r1, r2;

	for( i = 0; i < 28; i++ )
	{
		contents[i].lines = 0;
	}
	for( i = 0; i < 28; i++ )
	{
		for( k = 0; k < contents[i].planets.size(); k++ )
		{
			r1 = getNakshatraCoord( i );
			xr1 = r1.x + r1.width/2;
			yr1 = r1.y + r1.height/2;
			dc->SetPen( wxPen( ( isbenefic[contents[i].planets[k]] ? a1color : a2color ), 1, wxSOLID ));

			for( j = 1; j <= 3; j++ )
			{
				r2 = getNakshatraCoord( nak_aspects[i][j] );
				xr2 = r2.x + r2.width/2;
				yr2 = r2.y + r2.height/2;

				sw = 2 * ( contents[i].lines + contents[nak_aspects[i][j]].lines);
				dc->DrawLine( xr1 + sw, yr1 + sw, xr2 + sw, yr2 + sw );
				contents[i].lines++;
				contents[nak_aspects[i][j]].lines++;
			}
		}
	}
	dc->SetPen( wxPen( fgcolor, 1, wxSOLID ));
}

/*****************************************************
**
**   SbcWriter   ---   doPaint 
**
******************************************************/
void SbcWriter::doPaint( wxDC *p, const wxRect &r )
{
	assert( p );
	dc = p;
	wxCoord w, h;
	//xmax = r.width;
	//ymax = r.height;
	dc->GetSize( &w, &h );
	xmax = w;
	ymax = h;
	x0 = r.x + xmax/2;
	y0 = r.y + ymax/2;
  xAussen = xmax * 0.475;
  yAussen = ymax * 0.475;
	xstep = xAussen / 4.5;
	ystep = yAussen / 4.5;

	dc->SetBackground( wxBrush( bgcolor, wxTRANSPARENT ));
	dc->SetBrush( wxBrush( bgcolor, wxTRANSPARENT ));
	dc->SetBackgroundMode( wxTRANSPARENT );	
	dc->SetPen( wxPen( fgcolor, 1, wxSOLID ));
	dc->SetTextBackground( bgcolor );
	dc->SetTextForeground( fgcolor );

	paintChart();
}

/*****************************************************
**
**   SbcWriter   ---   drawTextFormatted
**
******************************************************/
void SbcWriter::drawTextFormatted( const wxRect &rect, const wxString t, const int& align )
{
	DcHelper( dc ).drawTextFormatted( rect, t, align );
}

