/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           TextView.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "TextView.h"
#endif

#include "TextView.h"

#include <wx/string.h>

#include <wx/choice.h>
#include <wx/filedlg.h>
#include <wx/filename.h>
#include <wx/fontdlg.h>
#include <wx/log.h>
#include <wx/spinctrl.h>
#include <wx/statbox.h>
#include <wx/textctrl.h>

#include "Base.h"
#include "Document.h"
#include "FontProvider.h"
#include "MenuProvider.h"

#include "Ashtakavarga.h"
#include "Conf.h"
#include "Dasa.h"
#include "Eclipse.h"
#include "Hora.h"
#include "Jaimini.h"
#include "Lang.h"
#include "Nakshatra.h"
#include "TextHelper.h"
#include "ToolPanel.h"
#include "Varga.h"
#include "Writer.h"
#include "WriterFactory.h"

#include "dialogs/DialogElements.h"


extern Config *config;

IMPLEMENT_CLASS( TextViewBase, MView )
BEGIN_EVENT_TABLE( TextViewBase, MView )
END_EVENT_TABLE()

IMPLEMENT_CLASS( MainView, TextViewBase )
BEGIN_EVENT_TABLE( MainView, TextViewBase )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DefaultTextView, TextViewBase )
BEGIN_EVENT_TABLE( DefaultTextView, TextViewBase )
	EVT_CHOICE( TBS_TEXTMODE, DefaultTextView::OnChoiceCommand )
	EVT_CHOICE(	TBS_VARGA, DefaultTextView::OnChoiceCommand )
	EVT_CHOICE( TBS_DASA, DefaultTextView::OnChoiceCommand )
END_EVENT_TABLE()


IMPLEMENT_CLASS( EclipseView, TextViewBase )
BEGIN_EVENT_TABLE( EclipseView, TextViewBase )
  EVT_SPINCTRL( TBS_YEAR, EclipseView::OnSpinCommand )
  EVT_TEXT_ENTER( TBS_YEAR, EclipseView::OnChoiceCommand )
  EVT_SPINCTRL( TBS_NBYEARS, EclipseView::OnSpinCommand )
  EVT_TEXT_ENTER( TBS_NBYEARS, EclipseView::OnChoiceCommand )
  EVT_CHOICE( TBS_ECL_TYPE, EclipseView::OnChoiceCommand )
	EVT_IDLE( EclipseView::OnIdle )
END_EVENT_TABLE()

IMPLEMENT_CLASS( HoraView, TextViewBase )
BEGIN_EVENT_TABLE( HoraView, TextViewBase )
  EVT_MENU( TBB_NOW, HoraView::OnNow )
	EVT_COMMAND( TBS_DATE, COMMAND_DATE_SPIN_CHANGED, HoraView::OnDateSpin )
END_EVENT_TABLE()


/*****************************************************
**
**   TextViewBase   ---   Constructor 
**
******************************************************/
TextViewBase::TextViewBase( MViewInfo info )
 : MView( info )
{
	theText = new TextWidget(this, config->preferHtml2Text );
	widget = theText;
}

/*****************************************************
**
**   TextViewBase   ---   OnPreferVedic 
**
******************************************************/
void TextViewBase::OnPreferVedic( bool f )
{
	preferVedic = f;
	writeContents();
}

/*****************************************************
**
**   TextViewBase   ---   SetContextMenuInfo 
**
******************************************************/
void TextViewBase::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
	menuinfo.textstyle = 1;
	MView::SetContextMenuInfo( menuinfo );
}

/*****************************************************
**
**   TextViewBase   ---   dispatchCommand 
**
******************************************************/
bool TextViewBase::dispatchCommand( const int &command )
{
	if ( command == CHILD_SELECT_ALL )
	{
		 theText->selectAll();
		 return true;
	}
	else if ( command == CHILD_COPY )
	{
		theText->doCopy();
		return true;
	}
	else if ( command == CHILD_SAVETEXT )
	{
		doExport();
		return true;
	}
	else return MView::dispatchCommand( command );
}

/*****************************************************
**
**   TextViewBase   ---   doExport
**
******************************************************/
void TextViewBase::doExport()
{
  wxString filename, s;
  wxString filetypes = _( "Text Files (*.txt)|*.txt|Html Files (*.html)|*.html| All Files|*.*" );
	int htmlmode = config->preferHtml2Text;
	wxString defFilename = ( htmlmode ? wxT( "out.html" ) : wxT( "out.txt" ));
  
  wxFileDialog dialog( this, _("Save Text" ), config->defExportPath, defFilename, filetypes,
    wxSAVE|wxOVERWRITE_PROMPT, wxDefaultPosition );

  if ( dialog.ShowModal() == wxID_OK )
  {
		if ( dialog.GetFilename().Right( 4 ) == wxT( "html" )
			|| dialog.GetFilename().Right( 3 ) == wxT( "htm" ) ) htmlmode = true;
		else htmlmode = false;

		Writer *w = WriterFactory().getWriter( htmlmode ? WRITER_HTML : WRITER_TEXT );
    filename = dialog.GetDirectory() + wxFileName::GetPathSeparator() + dialog.GetFilename();
		wxFile file;
		file.Create( filename, true );
		w->beginWriting();
		write( w );
		w->endWriting();
		file.Write(  w->getContents());
    config->defExportPath = dialog.GetDirectory();
		delete w;
  }
}

/*****************************************************
**
**   TextViewBase   ---   OnDataChanged 
**
******************************************************/
void TextViewBase::OnDataChanged() { writeContents(); }

/*****************************************************
**
**   TextViewBase   ---   writeContents 
**
******************************************************/
void TextViewBase::writeContents()
{
	Writer *writer = theText->getWriter();
	writer->beginWriting();
	write( writer );
	writer->endWriting();
	theText->setContents( writer->getContents());
}

/*****************************************************
**
**   DefaultTextView   ---   Constructor 
**
******************************************************/
DefaultTextView::DefaultTextView( MViewInfo info )
 : TextViewBase( info )
{
	supportEWToggle = false;
	if ( viewinfo.mode != -1 ) mode = viewinfo.mode;
	else mode = config->textWindowPage;
	varga = dasa = 0;
	initToolItems();
	OnToolbarCommand();
}

/**************************************************************
***
**   DefaultTextView   ---   initToolItems
***
***************************************************************/
void DefaultTextView::initToolItems()
{
	if( toolbar )
	{
		wxChoice *choice_viewmode = (wxChoice*)toolbar->FindControl( TBS_TEXTMODE );
		if( choice_viewmode ) choice_viewmode->SetSelection( mode );

		wxChoice *choice_varga = (wxChoice*)toolbar->FindControl( TBS_VARGA );
		if( choice_varga )
		{
			choice_varga->SetSelection( varga );
			choice_varga->Enable( mode == TASHTAKAVARGA || mode == TJAIMINI );
		}

		wxChoice *choice_dasa = (wxChoice*)toolbar->FindControl( TBS_DASA );
		if( choice_dasa )
		{
			choice_dasa->SetSelection( dasa );
			choice_dasa->Enable( mode == TDASA || mode == TDASALONG );
		}
	}
}

/**************************************************************
***
**   DefaultTextView   ---   write
***
***************************************************************/
void DefaultTextView::write( Writer *writer )
{
  VargaView view( doc, varga );
  VargaExpert vexpert( doc );
  NakshatraExpert nexpert( &view );
  AshtakavargaExpert aexpert( &view );
  JaiminiExpert jexpert( &view );
  Session *session = Session::get();
	TextHelper th;
    
  switch( mode )
  {
    case TBASE: th.writeBaseData( doc, writer ); break;
    case TINDIAN: doc->writeVedicPlanets( writer ); break;
    case TWESTERN: doc->writeWesternPlanets( writer ); break;
    case TNAKSHATRA: nexpert.write( writer ); break;
    case TASPECTS: doc->writeAspects( writer ); break;
    case TASHTAKAVARGA: aexpert.update(); aexpert.write( writer ); break;
    case TKP: doc->writeKp( writer, dasa ); break;
    case TBHAVA: doc->writeBhavas( writer, true ); break;
    case TASTRONOMICAL: doc->writeAstronomicalData( writer ); break;
    case TVARGA: vexpert.write( writer ); break;
    case TDASA:
      assert( dasa < session->getNoOfDasaExperts());
      session->getDasaExpert( dasa )->write( writer, doc, varga );
    break;
    case TDASALONG:
      assert( dasa < session->getNoOfDasaExperts());
      session->getDasaExpert( dasa )->writeMore( writer, doc, varga );
    break;
    case TJAIMINI: jexpert.update(); jexpert.write( writer ); break;
    default:
			th.writeBaseData( doc, writer );
      //assert( false );
    break;
  }
}

/**************************************************************
***
**   DefaultTextView   ---   getWindowLabel
***
***************************************************************/
const wxChar *DefaultTextView::getWindowLabel( const bool shortname )
{
	return shortname ? _( "Text" ) : _( "Text Window" );
}

/*****************************************************
**
**   DefaultTextView   ---   OnToolbarCommand 
**
******************************************************/
void DefaultTextView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxChoice *choice_viewmode = (wxChoice*)toolbar->FindControl( TBS_TEXTMODE );
		if( choice_viewmode )
		{
			mode = choice_viewmode->GetSelection();
			config->textWindowPage = mode;
		}

		wxChoice *choice_varga = (wxChoice*)toolbar->FindControl( TBS_VARGA );
		if( choice_varga )
		{
			varga = choice_varga->GetSelection();
			choice_varga->Enable( mode == TASHTAKAVARGA || mode == TJAIMINI );
		}

		wxChoice *choice_dasa = (wxChoice*)toolbar->FindControl( TBS_DASA );
		if( choice_dasa )
		{
			dasa = choice_dasa->GetSelection();
			choice_dasa->Enable( mode == TDASA || mode == TDASALONG || mode == TKP );
		}
	}
	writeContents();
}

/**************************************************************
***
**   ViewFactory   ---   createTextView
***
***************************************************************/
MView *ViewFactory::createTextView( MViewInfo info )
{
	return new DefaultTextView( info  );
}


/**************************************************************
***
**   MainView   ---   Constructor
***
***************************************************************/
MainView::MainView( MViewInfo info )
 : TextViewBase( info )
{
  supportEWToggle = true;
  istyle = config->iMainChildStyle;
  wstyle = config->wMainChildStyle;
  writeContents();
}

/**************************************************************
***
**   MainView   ---   Destructor
***
***************************************************************/
MainView::~MainView()
{
}

/**************************************************************
***
**   MainView   ---   dispatchCommand
***
***************************************************************/
bool MainView::dispatchCommand( const int &command )
{
  switch( command )
  {
    case CHILD_MAIN_SHOW_NAVAMSA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_NAVAMSA );
    break;
    case CHILD_MAIN_SHOW_KARAKA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_KARAKA );
    break;
    case CHILD_MAIN_SHOW_SHASTIAMSA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_SHASTIAMSA );
    break;
    case CHILD_MAIN_SHOW_NAKSHATRA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_NAKSHATRA );
    break;
    case CHILD_MAIN_SHOW_DASAVARGA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_DASAVARGA );
    break;
    case CHILD_MAIN_SHOW_BHAVA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_BHAVA );
    break;
    case CHILD_MAIN_SHOW_KP:
      toggleSingleStyle( IMAIN_CHILD_SHOW_KP );
    break;
    case CHILD_MAIN_SHOW_ASHTAKA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_ASHTAKA );
    break;
    case CHILD_MAIN_SHOW_PADA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_PADA );
    break;
    case CHILD_MAIN_SHOW_LAGNAS:
      toggleSingleStyle( IMAIN_CHILD_SHOW_LAGNAS );
    break;
    case CHILD_MAIN_SHOW_UPA:
      toggleSingleStyle( IMAIN_CHILD_SHOW_UPA );
    break;

    case CHILD_MAIN_SHOW_WHOUSES:
      toggleSingleStyle( WMAIN_CHILD_SHOW_HOUSES );
    break;
    case CHILD_MAIN_SHOW_IHOUSES:
      toggleSingleStyle( IMAIN_CHILD_SHOW_HOUSES );
    break;
    case CHILD_MAIN_SHOW_HOUSEPOS:
      toggleSingleStyle( WMAIN_CHILD_SHOW_HOUSEPOS );
    break;
    case CHILD_MAIN_SHOW_URANIAN:
      toggleSingleStyle( WMAIN_CHILD_SHOW_URANIAN );
    break;

    default:
      return TextViewBase::dispatchCommand( command );
    break;
  }
  return true;
}

/*****************************************************
**
**   MainView   ---   toggleSingleStyle 
**
******************************************************/
void MainView::toggleSingleStyle( const int &which )
{
  if ( preferVedic ) istyle = ( istyle & which ? istyle & ~which : istyle | which );
  else wstyle = ( wstyle & which ? wstyle & ~which : wstyle | which );
  writeContents();
}

/*****************************************************
**
**   MainView   ---   SetContextMenuInfo 
**
******************************************************/
void MainView::SetContextMenuInfo( ContextMenuInfo &menuinfo )
{
  menuinfo.masterdoc = ( preferVedic ? getIStyle() : getWStyle() );
  TextViewBase::SetContextMenuInfo( menuinfo );
} 

/**************************************************************
***
**   MainView   ---   getWindowLabel
***
***************************************************************/
const wxChar *MainView::getWindowLabel( const bool shortname )
{
	return shortname ? wxT( "" ) : _( "Main Window" );
}

/**************************************************************
***
**   MainView   ---   write
***
***************************************************************/
void MainView::write( Writer *writer )
{
	TextHelper th;
	if ( preferVedic ) th.listVedicPlanets( doc, writer, istyle );
	else th.listWesternPlanets( doc, writer, wstyle );
}

/**************************************************************
***
**   ViewFactory   ---   createMainView
***
***************************************************************/
MView *ViewFactory::createMainView( MViewInfo info )
{
	info.showtoolbar = false;
	MView *view = new MainView( info );
	return view;
}

/*****************************************************
**
**   EclipseView   ---   Constructor 
**
******************************************************/
EclipseView::EclipseView( MViewInfo info )
 : TextViewBase( info )
{
	supportEWToggle = true;
  double jd = JDate().getJD();
  int day, month;
  Formatter *formatter = Formatter::get();
  formatter->getDateIntsFromJD( jd, day, month, startyear );
	years = 3;
	type = 0;

	initToolItems();
	updateView = true;
}

/**************************************************************
***
**   EclipseView   ---   initToolItems
***
***************************************************************/
void EclipseView::initToolItems()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_start = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if( spin_start ) spin_start->SetValue( startyear );

		wxSpinCtrl *spin_years = (wxSpinCtrl*)toolbar->FindControl( TBS_NBYEARS );
		if( spin_years ) spin_years->SetValue( years );

		wxChoice *choice_type = (wxChoice*)toolbar->FindControl( TBS_ECL_TYPE );
		if( choice_type ) choice_type->SetSelection( type );
	}
}

/*****************************************************
**
**   EclipseView   ---   OnToolbarCommand 
**
******************************************************/
void EclipseView::OnToolbarCommand()
{
	if ( toolbar )
	{
		wxSpinCtrl *spin_start = (wxSpinCtrl*)toolbar->FindControl( TBS_YEAR );
		if( spin_start ) startyear = spin_start->GetValue();

		wxSpinCtrl *spin_years = (wxSpinCtrl*)toolbar->FindControl( TBS_NBYEARS );
		if( spin_years ) years = spin_years->GetValue();

		wxChoice *choice_type = (wxChoice*)toolbar->FindControl( TBS_ECL_TYPE );
		if( choice_type ) type = choice_type->GetSelection();
	}
	updateView = true;
}

/**************************************************************
***
**   EclipseView   ---   OnIdle
***
***************************************************************/
void EclipseView::OnIdle( wxIdleEvent &event )
{
	if ( ! updateView ) return;
  writeContents();
	updateView = false;
}

/**************************************************************
***
**   EclipseView   ---   write
***
***************************************************************/
void EclipseView::write( Writer *writer )
{
	double startjd, endjd;
	JDate d;
	d.setDate( 1, 1, startyear, 0 );
	startjd = d.getJD();
	d.setDate( 1, 1, startyear + years, 0 );
	endjd = d.getJD();

	EclipseExpert e;
	e.update( startjd, endjd, type, 0, preferVedic );
	e.write( writer );
}

/**************************************************************
***
**   EclipseView   ---   getWindowLabel
***
***************************************************************/
const wxChar *EclipseView::getWindowLabel( const bool shortname ) { return _( "Eclipse" ); }

/**************************************************************
***
**   ViewFactory   ---   createEclipseView
***
***************************************************************/
MView *ViewFactory::createEclipseView( MViewInfo info )
{
	return new EclipseView( info );
}


/*****************************************************
**
**   HoraView   ---   Constructor 
**
******************************************************/
HoraView::HoraView( MViewInfo info )
 : TextViewBase( info )
{
	dataset = new DataSet();
	dataset->setLocation( config->defaultLocation );
	jd = dataset->getJD();
	writeContents();
}

/*****************************************************
**
**   HoraView   ---   Destructor 
**
******************************************************/
HoraView::~HoraView() { delete dataset; }

/*****************************************************
**
**   HoraView   ---   OnNow 
**
******************************************************/
void HoraView::OnNow( wxCommandEvent &event )
{
	if ( toolbar )
	{
		DateSpin *ds = (DateSpin*)toolbar->FindControl( TBS_DATE );
		if( ds )
		{
			ds->setActual();
			jd = ds->getJD();
		}
	}
	writeContents();
}

/*****************************************************
**
**   HoraView   ---   OnDateSpin 
**
******************************************************/
void HoraView::OnDateSpin( wxCommandEvent &event )
{
	if ( toolbar )
	{
		DateSpin *ds = (DateSpin*)toolbar->FindControl( TBS_DATE );
		if ( ds ) jd = ds->getJD();
	}
	writeContents();
}

/*****************************************************
**
**   HoraView   ---   write
**
******************************************************/
void HoraView::write( Writer *writer )
{
	wxString s;
	dataset->setActualDate();
	HoraExpert expert( dataset );
	expert.update( jd );
	expert.write( writer, false );
}

/**************************************************************
***
**   HoraView   ---   getWindowLabel
***
***************************************************************/
const wxChar *HoraView::getWindowLabel( const bool shortname ) { return _( "Hora" ); }

/**************************************************************
***
**   ViewFactory   ---   createHoraView
***
***************************************************************/
MView *ViewFactory::createHoraView( MViewInfo info )
{
	return new HoraView( info );
}




