/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           VedicGraphic.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "VedicGraphic.h"
#endif

#include "VedicGraphic.h"

#include <wx/dc.h>

#include "Base.h"
#include "func.h"
#include "Conf.h"
#include "Horoscope.h"
#include "Jaimini.h"
#include "Lang.h"
#include "Painter.h"
#include "Varga.h"

enum { LONG_FORMAT, SHORT_FORMAT };

extern Config *config;

/*****************************************************
**
**   VedicGraphic   ---   Constructor 
**
******************************************************/
VedicGraphic::VedicGraphic( Painter *p )
{
	painter = p;
}

/*****************************************************
**
**   VedicGraphic   ---   paint 
**
******************************************************/
void VedicGraphic::paint( Horoscope *h, int s, int d, int x0, int y0, int xmax, int ymax )
{
	paint( h, style, division, wxRect( x0, y0, xmax, ymax ));
}

/*****************************************************
**
**   VedicGraphic   ---   paint 
**
******************************************************/
void VedicGraphic::paint( Horoscope *h, const int &s, const int &d, const wxRect &r )
{
	horoscope = h;
	division = d;
	style = s;
	if ( style & IGRAPHIC_STYLE_EAST_INDIAN ) paintEast( r );
	else if ( style & IGRAPHIC_STYLE_NORTH_INDIAN ) paintNorth( r );
	else paintSouth( r );
}

/*****************************************************
**
**   VedicGraphic   ---   drawRasiText 
**
******************************************************/
void VedicGraphic::drawRasiText( const double &x, const double &y, const double &w, const double &h,
	const int &r, const int& align )
{
	drawRasiText( wxRect( (int)x, (int)y, (int)w, (int)h ), r, align );
}

/*****************************************************
**
**   VedicGraphic   ---   drawRasiText 
**
******************************************************/
void VedicGraphic::drawRasiText( const wxRect &rect, const int &r, const int& align )
{
	unsigned int i;
	int rcount = 0;
	double w, h, height = 0;
	int xmitte, ymitte, currenty, xleft = 0;

	xmitte = 2 * rect.x + rect.width;
	xmitte /= 2;
	// count number of lines
	for( i = 0; i < rs.size(); i++ )
	{
		if ( rs[i].rasi == r )
		{
			rcount++;
			painter->getTextExtent( rs[i].long_name, &w, &h );
			height = Max( h, height );
		}
	}
	if ( ! height ) return; // nothing to do
	ymitte = 2 * rect.y + rect.height;
	ymitte /= 2;
	if ( align & Align::VCenter ) currenty = ymitte - rcount * (int)height /2;
	else if ( align & Align::Top ) currenty = rect.y;
	else currenty = rect.y + rect.height - rcount * (int)height;

	for( i = 0; i < rs.size(); i++ )
	{
		if ( rs[i].rasi == r )
		{
			painter->getTextExtent( rs[i].long_name, &w, &h );
			if ( align & Align::HCenter ) xleft = xmitte - (int)w/2;
			else if ( align & Align::Left ) xleft = rect.x;
			else if ( align & Align::Right ) xleft = rect.x + rect.width - (int)w;
			painter->drawTextFormatted( xleft, currenty, w, h, rs[i].long_name, Align::Left+Align::Top );
			currenty += (int)height;
		}
	}
}

/*****************************************************
**
**   VedicGraphic   ---   writeContent 
**
******************************************************/
void VedicGraphic::writeContent( int s, bool printAs )
{
	int pcount[12], i, rasi, mystyle, zeile[12];
	unsigned int ii;
	wxString lname, sname;

	Lang *lang = Lang::get();

	rs.clear();
  for( i = 0; i < 12; i++ )
	{
		pcount[i] = zeile[i] = 0;
		content[i] = wxT( "" );
	}

	VargaView view( horoscope, division );
	for( i = ISUN; i <= ( printAs ? IASCENDANT : IKETU ); i++ )
	{
		rasi = view.getRasi( i );
		pcount[rasi]++;
		lname = lang->getObjectName(i, TLARGE );
		sname = lang->getObjectName(i, TMEDIUM );
		if (( i >= IMARS ) && ( i <= ISATURN ) && ( style & IGRAPHIC_STYLE_SHOW_RETRO ) && ( view.isRetrograde( i )))
		{
			lname += wxT( " " );
			lname += _( "(R)" );
			sname += _( "R" );
		}
		rs.push_back( rasi_cont( rasi, lname, sname ));
	}
	
	if ( style & IGRAPHIC_STYLE_SHOW_SPECIAL_LAGNAS )
	{
		for( i = IBHAVA_LAGNA; i <= IGHATIKA_LAGNA; i++ )
		{
			rasi = getRasi( horoscope->getObjectLength( i ));
			pcount[rasi]++;
			rs.push_back( rasi_cont( rasi, lang->getObjectName( i, TSHORT ), lang->getObjectName( i, TSHORT )));
		}
	}

	if ( style & IGRAPHIC_STYLE_SHOW_ARUDHAS )
	{
		JaiminiExpert expert( &view );
		for( i = 0; i < 12; i++ )
		{
			rasi = expert.calcPada( i );
			pcount[rasi]++;
			if ( i == 0 ) sname = _( "AL" ); 
			else if ( i == 11 ) sname = _( "UL" );
			else sname.sprintf( _( "A%d" ), i+1 );
			rs.push_back( rasi_cont( rasi, sname, sname ));
		}
	}

	for( ii = 0; ii < rs.size(); ii++ )
	{
		rasi = rs[ii].rasi;
		zeile[rasi]++;
		mystyle = pcount[rasi] < 4 ? s : SHORT_FORMAT;
		if ( content[rasi] != wxT( "" ))
		{
			if ( mystyle == LONG_FORMAT ) content[rasi] += wxT( "\r" );
			else
			{
				if ( zeile[rasi] % 3 ) content[rasi] += wxT( " " );
				else content[rasi] += wxT( "\n" );
			}
		}
		mystyle == SHORT_FORMAT ? content[rasi].append( rs[ii].short_name ) : content[rasi].append( rs[ii].long_name );
	}
}

/*****************************************************
**
**   VedicGraphic   ---   getInternalVargaString
**
******************************************************/
void VedicGraphic::getInternalVargaString( wxString &s )
{
	VargaExpert expert;
	if ( config->iChartCenter == 1 || division ==  NB_VARGAS-1 ) { s = expert.getVargaName( division ); }
	else { s.sprintf( _( "D-%d" ), expert.getVargaDivision( division )); }
}

/*****************************************************
**
**   VedicGraphic   ---   paintNorth 
**
******************************************************/
void VedicGraphic::paintNorth( const wxRect &rect )
{
  wxRect r;
	wxString s;
	int current, asc;
	int xmax, ymax, xmitte, ymitte, rd, r0, rtol, xtol;
	double w, h;
	VargaExpert expert;

	xmax = rect.GetWidth();
	ymax = rect.GetHeight();
	xmitte =  rect.GetLeft() + xmax/2;
	ymitte =  rect.GetTop() + ymax/2;
  rd = Min( xmax, ymax ) / 2;

  r0 = (int)(rd * 0.9);
	rtol = xmax/10;
	xtol = xmax / 70;

	painter->drawRectangle( xmitte-r0, ymitte-r0, 2 * r0,  2 * r0 );
	painter->drawLine( xmitte-r0, ymitte-r0, xmitte+r0, ymitte+r0 );
	painter->drawLine( xmitte-r0, ymitte+r0, xmitte+r0, ymitte-r0 );

	painter->drawLine( xmitte, ymitte+r0, xmitte+r0, ymitte );
	painter->drawLine( xmitte, ymitte+r0, xmitte-r0, ymitte );
	painter->drawLine( xmitte, ymitte-r0, xmitte-r0, ymitte );
	painter->drawLine( xmitte, ymitte-r0, xmitte+r0, ymitte );

	writeContent( SHORT_FORMAT, false );

	asc = horoscope->getVarga( IASCENDANT, division )->getRasi();

	/*********************************************************
	**  draw number of houses
	*********************************************************/
	// 1
	current = asc;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0/2, ymitte- rtol, r0, rtol ), s );

	// 2
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0, ymitte-r0/2-rtol, r0, rtol ), s );

	// 3
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0/2-rtol, ymitte-r0, rtol, r0 ), s );

	// 4
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-rtol, ymitte- r0/2, rtol, r0 ), s );

	// 5
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0/2-rtol, ymitte, rtol, r0 ), s );

	// 6
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0, ymitte+r0/2, r0, rtol ), s );

	// 7
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte-r0/2, ymitte, r0, rtol ), s );

	// 8
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte, ymitte+r0/2, r0, rtol ), s );

	// 9
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte+r0/2, ymitte, rtol, r0 ), s );

	// 10
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte, ymitte- r0/2, rtol, r0 ), s );

	// 11
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte+r0/2, ymitte-r0, rtol, r0 ), s );

	// 12
	if ( ++current == 12 ) current = 0;
	s.sprintf( wxT( "%d" ),  current + 1 );
  painter->drawSimpleText( wxRect( xmitte, ymitte-r0/2-rtol, r0, rtol ), s );

	/*********************************************************
	**  draw planets
	*********************************************************/
	// first house
	current = asc;
	drawRasiText( wxRect( xmitte-r0/2, ymitte-r0, r0, r0 ), current, Align::Center );

	// second house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0, ymitte-r0+xtol, r0, r0/2 ), current, Align::Top+Align::HCenter );

	// third house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0+xtol, ymitte-r0, r0, r0 ), current, Align::Left+Align::VCenter );

	// fourth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0, ymitte-r0/2, r0, r0 ), current, Align::Center );

	// fifth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0+xtol, ymitte, r0, r0 ), current, Align::Left+Align::VCenter );

	// sixth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0+xtol, ymitte, r0, r0 ), current, Align::Bottom+Align::HCenter );

	// seventh house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte-r0/2, ymitte, r0, r0 ), current, Align::Center );

	// eigth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte, ymitte+r0/2, r0, r0/2-xtol ), current, Align::Bottom+Align::HCenter );

	// ninth house
	if ( ++current == 12 ) current = 0;
	r = wxRect( xmitte, ymitte, r0-xtol, r0 );	
	drawRasiText( wxRect( xmitte, ymitte, r0-xtol, r0 ), current, Align::Right+Align::VCenter );

	// tenth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( r = wxRect( xmitte, ymitte-r0/2, r0, r0 ), current, Align::Center );

	// eleventh house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte, ymitte-r0, r0-xtol, r0 ), current, Align::Right+Align::VCenter );

	// twelfth house
	if ( ++current == 12 ) current = 0;
	drawRasiText( wxRect( xmitte, ymitte-r0+xtol, r0, r0/2 ), current, Align::Top+Align::HCenter );

	if ( config->iChartCenter )
	{
		getInternalVargaString( s );
		painter->getTextExtent( s, &w, &h );
		r = wxRect( (int)(xmitte-w/2 ), (int)(ymitte+r0), (int)w, (int)h );
		painter->drawTextFormatted( r, s, Align::Top+Align::HCenter );
	}
}


/*****************************************************
**
**   VedicGraphic   ---   paintSouth 
**
******************************************************/
void VedicGraphic::paintSouth( const wxRect &rect )
{
	VargaExpert expert;
  wxRect r;
	wxString s;
	double xmax = rect.GetWidth();
	double ymax = rect.GetHeight();
	double x0 = rect.GetLeft() + xmax/2;
	double y0 = rect.GetTop() + ymax/2;

// innerer Rand
  double xAussen = xmax * 0.475;
  double yAussen = ymax * 0.475;
	double atol = xAussen/80;
	painter->drawRectangle( x0-xAussen, y0-yAussen, 2* xAussen, 2*yAussen );
  painter->drawRectangle( x0-xAussen - atol, y0-yAussen - atol, 2* xAussen + 2*atol, 2*yAussen + 2 * atol );

	double xstep = xAussen / 2;
	double ystep = yAussen / 2;

  painter->drawLine( x0-xAussen + xstep, y0-yAussen, x0-xAussen + xstep, y0+yAussen  );
  painter->drawLine( x0-xAussen + 3 * xstep, y0-yAussen, x0-xAussen + 3 *xstep, y0+yAussen  );

  painter->drawLine( x0-xAussen + 2 * xstep, y0-yAussen, x0-xAussen + 2 *xstep, y0-yAussen + ystep  );
  painter->drawLine( x0-xAussen + 2 * xstep, y0+yAussen -ystep, x0-xAussen + 2 *xstep, y0+yAussen  );

  painter->drawLine( x0-xAussen, y0-yAussen + ystep, x0+xAussen, y0-yAussen + ystep  );
  painter->drawLine( x0-xAussen, y0-yAussen + 3 * ystep, x0+xAussen, y0-yAussen + 3 * ystep  );

  painter->drawLine( x0-xAussen, y0-yAussen + 2 * ystep, x0-xAussen + xstep, y0-yAussen + 2 * ystep  );
  painter->drawLine( x0+xAussen - xstep, y0-yAussen + 2 * ystep, x0+xAussen, y0-yAussen + 2 * ystep  );
	writeContent();

	// Aries
	drawRasiText( x0-xstep, y0-yAussen, xstep, ystep, 0, Align::Center );

	// Taurus
	drawRasiText( x0, y0-yAussen, xstep, ystep, 1, Align::Center );

	// Gemini
	drawRasiText( x0+xstep, y0-yAussen, xstep, ystep, 2, Align::Center );

	// Cancer
	drawRasiText( x0+xstep, y0-ystep, xstep, ystep, 3, Align::Center );

	// Leo
	drawRasiText( x0+xstep, y0, xstep, ystep, 4, Align::Center );

	// Virgo
	drawRasiText( x0+xstep, y0+ystep, xstep, ystep, 5, Align::Center );

	// Libra
	drawRasiText( x0, y0+ystep, xstep, ystep, 6, Align::Center );

	// Scorpio
	drawRasiText( x0-xstep, y0+ystep, xstep, ystep, 7, Align::Center );

	// Sagittarius
	drawRasiText( x0-xAussen, y0+ystep, xstep, ystep, 8, Align::Center );

	// Capricornus
	drawRasiText( x0-xAussen, y0, xstep, ystep, 9, Align::Center );

	// Aquarius
	drawRasiText( x0-xAussen, y0-ystep, xstep, ystep, 10, Align::Center );

	// Pisces
	drawRasiText( x0-xAussen, y0-yAussen, xstep, ystep, 11, Align::Center );

	if ( config->iChartCenter )
	{
		r = wxRect( (int)(x0-xAussen), (int)(y0-yAussen), (int)(2*xAussen), (int)(2*yAussen ));
		getInternalVargaString( s );
		painter->drawSimpleText( r, s );
	}
}

/*****************************************************
**
**   VedicGraphic   ---   paintEast 
**
******************************************************/
void VedicGraphic::paintEast( const wxRect &rect )
{
	VargaExpert expert;
  wxRect loc[13];
	wxString s;
	int xmax = rect.GetWidth();
	int ymax = rect.GetHeight();
	int x0 = rect.GetLeft() + xmax/2;
	int y0 = rect.GetTop() + ymax/2;
	int rd = Min( xmax, ymax );
	rd /= 2;
	const bool drawsigns = false;

// innerer Rand
  int rAussen = (int)(rd * 0.85);

// Aussenrand
  int rAussenrand = (int)(rd * 0.95);
	if ( ! drawsigns ) rAussen = rAussenrand;

// Innenquadrat
  int rInnen = (int)(rAussen*0.28);

//  innerer Rand fuer Schrift
  int rAussenIntern = (int)(rAussen * 0.95);

// Innenquadrat fuer Schhrift
  int rInnenExtern = (int)(rInnen * 1.15);

// Innenquadrat fuer Schrift vo innen
  int rInnenIntern = (int)(rInnen * 0.9);

  if ( drawsigns ) painter->drawRectangle( x0-rAussen, y0-rAussen, 2* rAussen, 2*rAussen );
  painter->drawRectangle( x0-rAussenrand, y0-rAussenrand, 2*rAussenrand, 2*rAussenrand );

  painter->drawLine( x0-rInnen, y0-rAussenrand, x0-rInnen, y0+rAussenrand );

  painter->drawLine( x0+rInnen, y0-rAussenrand, x0+rInnen, y0+rAussenrand );

  painter->drawLine( x0-rAussenrand, y0+rInnen, x0+rAussenrand, y0+rInnen );
  painter->drawLine( x0-rAussenrand, y0-rInnen, x0+rAussenrand, y0-rInnen );

  painter->drawLine( x0-rAussenrand, y0+rAussenrand, x0-rInnen, y0+rInnen );
  painter->drawLine( x0-rAussenrand, y0-rAussenrand, x0-rInnen, y0-rInnen );

  painter->drawLine( x0+rAussenrand, y0+rAussenrand, x0+rInnen, y0+rInnen );
  painter->drawLine( x0+rAussenrand, y0-rAussenrand, x0+rInnen, y0-rInnen );

	if ( drawsigns )
	{
		loc[0] = wxRect( x0-rInnen, y0-rAussenrand, 2*rInnen, rAussenrand - rAussen );
		loc[1] = wxRect( x0-rAussen, y0-rAussenrand, rAussen-rInnen, rAussenrand - rAussen );
		loc[2] = wxRect( x0-rAussenrand, y0-rAussen, rAussenrand-rAussen, rAussen-rInnen );

		loc[3] = wxRect( x0-rAussenrand, y0-rInnen, rAussenrand-rAussen, 2*rInnen );
		loc[4] = wxRect( x0-rAussenrand, y0+rInnen, rAussenrand-rAussen, rAussen - rInnen );
		loc[5] = wxRect( x0-rAussen, y0+rAussen, rAussen-rInnen, rAussenrand - rAussen );

		loc[6] = wxRect( x0-rInnen, y0+rAussen, 2*rInnen, rAussenrand - rAussen );
		loc[7] = wxRect( x0+rInnen, y0+rAussen, rAussen - rInnen, rAussenrand - rAussen );
		loc[8] = wxRect( x0+rAussen, y0+rInnen, rAussenrand - rAussen, rAussen-rInnen );

		loc[9] = wxRect( x0+rAussen, y0-rInnen, rAussenrand - rAussen, 2*rInnen );
		loc[10] = wxRect( x0+rAussen, y0-rAussen, rAussenrand - rAussen, rAussen - rInnen );
		loc[11] = wxRect( x0+rInnen, y0-rAussenrand, rAussen - rInnen, rAussenrand - rAussen );

		loc[12] = wxRect( -rInnen, -rInnen, 2*rInnen, 2*rInnen );
		for ( int i = 0; i <= 11; i++ ) painter->drawSimpleText( loc[i], Lang::get()->getSignName(i) );
	}
	writeContent();

	// Aries
	drawRasiText( x0-rInnen, y0-rAussenIntern, 2*rInnen, rAussenIntern-rInnenExtern, 0, Align::Center );

	// Cancer
	drawRasiText( x0-rAussenIntern, y0-rInnenIntern, rAussenIntern-rInnenExtern, 2*rInnenIntern, 3, Align::Center );

	// Libra
	drawRasiText( x0-rInnen, y0+rInnenExtern, 2*rInnen, rAussenIntern-rInnenExtern, 6, Align::Center );

	// Capricornus
	drawRasiText( x0+rInnenExtern, y0-rInnenIntern, rAussenIntern-rInnenExtern, 2*rInnenIntern, 9, Align::Center );

	// Taurus
	drawRasiText( x0-rAussen, y0-rAussenIntern, rAussen-rInnenExtern, rAussenIntern-rInnenIntern, 1,
		Align::Right+Align::Top );

	// Gemini
	drawRasiText( x0-rAussenIntern, y0-rAussenIntern, rAussenIntern-rInnen, rAussenIntern-rInnenExtern, 2,
		Align::Left+Align::Bottom );

	// Leo
	drawRasiText( x0-rAussenIntern, y0+rInnenExtern, rAussenIntern-rInnen, rAussenIntern-rInnenExtern, 4,
		Align::Left+Align::Top );

	// Virgo
	drawRasiText( x0-rAussen, y0+rInnenExtern, rAussen-rInnenExtern, rAussenIntern-rInnenExtern, 5,
		Align::Right+Align::Bottom );

	// Scorpio
	drawRasiText( x0+rInnenExtern, y0+rInnenExtern, rAussenIntern-rInnenExtern, rAussenIntern-rInnenExtern, 7,
		Align::Left+Align::Bottom );

	// Sagittarius
	drawRasiText( x0+rInnenExtern, y0+rInnenExtern, rAussenIntern-rInnenExtern, rAussenIntern-rInnenExtern, 8,
		Align::Right+Align::Top );

	// Aquarius
	drawRasiText( x0+rInnenExtern, y0-rAussenIntern, rAussenIntern-rInnenExtern, rAussenIntern-rInnenExtern, 10,
		Align::Right+Align::Bottom );

	// Pisces
	drawRasiText( x0+rInnenExtern, y0-rAussenIntern, rAussenIntern-rInnenExtern, rAussenIntern-rInnenExtern, 11,
		Align::Left+Align::Top );

	if ( config->iChartCenter )
	{
		getInternalVargaString( s );
		painter->drawSimpleText( wxRect( (int)(x0-rInnen), (int)(y0-rInnen), (int)(2*rInnen), (int)(2*rInnen) ), s );
	}

  return;
}

