/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           WesternGraphic.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "WesternGraphic.h"
#endif

#include "WesternGraphic.h"

#include <math.h>

#include <wx/dc.h>

#include "constants.h"
#include "func.h"
#include "Conf.h"
#include "Lang.h"
#include "Painter.h"
#include "SimpleHoroscope.h"

extern Config *config;

/*****************************************************
**
**   WesternGraphic   ---   Constructor
**
******************************************************/
WesternGraphic::WesternGraphic( Painter *p )
{
	painter = p;
}

/*****************************************************
**
**   WesternGraphic   ---   paint 
**
******************************************************/
void WesternGraphic::paint( PseudoHoroscope *doc, const wxRect &rect, const int &style )
{
  this->doc = doc;
	this->style = style;
	paintWestern( rect );
}

/*****************************************************
**
**   WesternGraphic   ---   paintSingleAspect 
**
******************************************************/
void WesternGraphic::paintSingleAspect( const double &len1, const double &len2, const int &type )
{
	if ( type == 0 )
	{
		painter->setLineStyle( wxDOT );
	}
	else
	{
		painter->setLineStyle( wxLONG_DASH );
	}
	double glen1 = red_deg( 180 + len1 - doc->getHouse( HOUSE1, false ));
	double x1 = r4 * cos( glen1 * PI / 180 );
	double y1 = -r4 * sin( glen1 * PI / 180 );

	double glen2 = red_deg( 180 + len2 - doc->getHouse( HOUSE1, false ));
	double x2 = r4 * cos( glen2 * PI / 180 );
	double y2 = -r4 * sin( glen2 * PI / 180 );

	painter->drawLine( xmitte+x1, ymitte+y1, xmitte+x2, ymitte+y2 );
	painter->setLineStyle( wxSOLID );
}

/*****************************************************
**
**   WesternGraphic   ---   paintAspects 
**
******************************************************/
void WesternGraphic::paintAspects()
{
	int p1, p2;
	double len1, len2, diff;
	double orbis = 5;

	for( p1 = WSUN; p1 <= WLUNARNODE; p1++ )
	{
		len1 = doc->getObjectLength( p1 );
		for( p2 = p1+1; p2 <= WLUNARNODE; p2++ )
		{
			len2 = doc->getObjectLength( p2 );
			diff = red_deg( len2 - len1 );
			if ( diff > 180 ) diff -= 180;

			if ( fabs( diff - 60 ) < orbis )
			{
				paintSingleAspect( len1, len2, 0 );
			}
			if ( fabs( diff - 90 ) < orbis )
			{
				paintSingleAspect( len1, len2, 1 );
			}
			else if ( fabs( diff - 120 ) < orbis )
			{
				paintSingleAspect( len1, len2, 0 );
			}
			else if ( fabs( diff - 180 ) < orbis )
			{
				paintSingleAspect( len1, len2, 1 );
			}
		}
	}
}

/*****************************************************
**
**   WesternGraphic   ---   paintHouses 
**
******************************************************/
void WesternGraphic::paintHouses()
{
	painter->setLineStyle( wxDOT_DASH );

	double len;
	for( int i = HOUSE2; i <= HOUSE12; i++ )
	{
		if ( i == HOUSE4 || i == HOUSE7 || i == HOUSE10 ) continue;
		len = doc->getHouse( i, false );
		paintSymLine( aries + len, rh1, rh2 );
	}
	painter->setLineStyle( wxSOLID );
}

/*****************************************************
**
**   WesternGraphic   ---   getRelativeLength 
**
******************************************************/
double WesternGraphic::getRelativeLength(const int &pl )
{
  assert( pl >= WSUN && pl <= WPOSEIDON );
	return ( pl != WASCENDANT ? doc->getObjectLength( pl ) : red_deg( doc->getObjectLength( WLUNARNODE ) + 180 ));
}

/*****************************************************
**
**   WesternGraphic   ---   showRetro 
**
******************************************************/
void WesternGraphic::showRetro()
{
	int pindex;
	double x, y;
	for( int pl = WMERCURY; pl <= WASCENDANT; pl++ )
	{
		if ( pl >= WLUNARNODE && config->wLunarNodeMode == LUNAR_NODE_MEAN ) continue;
		pindex = ( pl <= WLUNARNODE ? pl : WLUNARNODE );
		if ( ! doc->isObjectRetrograde( pindex )) continue;
		getXYValues( x, y, pl );
		painter->drawSimpleText( wxRect( (int)(xmitte+x), (int)(ymitte+y+psize), (int)(psize), (int)(psize)), _( "R" ));
	}
}

/*****************************************************
**
**   WesternGraphic   ---   getPRadius 
**
******************************************************/
double WesternGraphic::getPRadius( const int &planet )
{
	assert( planet >= WSUN && planet <= WPOSEIDON );

	double ret = rp;
	double len, dist;
	double mylen = getRelativeLength( planet );
	for( int i = WSUN; i < planet; i++ )
	{
		len = getRelativeLength( i );
		dist = ::fabs( mylen - len );
		if ( dist > 300 ) dist = fabs( dist - 360 );
		if ( dist < 10 )
		{
			ret -= Min( 1.5 * psize, 1.5 * psize * ( 10 - dist)/5 );
		}
	}
	return ret;
}

/*****************************************************
**
**   WesternGraphic   ---   paintDummies 
**
******************************************************/
void WesternGraphic::paintDummies()
{
	int pl;
	double len, glen, x, y;
	for( pl = WSUN; pl <= WASCENDANT; pl++ )
		paintSymLine( aries + getRelativeLength( pl ), r6, r7 );

	if ( style & WGRAPHIC_STYLE_SHOW_URANIAN )
	{
		for( pl = WCUPIDO; pl <= WPOSEIDON; pl++ )
			paintSymLine( aries + getRelativeLength( pl ), r6, r7 );
	}

	wxColor bgColor = painter->getBrushColor();
	wxColor fgColor = painter->getPenColor();
	painter->setBrushColor( bgColor );
	painter->setPenColor( bgColor );
	double mysize = psize * 1.3;
	for( pl = WSUN; pl <= WASCENDANT; pl++ )
	{
		len = getRelativeLength( pl );
		glen = red_deg( 180 + len - doc->getHouse( HOUSE1, false ));
		double r = getPRadius( pl );
		x = r * cos( glen * PI / 180 );
		y = -r * sin( glen * PI / 180 );
		painter->drawEllipse( xmitte+x - mysize, ymitte+y - mysize, 2 * mysize, 2 * mysize );
	}
	if ( style & WGRAPHIC_STYLE_SHOW_URANIAN )
		for( pl = WCUPIDO; pl <= WPOSEIDON; pl++ )
		{
			len = getRelativeLength( pl );
			glen = red_deg( 180 + len - doc->getHouse( HOUSE1, false ));
			double r = getPRadius( pl );
			x = r * cos( glen * PI / 180 );
			y = -r * sin( glen * PI / 180 );
			painter->drawEllipse( xmitte+x - mysize, ymitte+y - mysize, 2 * mysize, 2 * mysize );
		}
	painter->setTransparentBrush();
	painter->setPenColor( fgColor );
}

/*****************************************************
**
**   WesternGraphic   ---   getXYValues 
**
******************************************************/
void WesternGraphic::getXYValues( double &x, double &y, const int &planet )
{
	double len = getRelativeLength( planet );
	double glen = red_deg( 180 + len - doc->getHouse( HOUSE1, false ));
	double r = getPRadius( planet );
	x = r * cos( glen * PI / 180 );
	y = -r * sin( glen * PI / 180 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintSun 
**
******************************************************/
void WesternGraphic::paintSun( )
{
	double x, y;
	double pmid = psize/15;
	getXYValues( x, y, WSUN );
	painter->drawEllipse( xmitte+x - psize, ymitte+y - psize, 2 * psize, 2 * psize );
	painter->drawEllipse( xmitte+x - pmid, ymitte+y - pmid, 2*pmid, 2*pmid );
}

/*****************************************************
**
**   WesternGraphic   ---   paintMoon 
**
******************************************************/
void WesternGraphic::paintMoon( )
{
	double x, y;
	getXYValues( x, y, WMOON );
	painter->drawArc( xmitte +x - 2*psize, ymitte +y - psize, 3 * psize, 2 * psize, -90, 90 );
	painter->drawArc( xmitte + x - 1.8*psize, ymitte + y - psize, 2 * psize, 2 * psize, -90, 90 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintMercury 
**
******************************************************/
void WesternGraphic::paintMercury( )
{
	double x, y;
	getXYValues( x, y, WMERCURY );
	painter->drawEllipse( xmitte + x - psize /2, ymitte + y - psize, psize, psize );
	painter->drawLine( xmitte + x, ymitte + y + psize, xmitte+x, ymitte + y );
	painter->drawLine( xmitte + x - psize/2, ymitte + y + psize/2, xmitte + x + psize/2, ymitte + y + psize/2 );
	painter->drawArc( xmitte + x - psize/2, ymitte + y - 2 * psize, psize, psize, -180, 0 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintVenus 
**
******************************************************/
void WesternGraphic::paintVenus( )
{
	double x, y;
	getXYValues( x, y, WVENUS );
	painter->drawEllipse( xmitte+x - psize /2, ymitte+y - psize, psize, psize );
	painter->drawLine( xmitte+x, ymitte+y + psize, xmitte+x, ymitte+y );
	painter->drawLine( xmitte+x - psize/2, ymitte+y + psize/2, xmitte+x + psize/2, ymitte+y + psize/2 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintMars 
**
******************************************************/
void WesternGraphic::paintMars( )
{
	double x, y;
	getXYValues( x, y, WMARS );
	double xmitteM = xmitte + psize/5;
	double ymitteM = ymitte - psize/5;
	painter->drawEllipse( xmitteM+x - psize, ymitteM+y + psize, psize, - psize );
	painter->drawLine( xmitteM+x-.707*psize/4, ymitteM+y+.707*psize/4, xmitteM+x + psize/2, ymitteM+y - psize/2 );
	painter->drawLine( xmitteM+x+ psize/2, ymitteM+y- psize/2, xmitteM+x-psize/2, ymitteM+y - psize/2 );
	painter->drawLine( xmitteM+x+ psize/2, ymitteM+y- psize/2, xmitteM+x+ psize/2, ymitteM+y+psize/2 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintJupiter 
**
******************************************************/
void WesternGraphic::paintJupiter( )
{
	double x, y;
	getXYValues( x, y, WJUPITER );
	painter->drawLine( xmitte+x + psize/2, ymitte+y - psize, xmitte+x + psize/2, ymitte+y + psize );
	painter->drawLine( xmitte+x - .5 * psize, ymitte+y + psize/2, xmitte+x + psize, ymitte+y + psize/2 );
	painter->drawArc( xmitte+x - 1.2 * psize, ymitte+y - psize, 1.5 * psize, psize * 3/2, -90, 90 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintSaturn 
**
******************************************************/
void WesternGraphic::paintSaturn( )
{
	double x, y;
	getXYValues( x, y, WSATURN );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x, ymitte+y );
	painter->drawLine( xmitte+x - psize/2, ymitte+y - psize/2, xmitte+x + psize/2, ymitte+y - psize/2 );
	painter->drawArc( xmitte+x - psize, ymitte+y, 2 * psize, psize, -160, 90 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintUranus 
**
******************************************************/
void WesternGraphic::paintUranus( )
{
	double x, y;
	double pmid = psize/15;
	getXYValues( x, y, WURANUS );
	painter->drawEllipse( xmitte+x - psize/2, ymitte+y, psize, psize );
	painter->drawLine( xmitte+x, ymitte+y, xmitte+x, ymitte+y - psize );
	painter->drawEllipse( xmitte+x - pmid, ymitte+y +psize/2- pmid, 2*pmid, 2*pmid );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x + psize/2, ymitte+y );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x - psize/2, ymitte+y );
}

/*****************************************************
**
**   WesternGraphic   ---   paintNeptune 
**
******************************************************/
void WesternGraphic::paintNeptune()
{
	double x, y;
	getXYValues( x, y, WNEPTUNE );
	painter->drawArc( xmitte+x - .75 * psize, ymitte+y - 2 * psize, 1.5 * psize, 2 * psize, -180, 0 );
	painter->drawLine( xmitte+x, ymitte+y + psize, xmitte+x, ymitte+y - psize );
	painter->drawLine( xmitte+x - psize/2, ymitte+y + psize/2, xmitte+x + psize/2, ymitte+y + psize/2 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintPluto 
**
******************************************************/
void WesternGraphic::paintPluto( )
{
	double x, y;
	getXYValues( x, y, WPLUTO );
	painter->drawArc( xmitte+x - .75*psize, ymitte+y - 2 * psize, 1.5 * psize, 2 * psize, -180, 0 );
	painter->drawEllipse( xmitte+x - psize /2, ymitte+y - 1.2 * psize, psize, psize );
	painter->drawLine( xmitte+x, ymitte+y + psize, xmitte+x, ymitte+y );
	painter->drawLine( xmitte+x - psize/2, ymitte+y + psize/2, xmitte+x + psize/2, ymitte+y + psize/2 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintNNode 
**
******************************************************/
void WesternGraphic::paintNNode( )
{
	double x, y;
	getXYValues( x, y, WLUNARNODE );
	painter->drawArc( xmitte+x - .75*psize, ymitte+y - psize, 1.5 * psize, 2 * psize, -40, 0 );
	painter->drawArc( xmitte+x - .75*psize, ymitte+y - psize, 1.5 * psize, 2 * psize, 0, 220 );
	painter->drawEllipse( xmitte+x - .75*psize, ymitte+y + .5*psize, .5 * psize, .5 * psize );
	painter->drawEllipse( xmitte+x + .25*psize, ymitte+y + .5*psize, .5 * psize, .5 * psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintSNode 
**
******************************************************/
void WesternGraphic::paintSNode( )
{
	double x, y;
	getXYValues( x, y, WASCENDANT );
	painter->drawArc( xmitte+x - .75*psize, ymitte+y - psize, 1.5 * psize, 2 * psize, -220, 40 );
	painter->drawEllipse( xmitte+x - .75*psize, ymitte+y - psize, .5 * psize, .5 * psize );
	painter->drawEllipse( xmitte+x + .25*psize, ymitte+y - psize, .5 * psize, .5 * psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintCupido 
**
******************************************************/
void WesternGraphic::paintCupido( )
{
	double x, y;
	getXYValues( x, y, WCUPIDO );
	painter->drawLine( xmitte+x + psize/2, ymitte+y, xmitte+x + psize/2, ymitte+y + 1.5 * psize );
	painter->drawLine( xmitte+x - .5 * psize, ymitte+y + psize/2, xmitte+x + psize, ymitte+y + psize/2 );
	painter->drawLine( xmitte+x + .25 * psize, ymitte+y + .75 *psize, xmitte+x + .75 * psize, ymitte+y + .75 * psize );
	painter->drawArc( xmitte+x - 1.2 * psize, ymitte+y - psize, 1.5 * psize, psize * 3/2, -90, 90 );
	painter->drawEllipse( xmitte+x + .25 * psize, ymitte+y - .5 * psize, psize/2, psize/2 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintHades 
**
******************************************************/
void WesternGraphic::paintHades( )
{
	double x, y;
	getXYValues( x, y, WHADES );
	//painter->drawArc( xmitte +x - psize, ymitte +y - psize, 2 * psize, 1.5 * psize, -45, -315 ); // -315 = 45
	painter->drawArc( xmitte +x - psize, ymitte +y - psize, 2 * psize, 1.5 * psize, 45, 180 );
	painter->drawArc( xmitte +x - psize, ymitte +y - psize, 2 * psize, 1.5 * psize, 180, 315 );
	painter->drawLine( xmitte + x, ymitte + y - psize/2, xmitte+x, ymitte + y + psize );
	painter->drawLine( xmitte + x - .5 * psize, ymitte + y - .25 * psize, xmitte+x + .5 * psize, ymitte + y - .25 * psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintZeus 
**
******************************************************/
void WesternGraphic::paintZeus( )
{
	double x, y;
	getXYValues( x, y, WZEUS );
	painter->drawLine( xmitte+x, ymitte+y + psize, xmitte+x, ymitte+y - psize );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x + psize/2, ymitte+y );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x - psize/2, ymitte+y );
	painter->drawLine( xmitte + x - psize/2, ymitte + y + .25 * psize, xmitte + x + psize/2, ymitte + y + .75 * psize );
	painter->drawLine( xmitte + x + psize/2, ymitte + y + .25 * psize, xmitte + x - psize/2, ymitte + y + .75 * psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintKronos 
**
******************************************************/
void WesternGraphic::paintKronos( )
{
	double x, y;
	getXYValues( x, y, WKRONOS );
	//painter->drawArc( xmitte +x - .5 * psize, ymitte +y - psize, psize, .5 * psize, 180, 0 );
	painter->drawArc( xmitte +x - .5 * psize, ymitte +y - psize, psize, .5 * psize, 0, 180 );
	painter->drawLine( xmitte + x, ymitte + y - .75 *  psize, xmitte+x, ymitte + y + psize );
	painter->drawLine( xmitte + x - .5 * psize, ymitte + y - .25 * psize, xmitte+x + .5 * psize, ymitte + y - .25 * psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintApollon 
**
******************************************************/
void WesternGraphic::paintApollon( )
{
	double x, y;
	getXYValues( x, y, WAPOLLON );
	painter->drawLine( xmitte+x + psize/2, ymitte+y - psize, xmitte+x + psize/2, ymitte+y + psize );
	painter->drawLine( xmitte+x + .75 * psize, ymitte+y - psize, xmitte+x + .75 * psize, ymitte+y + psize );
	painter->drawLine( xmitte+x - .5 * psize, ymitte+y + psize/2, xmitte+x + psize, ymitte+y + psize/2 );
	painter->drawArc( xmitte+x - 1.2 * psize, ymitte+y - psize, 1.5 * psize, psize * 3/2, -90, 90 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintVulkanus 
**
******************************************************/
void WesternGraphic::paintVulkanus( )
{
	double x, y;
	getXYValues( x, y, WVULKANUS );
	painter->drawLine( xmitte+x, ymitte+y + psize/2, xmitte+x, ymitte+y - psize );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x + psize/2, ymitte+y );
	painter->drawLine( xmitte+x, ymitte+y - psize, xmitte+x - psize/2, ymitte+y );
	painter->drawLine( xmitte+x, ymitte+y + psize/2, xmitte+x + psize/2, ymitte+y + psize );
	painter->drawLine( xmitte+x, ymitte+y + psize/2, xmitte+x - psize/2, ymitte+y + psize );
	painter->drawLine( xmitte+x - psize/2, ymitte+y + psize, xmitte+x + psize/2, ymitte+y + psize );
}

/*****************************************************
**
**   WesternGraphic   ---   paintAdmetos 
**
******************************************************/
void WesternGraphic::paintAdmetos( )
{
	double x, y;
	getXYValues( x, y, WADMETOS );
	painter->drawEllipse( xmitte + x - psize /2, ymitte + y - psize, psize, psize );
	painter->drawLine( xmitte + x, ymitte + y + psize, xmitte+x, ymitte + y - psize );
	painter->drawLine( xmitte + x - psize/2, ymitte + y - psize/2, xmitte + x + psize/2, ymitte + y - psize/2 );
	painter->drawArc( xmitte + x - psize/2, ymitte + y - 2 * psize, psize, psize, -180, 0 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintPoseidon 
**
******************************************************/
void WesternGraphic::paintPoseidon( )
{
	double x, y;
	getXYValues( x, y, WPOSEIDON );
	//painter->drawArc( xmitte + x - psize/2, ymitte + y - 1.25 * psize, psize, psize, 0, -180 );
	//painter->drawArc( xmitte + x - psize/2, ymitte + y + .25 * psize, psize, psize, 180, 0 );
	painter->drawArc( xmitte + x - psize/2, ymitte + y - 1.25 * psize, psize, psize, -180, 0 );
	painter->drawArc( xmitte + x - psize/2, ymitte + y + .25 * psize, psize, psize, 0, 180 );
	painter->drawLine( xmitte + x, ymitte + y + psize, xmitte+x, ymitte + y - psize );
	painter->drawLine( xmitte + x - .5 * psize, ymitte + y, xmitte+x + .5 * psize, ymitte + y );
}

/*****************************************************
**
**   WesternGraphic   ---   paintArrow 
**
******************************************************/
void WesternGraphic::paintArrow( const double &a, const double &r )
{
	double c = cos( a * PI / 180 );
	double x = r * c;
	double s = sin( a * PI / 180 );
	double y = r * s;

	painter->drawLine( xmitte-x, ymitte+y, xmitte+x, ymitte-y );
	double rtemp = r * .97;
	double ra = 1;
	double x0, y0;
	x0 = cos( (a - ra) * PI / 180 ) * rtemp;
	y0 = -sin( (a - ra) * PI / 180 ) * rtemp;
	painter->drawLine( xmitte+x, ymitte-y, xmitte+x0, ymitte+y0 );
	x0 = cos( (a + ra) * PI / 180 ) * rtemp;
	y0 = -sin( (a + ra) * PI / 180 ) * rtemp;
	painter->drawLine( xmitte+x, ymitte-y, xmitte+x0, ymitte+y0 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintSymLine 
**
******************************************************/
void WesternGraphic::paintSymLine( const double &a, const double &rmin, const double &rmax )
{
	double c = cos( a * PI / 180 );
	double x0 = rmin * c;
	double x1 = rmax * c;
	double s = sin( a * PI / 180 );
	double y0 = rmin * s;
	double y1 = rmax * s;

	painter->drawLine( xmitte+x0, ymitte-y0, xmitte+x1, ymitte-y1 );
}

/*****************************************************
**
**   WesternGraphic   ---   paintWestern 
**
******************************************************/
void WesternGraphic::paintWestern( const wxRect &rect )
{
	double w, h;
	double rmax = .5* Min( rect.width, rect.height );
	xmitte = rect.x + rmax;
	ymitte = rect.y + rmax;

	r0 = rmax * .98;
	r1 = rmax * .89;  // Aussenrand
	r2 = rmax * .79;  // aussen 5-Gradzahlen
	r3 = rmax * .77;  // Grenze 5-1 Gradzahlen
	r4 = rmax * .75;  // 1-Gradzahlen innen
	r5 = rmax * .4;   // Innenkreis
	rp = rmax * .67;  // Radius fuer Planetenposition

	r6 = rmax * .73;  // Innenrand des Markierungsstrichs
	r7 = rmax * .77;  // Aussenrand des Markierungsstrichs
	rh1 = r0;         // Haeuserstriche aussen
	rh2 = rmax * .3;  // Haeuserstriche innen

	painter->drawEllipse( xmitte-r1, ymitte-r1, 2 * r1, 2 * r1 );
	painter->drawEllipse( xmitte-r2, ymitte-r2, 2 * r2, 2 * r2 );
	painter->drawEllipse( xmitte-r3, ymitte-r3, 2 * r3, 2 * r3 );
	painter->drawEllipse( xmitte-r4, ymitte-r4, 2 * r4, 2 * r4 );
	painter->drawEllipse( xmitte-r5, ymitte-r5, 2 * r5, 2 * r5 );

	double asc = doc->getHouse( HOUSE1, false );
	
	paintArrow( 180, r0 );
	paintArrow( 180 + doc->getHouse( HOUSE10, false ) - asc, r0 );
	int i;
	aries = red_deg( 180 + - asc );
	double a = aries;
	for( i = 0; i < 12; i++ )
	{
		paintSymLine( a, r1, r5 );
		a += 30;
	}

	a = aries;
	for( i = 1; i < 72; i++ )
	{
		if ( i % 6 ) paintSymLine( a + i * 5, r2, r4 );
	}
	for( i = 1; i < 360; i++ )
	{
		if ( i % 5 ) paintSymLine( a + i, r3, r4 );
	}

	a = aries + 15;
	int xm, ym;
	double midrad = ( r1 + r2 ) /2;
	const int format = TSHORT;

	if ( config->useGraphicSymbols ) painter->setSymbolFont();
	else painter->setGraphicFont();
	painter->getTextExtent( Lang::get()->getSignName(0, format ), &w, &h );
	for( i = 0; i < 12; i++ )
	{
		xm = (int)( midrad * cos( a * PI / 180 ));
		ym = -(int)( midrad * sin( a * PI / 180 ));
		painter->drawTextFormatted( wxRect( (int)(xmitte+xm - w/2), (int)(ymitte+ym - h/2), (int)w, (int)h ),
			Lang::get()->getSignName(i, format ), Align::Center );
		a += 30;
	}

	psize = rmax/20;
	if ( style & WGRAPHIC_STYLE_SHOW_HOUSES ) paintHouses();
	if ( style & WGRAPHIC_STYLE_SHOW_ASPECTS ) paintAspects();
	paintDummies();

	paintSun();
	paintMoon( );
	paintMercury( );
	paintVenus( );
	paintMars( );
	paintJupiter( );
	paintSaturn( );
	paintUranus( );
	paintNeptune( );
	paintPluto( );
	paintNNode( );
	paintSNode();
	if ( style & WGRAPHIC_STYLE_SHOW_URANIAN )
	{
		paintCupido();
		paintHades();
		paintZeus();
		paintKronos();
		paintApollon();
		paintVulkanus();
		paintAdmetos();
		paintPoseidon();
	}
	if ( style & WGRAPHIC_STYLE_SHOW_RETRO ) showRetro();
}


