/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           DialogElements.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "DialogElements.h"
#endif

#include "DialogElements.h"
#include "Base.h"

#include "constants.h"
#include "func.h"
#include "Calculator.h"
#include "Conf.h"
#include "DataSet.h"
#include "Lang.h"
#include "Session.h"

#include <math.h>

#include <wx/log.h>
#include <wx/sizer.h>
#include <wx/settings.h>
#include <wx/spinbutt.h>
#include <wx/statbox.h>
#include <wx/statline.h>
#include <wx/stattext.h>
#include <wx/textctrl.h>
#include <wx/window.h>

extern Config *config;

DEFINE_EVENT_TYPE( COMMAND_DATE_SPIN_CHANGED )
DEFINE_EVENT_TYPE( COMMAND_TIME_SPIN_CHANGED )

enum { DATE_SPIN = wxID_HIGHEST + 650, DATE_TEXT, TIME_TEXT, TIME_SPIN, DATE_LABEL };

IMPLEMENT_CLASS( MyGrid, wxGrid )
BEGIN_EVENT_TABLE( MyGrid, wxGrid )
	EVT_MOUSEWHEEL( MyGrid::OnMouseEvent ) 
	EVT_KEY_DOWN( MyGrid::OnKeyDown )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DateSpin, wxControl )
BEGIN_EVENT_TABLE( DateSpin, wxControl )
	EVT_SPIN_UP( DATE_SPIN, DateSpin::OnUp )
  EVT_SPIN_DOWN( DATE_SPIN, DateSpin::OnDown )
	EVT_TEXT_ENTER( DATE_TEXT, DateSpin::OnTextChanged )
	EVT_KILL_FOCUS( DateSpin::OnKillFocus ) 
	EVT_SET_FOCUS( DateSpin::OnSetFocus )
END_EVENT_TABLE()

IMPLEMENT_CLASS( TimeSpin, wxControl )
BEGIN_EVENT_TABLE( TimeSpin, wxControl )
  EVT_SPIN_UP( TIME_SPIN, TimeSpin::OnUp )
  EVT_SPIN_DOWN( TIME_SPIN, TimeSpin::OnDown )
	EVT_TEXT_ENTER( TIME_TEXT, TimeSpin::OnTextChanged )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DateInputField, wxTextCtrl )
BEGIN_EVENT_TABLE( DateInputField, wxTextCtrl )
	EVT_KILL_FOCUS( DateInputField::OnKillFocus ) 
END_EVENT_TABLE()

IMPLEMENT_CLASS( ToolbarLabel, wxControl )
BEGIN_EVENT_TABLE( ToolbarLabel, wxControl )
	//EVT_SIZE( ToolbarLabel::OnSize )
END_EVENT_TABLE()

IMPLEMENT_CLASS( DateValidator, wxTextValidator )

/*****************************************************
**
**   MyGrid   ---   Constructor 
**
******************************************************/
MyGrid::MyGrid( wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size,
	long style, const wxString& name )
	: wxGrid( parent, id, pos, size, style, name )
{
}

/*****************************************************
**
**   MyGrid   ---   OnMouseEvent
**
******************************************************/
void MyGrid::OnMouseEvent( wxMouseEvent &event )
{
	int x, y;
	const int offset = 3;
	GetViewStart( &x, &y );
	if ( event.GetWheelRotation() < 0 ) y += offset;
	else y -= offset;
	Scroll( x, y );
}

/*****************************************************
**
**   MyGrid   ---   OnKeyDown 
**
******************************************************/
void MyGrid::OnKeyDown( wxKeyEvent &event )
{
	long keycode = event.GetKeyCode();
	if ( keycode == WXK_ESCAPE ) { GetParent()->Close();
	}
	event.Skip();
}

/*****************************************************
**
**   FilterChoice   ---   Constructor 
**
******************************************************/
FilterChoice::FilterChoice( wxWindow *parent, int id, int hsSystem, int selected )
 : wxChoice( parent, id, wxDefaultPosition, wxSize( 120, 25 ) )
{
	SetToolTip( _( "Filter" ));
	recreate( hsSystem );
	if ( selected <= GetCount()) SetSelection( selected );
}

/*****************************************************
**
**   FilterChoice   ---   recreate 
**
******************************************************/
void FilterChoice::recreate( int hsSystem )
{
	int i, node;
	Lang *lang = Lang::get();
	node = config->wLunarNodeMode == LUNAR_NODE_MEAN ? OMEANNODE : OTRUENODE;

	Clear();
	Append( _( "No Filter" ));

	if ( hsSystem )
	{ // Witte
		Append( lang->getOListObjectName( OMC, TLARGE ));
		Append( lang->getOListObjectName( OARIES, TLARGE ));
		Append( lang->getOListObjectName( OSUN, TLARGE ));
		Append( lang->getOListObjectName( OASCENDANT, TLARGE ));
		Append( lang->getOListObjectName( OMOON, TLARGE ));
		Append( lang->getOListObjectName( node, TLARGE ));
		for( i = OMERCURY; i <= OPLUTO; i++ ) Append( lang->getOListObjectName( i, TLARGE ));
		for( i = OCUPIDO; i <= OPOSEIDON; i++ ) Append( lang->getOListObjectName( i, TLARGE ));
	}
	else
	{ // Ebertin
		for( i = OSUN; i <= OPLUTO; i++ ) Append( lang->getOListObjectName( i, TLARGE ));
		Append( lang->getOListObjectName( node, TLARGE ));
		Append( lang->getOListObjectName( OASCENDANT, TLARGE ));
		Append( lang->getOListObjectName( OMC, TLARGE ));
	}
	SetSelection( 0 );
}

/*****************************************************
**
**   FilterChoice   ---   getFilter 
**
******************************************************/
int FilterChoice::getFilter()
{
  return GetSelection() -1;
}


/*****************************************************
**
**   YearLengthChoice   ---   Constructor 
**
******************************************************/
YearLengthChoice::YearLengthChoice( wxWindow *parent, int id, const bool vedic, int selected )
 : wxChoice( parent, id, wxDefaultPosition, wxSize( 120, 25 ))
{
	SetToolTip( _( "Year Length" ));
  Append( wxT( "365.25" ));
  Append( _( "Tropical (365.2422)" ));
  Append( wxT( "365" ));
  Append( wxT( "360" ));
  Append( _( "Custom" ));
	SetSelection( selected );
}

/*****************************************************
**
**   YearLengthChoice   ---   getYearLength 
**
******************************************************/
double YearLengthChoice::getYearLength()
{
	return calcYearLength( GetSelection(), vedic );
}

/*****************************************************
**
**   CircleDegreeChoice   ---   Constructor 
**
******************************************************/
CircleDegreeChoice::CircleDegreeChoice( wxWindow *parent, int id, int selected )
 : wxChoice( parent, id, wxDefaultPosition, wxSize( 80, 25 ))
{
	SetToolTip( _( "Degrees of the Circle" ));
  Append( wxT( "360" ));
  Append( wxT( "180" ));
  Append( wxT( "90" ));
  Append( wxT( "45" ));
  Append( wxT( "22.5" ));
  Append( wxT( "11.25" ));
  Append( wxT( "5.625" ));
	SetSelection( selected );
}

/*****************************************************
**
**   SortChoice   ---   Constructor 
**
******************************************************/
SortChoice::SortChoice( wxWindow *parent, int id, int selected )
 : wxChoice( parent, id, wxDefaultPosition, wxSize( 110, 25 ))
{
	SetToolTip( _( "Sort Order" ));
  Append( _( "Planets" ));
  Append( _( "Orbis" ));
  Append( _( "Orbis Absolute" ));
  Append( _( "Planets (2)" ));
  Append( _( "Aspects" ));
	SetSelection( selected );
}

/*****************************************************
**
**   TransitModeChoice   ---   Constructor 
**
******************************************************/
TransitModeChoice::TransitModeChoice( wxWindow *parent, int id, int selected )
 : wxChoice( parent, id, wxDefaultPosition, wxSize( 110, 25 ))
{
	SetToolTip( _( "Mode" ));
  Append( _( "Solar Arc" ));
  Append( _( "Transits" ));
  Append( _( "Directions" ));
  Append( _( "Constant Arc" ));
  Append( _( "Lunar Arc" ));
	SetSelection( selected );
}

/*****************************************************
**
**   DateSpin   ---   Constructor 
**
******************************************************/
DateSpin::DateSpin( wxWindow *parent, int id, double startjd, const wxPoint& pos, const wxSize& size )
 : wxControl( parent, id, pos, size )
{
	const int bspin = 15, blab = 40;
	SetToolTip( _( "Date" ));
	jd = startjd;

	wxSize size1 = wxDefaultSize;
	wxSize size2 = wxDefaultSize;
	wxSize size3 = wxDefaultSize;
	if ( size != wxDefaultSize )
	{
		size1 = size;
		size1.x -= ( bspin + blab );
		size1.y -= 8;
		size2 = size;
		size2.x = bspin;
		size2.y -= 8;
		size3.x = blab;
		size2.y -= 8;
	}

	theText = new DateInputField( this, DATE_TEXT, &date_value, wxDefaultPosition, size1 );
	theText->SetWindowStyle( wxTE_PROCESS_ENTER );

	spin = new wxSpinButton( this, DATE_SPIN, wxDefaultPosition, size2, wxSP_ARROW_KEYS);
	spin->SetRange( -20000, 5000000 );

	theLabel = new wxStaticText( this, DATE_LABEL, wxT( "Mon" ), wxDefaultPosition, size3 );

	wxBoxSizer* sizer = new wxBoxSizer( wxHORIZONTAL );
	sizer->Add( theText, 0, wxTOP|wxBOTTOM|wxEXPAND|wxALIGN_CENTER_VERTICAL, 0 );
	sizer->Add( spin, 0, wxTOP|wxBOTTOM|wxEXPAND|wxALIGN_CENTER_VERTICAL, 0 );
	sizer->Add( theLabel, 0, wxLEFT|wxALL|wxEXPAND|wxALIGN_CENTER_VERTICAL, 6 );

	SetAutoLayout(true);
	SetSizer(sizer);
	sizer->Fit(this);
	sizer->SetSizeHints(this);
	Layout();

	jd ? writeContents() : setActual();
}

/*****************************************************
**
**   DateSpin   ---   writeContents 
**
******************************************************/
void DateSpin::writeContents()
{
	Formatter *formatter = Formatter::get();
	Lang *lang = Lang::get();
	wxString s;

	double jjd = jd + ( config->defaultLocation.getTimeZone() + config->defaultLocation.getDST() ) / 24;
	date_value = formatter->getDateStringFromJD( jjd );
	theText->SetValue( date_value );
	s = lang->getWeekdayName( ((int)(jjd + 1.5 )) % 7).Left( 2 );
	theLabel->SetLabel(  s );
}

/*****************************************************
**
**   DateSpin   ---   OnUp 
**
******************************************************/
void DateSpin::OnUp( wxSpinEvent &event )
{
	jd++;
	writeContents();
	wxCommandEvent e( COMMAND_DATE_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   DateSpin   ---   OnDown 
**
******************************************************/
void DateSpin::OnDown( wxSpinEvent &event )
{
	jd--;
	writeContents();
	wxCommandEvent e( COMMAND_DATE_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   DateSpin   ---   OnSetFocus 
**
******************************************************/
void DateSpin::OnSetFocus( wxFocusEvent& )
{
	theText->SetFocus();
}

/*****************************************************
**
**   DateSpin   ---   OnKillFocus 
**
******************************************************/
void DateSpin::OnKillFocus( wxFocusEvent &event ) 
{
}

/*****************************************************
**
**   DateSpin   ---   setActual 
**
******************************************************/
void DateSpin::setActual()
{
	JDate date;
	double jjd = date.getJD();
	jd = floor( jjd + .5 );
	writeContents();
	wxCommandEvent e( COMMAND_DATE_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   DateSpin   ---   OnTextChanged 
**
******************************************************/
void DateSpin::OnTextChanged( wxCommandEvent &event )
{
	Formatter *formatter = Formatter::get();
	int day, month, year;

  if ( ! formatter->getDateIntsFromString( (const wxChar*)theText->GetValue(), day, month, year ))
	{
		setActual();
		return;
	}
	JDate date;
	date.setDate( day, month, year, 12 );
	jd = date.getJD();
	writeContents();
	wxCommandEvent e( COMMAND_DATE_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   TimeSpin   ---   Constructor 
**
******************************************************/
TimeSpin::TimeSpin( wxWindow *parent, int id, double startjd, const wxPoint& pos, const wxSize& size )
 : wxControl( parent, id, pos, size, wxTAB_TRAVERSAL )
{
	SetToolTip( _( "Time" ));
	double jjd =  ( startjd ?  startjd :
		JDate().getJD() + ( config->defaultLocation.getTimeZone() + config->defaultLocation.getDST() ) / 24);
	time = jjd - (double)(int)jjd + .5;
	time *= 24;

	wxSize size1 = wxDefaultSize;
	wxSize size2 = wxDefaultSize;
	if ( size != wxDefaultSize )
	{
		size1 = size;
		size1.x -= 15;
		//size1.y -= 8;
		size2 = size;
		size2.x = 15;
		//size2.y -= 8;
	}
	theText = new wxTextCtrl( this, TIME_TEXT, wxT( "" ), wxDefaultPosition, size1, wxTE_PROCESS_ENTER );
	spin = new wxSpinButton( this, TIME_SPIN, wxDefaultPosition, size2, wxSP_ARROW_KEYS);
	//theText = new wxTextCtrl( this, TIME_TEXT, wxT( "" ), wxDefaultPosition, wxDefaultSize, wxTE_PROCESS_ENTER );
	//spin = new wxSpinButton( this, TIME_SPIN, wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS);
	spin->SetRange( -20000, 5000000 );

	wxBoxSizer* sizer = new wxBoxSizer( wxHORIZONTAL );
	/*
	sizer->Add( theText, 0, wxTOP|wxBOTTOM|wxEXPAND|wxALIGN_CENTER_VERTICAL, 4 );
	sizer->Add( spin, 0, wxTOP|wxBOTTOM|wxEXPAND|wxALIGN_CENTER_VERTICAL, 4 );
	*/
	sizer->Add( theText, 1, wxEXPAND|wxALIGN_CENTER_VERTICAL, 0 );
	sizer->Add( spin, 0, wxEXPAND|wxALIGN_CENTER_VERTICAL, 0 );

	SetAutoLayout(true);
	SetSizer( sizer );
	sizer->Fit(this);
	sizer->SetSizeHints(this);
	Layout();
	writeContents();
}

/*****************************************************
**
**   TimeSpin   ---   writeContents 
**
******************************************************/
void TimeSpin::writeContents()
{
	Formatter *formatter = Formatter::get();
	time = a_red( time, 24 );
	theText->SetValue( formatter->getTimeFormated( time ));
}

/*****************************************************
**
**   TimeSpin   ---   DoGetBestSizDoGetBestSize
**
******************************************************/
wxSize TimeSpin::DoGetBestSize() const
{
	wxSize ret( wxControl::DoGetBestSize() );
	wxSize best(95, ret.y);
	//CacheBestSize(best);
	return best;
}

/*****************************************************
**
**   TimeSpin   ---   OnUp 
**
******************************************************/
void TimeSpin::OnUp( wxSpinEvent &event )
{
	time++;
	writeContents();
	wxCommandEvent e( COMMAND_TIME_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   TimeSpin   ---   OnDown 
**
******************************************************/
void TimeSpin::OnDown( wxSpinEvent &event )
{
	time--;
	writeContents();
	wxCommandEvent e( COMMAND_TIME_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   TimeSpin   ---   setActual 
**
******************************************************/
void TimeSpin::setActual()
{
	double jjd = JDate().getJD() + ( config->defaultLocation.getTimeZone() + config->defaultLocation.getDST() ) / 24;
	time = jjd - (double)(int)jjd + .5;
	time *= 24;

	writeContents();
	wxCommandEvent e( COMMAND_TIME_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   TimeSpin   ---   OnTextChanged 
**
******************************************************/
void TimeSpin::OnTextChanged( wxCommandEvent &event )
{
	Formatter *formatter = Formatter::get();
	int deg, minute, second;

  if ( ! formatter->getDegreeIntsFromString( (const wxChar*)theText->GetValue(), deg, minute, second ))
	{
		setActual();
		return;
	}
	time = second + 60 * ( minute + 60 * deg );
	time = a_red( time/3600, 24 );
	writeContents();
	wxCommandEvent e( COMMAND_TIME_SPIN_CHANGED, GetId() );
	wxPostEvent( GetParent(), e );
}

/*****************************************************
**
**   DateInputField   ---   Constructor 
**
******************************************************/
DateInputField::DateInputField( wxWindow *parent, int id, wxString *s, const wxPoint& pos, const wxSize& size )
	: wxTextCtrl( parent, id, wxT( "" ), pos, size )
{
	SetToolTip( _( "Date" ));
	SetValidator( DateValidator( s ));	
}

/*****************************************************
**
**   DateInputField   ---   OnKillFocus 
**
******************************************************/
void DateInputField::OnKillFocus( wxFocusEvent &event ) 
{
	//bool a = Validate();
	//else return wx
	//GetParent()->ProcessEvent( event );
	event.Skip();
}

/*****************************************************
**
**   ToolbarLabel   ---   Constructor 
**
******************************************************/
ToolbarLabel::ToolbarLabel( wxWindow *parent, wxWindowID id, wxPoint pos, wxSize size, wxString tooltip )
 : wxControl( parent, id, pos, size )
{
  wxSize sizep = wxDefaultSize;
  wxSize sizet = wxDefaultSize;
	if ( size != wxDefaultSize && 0 )
	{
		sizep = size;
		sizep.x -= 6;
		sizep.y -= 6;
		sizet = size;
		sizet.x -= 12;
		sizet.y -= 12;
	}
	wxPanel *panel = new wxPanel(this, -1, pos, sizep, wxEXPAND|wxNO_BORDER|wxTAB_TRAVERSAL);
	wxBoxSizer* sizerp = new wxBoxSizer(wxVERTICAL);
	wxBoxSizer* sizert = new wxBoxSizer(wxVERTICAL);
	sizerp->Add( panel, 1, wxALL|wxEXPAND|wxALIGN_CENTER_VERTICAL, 3 );

	theText = new wxStaticText( panel, -1, _("--"), pos, sizet );
	sizert->Add( theText, 1, wxALL|wxEXPAND|wxALIGN_CENTER_VERTICAL, 5 );
	panel->SetSizer(sizert);
	sizert->Fit(panel);
	sizert->SetSizeHints(panel);
	SetSizer(sizerp);
	Layout();
}

/*****************************************************
**
**   ToolbarLabel   ---   DoGetBestSize
**
******************************************************/
wxSize ToolbarLabel::DoGetBestSize() const
{
	wxSize ret( wxControl::DoGetBestSize() );

	wxSize best(95, ret.y);
	return best;
}

/*****************************************************
**
**   ToolbarLabel   ---   SetLabel
**
******************************************************/
void ToolbarLabel::SetLabel( wxString s )
{
	theText->SetLabel( s );
}

/*****************************************************
**
**   DateValidator   ---   Constructor 
**
******************************************************/
DateValidator::DateValidator( wxString *s )
: wxTextValidator( wxFILTER_INCLUDE_CHAR_LIST, s )
{
	wxArrayString a;
	wxString s1;
	for( int i = 0; i < 10; i++ )
	{
		s1.sprintf( wxT( "%d" ), i );
		a.Add( s1 );
	}
	a.Add( wxT( "." ));
	a.Add( wxT( "-" ));
	SetIncludes( a );
}

	/*****************************************************
**
**   DateValidator   ---   Validate 
**
******************************************************/
bool DateValidator::Validate(wxWindow* parent)
{
	int day, month, year;
	Formatter *formatter = Formatter::get();

	wxTextCtrl *ctrl = (wxTextCtrl*)GetWindow();
	if ( formatter->getDateIntsFromString( (const wxChar*)ctrl->GetValue(), day, month, year ) ) return true;

	wxString s;
	s.sprintf( wxT( "Error in %s input field" ), (const wxChar*)ctrl->GetName() );
	doMessageBox( parent, s, wxOK | wxCENTRE | wxICON_ERROR );
	return false;
}

/*****************************************************
**
**   DegreeValidator   ---   Constructor 
**
******************************************************/
DegreeValidator::DegreeValidator( wxString *s )
 : wxTextValidator( wxFILTER_INCLUDE_CHAR_LIST, s )
{
#ifdef WX_VERSION24
	const wxChar* e[11] = { wxT( "1" ), wxT( "2" ), wxT( "3" ), wxT( "4" ), wxT( "5" ), wxT( "6" ), wxT( "7" ),
		wxT( "8" ), wxT( "9" ), wxT( "0" ), wxT( ":" )};
	wxStringList lst;
	for( int i = 0; i < 12; i++ ) lst.Add( e[i] );
	SetIncludeList( lst );
#else
	wxArrayString a;
	wxString s1;
	for( int i = 0; i < 10; i++ )
	{
		s1.sprintf( wxT( "%d" ), i );
		a.Add( s1 );
	}
	a.Add( wxT( ":" ));
	SetIncludes( a );
#endif
}

/*****************************************************
**
**   DegreeValidator   ---   Validate 
**
******************************************************/
bool DegreeValidator::Validate(wxWindow* parent)
{
	int deg, min, sec;
	Formatter *formatter = Formatter::get();

	wxTextCtrl *ctrl = (wxTextCtrl*)GetWindow();
	if ( formatter->getDegreeIntsFromString( (const wxChar*)ctrl->GetValue(), deg, min, sec ) ) return true;

	wxString s;
	s.sprintf( wxT( "Error in %s input field" ), (const wxChar*)ctrl->GetName() );
	doMessageBox( parent, s, wxOK | wxCENTRE | wxICON_ERROR );
	return false;
}

/*****************************************************
**
**   AyanamsaChoice   ---   Constructor 
**
******************************************************/
AyanamsaChoice::AyanamsaChoice( wxWindow *parent, int id, int ayatype )
 : wxChoice( parent, id )
{
	Append( _( "None" ));
	Append( _( "Lahiri" ));
	Append( _( "Raman" ));
	Append( _( "Krishnamurti" ));
	Append( _( "Custom" ));
	SetSelection( ayatype );
}

/*****************************************************
**
**   AyanamsaChoice   ---   getConfigIndex 
**
******************************************************/
int AyanamsaChoice::getConfigIndex()
{
	return GetSelection();
}

/*****************************************************
**
**   DoubleValidator   ---   Constructor 
**
******************************************************/
DoubleValidator::DoubleValidator( wxString *s )
 : wxTextValidator( wxFILTER_INCLUDE_CHAR_LIST, s )
{
#ifdef WX_VERSION24
	const wxChar* e[13] = { wxT( "1" ), wxT( "2" ), wxT( "3" ), wxT( "4" ), wxT( "5" ), wxT( "6" ), wxT( "7" ),
		wxT( "8" ), wxT( "9" ), wxT( "0" ), wxT( "." ), wxT( "," ), wxT( "-"  )};
	wxStringList lst;
	for( int i = 0; i < 13; i++ ) lst.Add( e[i] );
	SetIncludeList( lst );
#else
	wxArrayString a;
	wxString s1;
	for( int i = 0; i < 10; i++ )
	{
		s1.sprintf( wxT( "%d" ), i );
		a.Add( s1 );
	}
	a.Add( wxT( "." ));
	a.Add( wxT( "," ));
	a.Add( wxT( "-" ));
	SetIncludes( a );
#endif
}




