/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           LocationDialog.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "LocationDialog.h"
#endif

#include "LocationDialog.h"

#include <math.h>
 
#include <wx/button.h>
#include <wx/choice.h>
#include <wx/choicdlg.h>
#include <wx/choicdlg.h>
#include <wx/log.h>
#include <wx/progdlg.h>
#include <wx/settings.h>
#include <wx/sizer.h>
#include <wx/statbox.h>
#include <wx/stattext.h>
#include <wx/string.h>
#include <wx/textctrl.h>
#include <wx/textfile.h>
#include <wx/tokenzr.h>

#include "LocationEditDialog.h"
#include "DialogElements.h"
#include "Base.h"
#include "IconProvider.h"
#include "FontProvider.h"

#include "func.h"
#include "constants.h"
#include "Conf.h"
#include "Lang.h"
#include "Session.h"

extern Config *config;

#define MAX_GRID_ROWS 100


enum { LD_APPLYFILTER = wxID_HIGHEST + 1, LD_FILTER,  LD_RESETFILTER, LD_COUNTRY, LD_STATE, LD_GRID,
	LD_ADD, LD_DELETE, LD_EDIT, LD_SAVE, LD_ADDCOUNTRY, LD_DELCOUNTRY };

BEGIN_EVENT_TABLE( LocationDialog, wxDialog )
	EVT_GRID_CELL_LEFT_DCLICK( LocationDialog::OnCellDClick )
	EVT_GRID_LABEL_LEFT_CLICK( LocationDialog::OnSelectLabel )
	EVT_GRID_SELECT_CELL( LocationDialog::OnCellChange )
	EVT_TEXT_ENTER( LD_FILTER, LocationDialog::OnFilterChanged )
	EVT_BUTTON( LD_APPLYFILTER, LocationDialog::OnApplyFilter )
	EVT_BUTTON( LD_RESETFILTER, LocationDialog::OnResetFilter )
	EVT_BUTTON( LD_ADD, LocationDialog::OnAddEntry )
	EVT_BUTTON( LD_DELETE, LocationDialog::OnDeleteEntry )
	EVT_BUTTON( LD_EDIT, LocationDialog::OnEditEntry )
	EVT_BUTTON( LD_SAVE, LocationDialog::OnSaveDB )
	EVT_BUTTON( LD_ADDCOUNTRY, LocationDialog::OnAddCountry )
	EVT_BUTTON( LD_DELCOUNTRY, LocationDialog::OnDelCountry )
	EVT_BUTTON( wxID_OK, LocationDialog::OnOK )
	EVT_BUTTON( wxID_CANCEL, LocationDialog::OnCancel )
	EVT_CHOICE( LD_COUNTRY, LocationDialog::OnCountryChoice )
END_EVENT_TABLE()

/*****************************************************
**
**   LocationDialog   ---   Constructor 
**
******************************************************/
LocationDialog::LocationDialog(wxWindow* parent )
  :  wxDialog(parent, -1, wxT( "Location Database" ),  wxDefaultPosition, wxDefaultSize, DEFAULT_DIALOG_STYLE )
{
		progressdialog = new wxProgressDialog( _("Location Dialog"), _( "Loading Data ..." ));

    // begin wxGlade: LocationDialog::LocationDialog
    sizer_actions_staticbox = new wxStaticBox(this, -1, _("Actions"));
    sizer_country_edit_staticbox = new wxStaticBox(this, -1, _("Country"));
    sizer_datadetails_staticbox = new wxStaticBox(this, -1, _("Data"));
    sizer_entry_staticbox = new wxStaticBox(this, -1, _("Entry"));
    sizer_filter_staticbox = new wxStaticBox(this, -1, _("Filter"));
    label_filtermatch = new wxStaticText(this, -1, _("Match"));
    filter_edit = new wxTextCtrl(this, LD_FILTER, wxT(""), wxDefaultPosition, wxDefaultSize, wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB);
    label_filtercountry = new wxStaticText(this, -1, _("Country"));
    const wxString country_choice_choices[] = {
        _("dummy"),
        _("dummy"),
        _("dummy"),
        _("dummy")
    };
    country_choice = new wxChoice(this, LD_COUNTRY, wxDefaultPosition, wxDefaultSize, 4, country_choice_choices, 0);
    applyfilterbutton = new wxButton(this, LD_APPLYFILTER, _("Apply Filter"));
    resetbutton = new wxButton(this, LD_RESETFILTER, _("Reset Filter"));
    button_addcountry = new wxButton(this, LD_ADDCOUNTRY, _("Add to List"));
    button_delcountry = new wxButton(this, LD_DELCOUNTRY, _("Delete From List"));
    locgrid = new MyGrid(this, LD_GRID);
    match_label = new wxStaticText(this, -1, _("0 Entries match"));
    static_line_match = new wxStaticLine(this, -1);
    label_showname = new wxStaticText(this, -1, _("Name"));
    label_name = new wxStaticText(this, -1, _("name"));
    label_showcountry = new wxStaticText(this, -1, _("Country"));
    label_country = new wxStaticText(this, -1, _("country"));
    label_showstate = new wxStaticText(this, -1, _("State"));
    label_state = new wxStaticText(this, -1, _("state"));
    label_showlong = new wxStaticText(this, -1, _("Longitude"));
    label_lon = new wxStaticText(this, -1, _("longitude"));
    label_showlat = new wxStaticText(this, -1, _("Latitude"));
    label_lat = new wxStaticText(this, -1, _("latitude"));
    addbutton = new wxButton(this, LD_ADD, _("New"));
    button_mod = new wxButton(this, LD_EDIT, _("Edit"));
    delbutton = new wxButton(this, LD_DELETE, _("Delete"));
    static_line_button = new wxStaticLine(this, -1);
    okbutton = new wxButton(this, wxID_OK, _("OK"));
    cacelbutton = new wxButton(this, wxID_CANCEL, _("Cancel"));
    button_save = new wxButton(this, LD_SAVE, _("Save DB"));

    set_properties();
    do_layout();
    // end wxGlade

		filename = Session::get()->getLocationFile();
		SetIcon( IconProvider::getIcon( BITMAP_WORLD ));
		moveWindow2Center( this );
		setDirty( false );
		selectedEntry = 0;
		doSetup();
		updateLocationData();
		delete progressdialog;
}


/**************************************************************
***
**
***
***************************************************************/
struct string_sort
{
  bool operator()( const wxString &s1, const wxString &s2 ) const
	  {
			return( wxStricmp( s1.c_str(), s2.c_str() ) < 0 );
		}
};

/**************************************************************
***
**   Sort 
***
***************************************************************/
struct LocationEntry_sort
{
  bool operator()( const LocationEntry &s1, const LocationEntry &s2 ) const
	{
		return( wxStrcmp( s1.name.c_str(), s2.name.c_str() ) < 0 );
	}
};

/*****************************************************
**
**   LocationDialog   ---   doSetup 
**
******************************************************/
bool LocationDialog::doSetup()
{
	wxString buf, tmp;
	int count = 0;
	bool found;
	int nbrecords = 18000;
	wxString fc, s;
	wxTextFile theFile;
	list<LocationEntry>::iterator iter;
	list<wxString>::iterator citer;


	//locgrid->SetSelectionMode( wxGrid::wxGridSelectRows );
#ifndef __WXMSW__
	locgrid->SetDefaultCellBackgroundColour( wxSystemSettings::GetColour( wxSYS_COLOUR_BACKGROUND ));
	locgrid->SetDefaultCellTextColour( wxSystemSettings::GetColour( wxSYS_COLOUR_WINDOWTEXT ));
	locgrid->SetLabelBackgroundColour( wxSystemSettings::GetColour( wxSYS_COLOUR_BTNFACE ));
	locgrid->SetLabelTextColour( wxSystemSettings::GetColour( wxSYS_COLOUR_BTNTEXT ));
#endif
	selectedEntry = 0;
	//if ( ! theFile.Open( filename, FILE_CONF_MODE ))
	if ( ! theFile.Open( filename, wxConvUTF8 ))
	{ 
		wxLogError( wxT( "Cannot open file %s" ), filename.c_str());
		return false;
	}

	entries.clear();
	countries.clear();
  buf = theFile.GetFirstLine();
	entries.push_back( LocationEntry( buf ));
  while( ! theFile.Eof())
  {
    buf = theFile.GetNextLine();
		entries.push_back( LocationEntry( buf ));
		count++;
		if ( ! ( count % 100 )) progressdialog->Update( (int)(count*100/nbrecords));
		assert( count < 50000 );
  }
	wxLogMessage( wxT( "%d location entries found" ), entries.size());
  theFile.Close();
	entries.sort( LocationEntry_sort() );

	// Generate country list
	for( iter = entries.begin(); iter != entries.end(); iter++ )
	{
		found = false;
		for( citer = countries.begin(); citer != countries.end(); citer++ )
		{
			if ( (*iter).country == (*citer) )
			{
				found = true;
				break;
			}
		}
		if ( ! found ) countries.push_back( (*iter).country );
	}
	countries.sort( string_sort() );
	wxLogMessage( wxT( "%d country entries found" ), countries.size());

	fav_countries.clear();
  fc = config->favouriteCountries;
	if ( fc.IsEmpty() ) fc = wxT( "USA;Germany;Switzerland;Austria;India" );
	wxStringTokenizer t( fc, wxT( ";" ));
	while( t.HasMoreTokens() ) { fav_countries.push_back( t.GetNextToken() ); }
	fav_countries.sort( string_sort() );
	updateCountryChoice();

	updateGrid();
	updateLocationData();
	return true;
}


/*****************************************************
**
**   LocationDialog   ---   updateCountryChoice 
**
******************************************************/
void LocationDialog::updateCountryChoice()
{
	list<wxString>::iterator citer;

	country_choice->Clear();
	country_choice->Append( _( "All" ) );
	for( citer = fav_countries.begin(); citer != fav_countries.end(); citer++ )
	{
		country_choice->Append( (*citer) );
	}
	country_choice->SetSelection( 0 );
}

/*****************************************************
**
**   LocationDialog   ---   updateGrid 
**
******************************************************/
void LocationDialog::updateGrid()
{
	wxString s;
	Formatter *formatter = Formatter::get();
	int row = 0;
	wxString locfilter = filter_edit->GetValue();
	int locfilter_len = locfilter.Len();
	list<LocationEntry>::iterator iter;

	selectedEntry = 0;
	int sel_country = country_choice->GetSelection();
	wxString selected_country( wxT( "" ));
	if ( sel_country )
	{
		selected_country = country_choice->GetString( sel_country );
	}

	locgrid->ClearGrid();
	for( iter = entries.begin(); iter != entries.end(); iter++ )
	{
		if ( locfilter_len )
		{
			if ( wxStrnicmp( locfilter.c_str(), (*iter).name.c_str(), locfilter_len ))
			{
				(*iter).row = -1;
				continue;
			}
		}
		if ( sel_country )
		{
			if ( wxStrcmp( selected_country.c_str(), (*iter).country.c_str() ))
			{
				(*iter).row = -1;
				continue;
			}
		}
		(*iter).row = row;
		if ( row < MAX_GRID_ROWS )
		{
			locgrid->SetCellValue( row, 0, (*iter).name.c_str() );
			locgrid->SetCellValue( row, 1, (*iter).country.c_str() );
			locgrid->SetCellValue( row, 2, (*iter).state.c_str() );

			s.Printf ( wxT( "%s %c" ), formatter->getDegreesFormated( fabs( (*iter).longitude ) ).c_str(),
				( (*iter).longitude >= 0 ? 'E' : 'W' ));
			locgrid->SetCellValue( row, 3, s );

			s.Printf ( wxT( "%s %c" ), formatter->getDegreesFormated( fabs( (*iter).latitude ) ).c_str(),
				( (*iter).latitude >= 0 ? 'N' : 'S' ));
			locgrid->SetCellValue( row, 4, s );

			s.Printf( wxT( "%02.1f" ), (*iter).timeZone );
			locgrid->SetCellValue( row, 5, s );
		}

		row++;
	}
	s.Printf( _( "%d Entries match" ), row );
	match_label->SetLabel( s );
	locgrid->AutoSizeRows();
	locgrid->Scroll( 0, 0 );
	locgrid->ForceRefresh();
}

/*****************************************************
**
**   LocationDialog   ---   updateLocationData 
**
******************************************************/
void LocationDialog::updateLocationData()
{
	wxString s;
	Formatter *formatter = Formatter::get();
	if ( selectedEntry )
	{
		label_name->SetLabel( selectedEntry->name.c_str() );
		label_country->SetLabel( selectedEntry->country.c_str() );
		label_state->SetLabel( selectedEntry->state.c_str() );

		s.Printf( wxT( "%s %c" ), formatter->getDegreesFormated( fabs( selectedEntry->longitude ) ).c_str(),
			( selectedEntry->longitude > 0 ? 'E' : 'W' ));
		label_lon->SetLabel( s );

		s.Printf( wxT( "%s %c" ), formatter->getDegreesFormated( fabs( selectedEntry->latitude ) ).c_str(),
			( selectedEntry->latitude > 0 ? 'N' : 'S' ));
		label_lat->SetLabel( s );
	}
	else
	{
		label_name->SetLabel( wxT( "" ));
		label_country->SetLabel( wxT( "" ));
		label_state->SetLabel( wxT( "" ));
		label_lon->SetLabel( wxT( "" ));
		label_lat->SetLabel( wxT( "" ));
	}
	button_mod->Enable( selectedEntry );
	delbutton->Enable( selectedEntry );
	okbutton->Enable( selectedEntry );
}

/*****************************************************
**
**   LocationDialog   ---   OnSelectCell 
**
******************************************************/
void LocationDialog::OnSelectCell( wxGridEvent& ev )
{
	list<LocationEntry>::iterator iter;
	selectedEntry = 0;
	if ( ! locgrid->GetNumberRows() ) return;
	for( iter = entries.begin(); iter != entries.end(); iter++ )
	{
		if ( (*iter).row == ev.GetRow() )
		{
			selectedEntry = (LocationEntry*)&(*iter);
			break;
		}
	}
	updateLocationData();
	ev.Skip();
}

/*****************************************************
**
**   LocationDialog   ---   OnSelectLabel
**
******************************************************/
void LocationDialog::OnSelectLabel( wxGridEvent& event )
{
	list<LocationEntry>::iterator iter;
	selectedEntry = 0;
	if ( ! locgrid->GetNumberRows() ) return;
	if ( event.GetCol() == -1 )
	{
		for( iter = entries.begin(); iter != entries.end(); iter++ )
		{
			if ( (*iter).row == event.GetRow() )
			{
				selectedEntry = (LocationEntry*)&(*iter);
				break;
			}
	}
	}
	locgrid->ClearSelection();
	updateLocationData();
	event.Skip();
}

/*****************************************************
**
**   LocationDialog   ---   OnCellChange
**
******************************************************/
void LocationDialog::OnCellChange( wxGridEvent& event )
{
	list<LocationEntry>::iterator iter;
	selectedEntry = 0;
	if ( ! locgrid->GetNumberRows() ) return;
	for( iter = entries.begin(); iter != entries.end(); iter++ )
	{
		if ( (*iter).row == event.GetRow() )
		{
			selectedEntry = (LocationEntry*)&(*iter);
			break;
		}
	}
	updateLocationData();
	event.Skip();
}

/*****************************************************
**
**   LocationDialog   ---   OnCellDClick
**
******************************************************/
void LocationDialog::OnCellDClick( wxGridEvent& event )
{
	if ( ! selectedEntry ) return;

	LocationEditDialog dialog( this, -1, wxT( "Edit Location" ));
	dialog.setEntry( selectedEntry );
	if ( dialog.ShowModal() == wxID_OK )
	{
		*selectedEntry = *(dialog.getEntry());
		selectedEntry = 0;
		setDirty( true );
		entries.sort( LocationEntry_sort() );
		updateLocationData();
		updateGrid();
	}
}

/*****************************************************
**
**   LocationDialog   ---   OnFilterChanged 
**
******************************************************/
void LocationDialog::OnFilterChanged( wxCommandEvent &event )
{
	selectedEntry = 0;
	updateLocationData();
	updateGrid();
}

/*****************************************************
**
**   LocationDialog   ---   OnApplyFilter 
**
******************************************************/
void LocationDialog::OnApplyFilter( wxCommandEvent &event )
{
	selectedEntry = 0;
	updateLocationData();
	updateGrid();
}

/*****************************************************
**
**   LocationDialog   ---   OnResetFilter 
**
******************************************************/
void LocationDialog::OnResetFilter( wxCommandEvent &event )
{
	selectedEntry = 0;
	updateLocationData();
	filter_edit->SetValue( wxT( "" ));
	updateGrid();
}

/*****************************************************
**
**   LocationDialog   ---   OnCountryChoice 
**
******************************************************/
void LocationDialog::OnCountryChoice( wxCommandEvent &event )
{
	selectedEntry = 0;
	updateLocationData();
	updateGrid();
}

/*****************************************************
**
**   LocationDialog   ---   OnAddEntry 
**
******************************************************/
void LocationDialog::OnAddEntry( wxCommandEvent &event )
{
	LocationEditDialog dialog( this, -1, _( "New Location" ));
	dialog.setEntry( 0 );
	if ( dialog.ShowModal() == wxID_OK )
	{
		entries.push_back( LocationEntry( dialog.getEntry()->name.c_str(), dialog.getEntry()->country.c_str(),
			dialog.getEntry()->state.c_str(), dialog.getEntry()->longitude,
			dialog.getEntry()->latitude, dialog.getEntry()->timeZone )
		);
		selectedEntry = 0;
		entries.sort( LocationEntry_sort() );
		setDirty( true );
		updateLocationData();
		updateGrid();
	}
}

/*****************************************************
**
**   LocationDialog   ---   OnDeleteEntry 
**
******************************************************/
void LocationDialog::OnDeleteEntry( wxCommandEvent &event )
{
	wxString s;
	list<LocationEntry>::iterator iter;
	Formatter *formatter = Formatter::get();
	assert( selectedEntry );


	s.Printf( wxT( "%s\n%s\n%s\n%s ?" ), _( "Do you want to delete the item" ),
		selectedEntry->name.c_str(), 
		formatter->getLongitudeFormated( selectedEntry->longitude ).c_str(),
		formatter->getLatitudeFormated( selectedEntry->latitude ).c_str());
	if ( doMessageBox( this, s, wxYES | wxNO | wxCENTRE | wxICON_QUESTION ) == wxID_YES )
	{
		for( iter = entries.begin(); iter != entries.end(); iter++ )
		{
			if ( (*iter).row == selectedEntry->row )
			{
				entries.erase( iter );
				//delete selectedEntry;
				selectedEntry = 0;
				setDirty( true );
				updateLocationData();
				updateGrid();
				return;
			}
		}
	}
}

/*****************************************************
**
**   LocationDialog   ---   OnEditEntry 
**
******************************************************/
void LocationDialog::OnEditEntry( wxCommandEvent &event )
{
	assert( selectedEntry );
	LocationEditDialog dialog( this, -1, wxT( "Edit Location" ));
	dialog.setEntry( selectedEntry );
	if ( dialog.ShowModal() == wxID_OK )
	{
		*selectedEntry = *(dialog.getEntry());
		selectedEntry = 0;
		setDirty( true );
		entries.sort( LocationEntry_sort() );
		updateLocationData();
		updateGrid();
	}
}

/*****************************************************
**
**   LocationDialog   ---   OnSaveDB 
**
******************************************************/
void LocationDialog::OnSaveDB( wxCommandEvent &event )
{
	if ( saveDB() )
	{
		setDirty( false );
	}
}

/*****************************************************
**
**   LocationDialog   ---   setDirty 
**
******************************************************/
void LocationDialog::setDirty( const bool& d )
{
	dirty = d;
	button_save->Enable( dirty );
}

/*****************************************************
**
**   LocationDialog   ---   saveDB 
**
******************************************************/
bool LocationDialog::saveDB()
{
	wxString s;
	list<LocationEntry>::iterator iter;

	if ( config->locationBackupMode > 0 )
	{
		wxString backupfile = Session::get()->getLocationBackupFile( config->locationBackupMode );
		wxCopyFile( filename, backupfile, true );
	}

  wxTextFile wfile( filename );
  if ( ! wfile.Exists() )
  {
    if ( ! wfile.Create( filename ) )
    {
      wxLogError( wxT( "Cannot create file %s" ), filename.c_str());
      return false;
    }
  }
  else
  {
    if ( ! wfile.Open( FILE_CONF_MODE ) )
    {
      wxLogError( wxT( "Cannot open file %s" ), filename.c_str());
      return false;
    }
    wfile.Clear();
  }
	//for( unsigned int i = 0; i < entries.GetCount(); i++ )
	for( iter = entries.begin(); iter != entries.end(); iter++ )
	{
	
		s.Printf( wxT( "%s;%s;%s;%f;%f;%f" ), (*iter).name.c_str(), (*iter).country.c_str(),
			(*iter).state.c_str(), (*iter).longitude, (*iter).latitude, (*iter).timeZone  );
		wfile.AddLine( s );
	}
	wfile.Write( wxTextFileType_None, FILE_CONF_MODE );
	wfile.Close();
	return true;
}

/*****************************************************
**
**   LocationDialog   ---   OnOK 
**
******************************************************/
void LocationDialog::OnOK( wxCommandEvent &event )
{
	saveFavouriteContries();
	if ( dirty ) { if ( ! saveDB()) return; }
	wxDialog::OnOK( event );
}

/*****************************************************
**
**   LocationDialog   ---   OnCancel 
**
******************************************************/
void LocationDialog::OnCancel( wxCommandEvent &event )
{
	saveFavouriteContries();
	if ( dirty )
	{
		if ( doMessageBox( this, _( "The database has changed. Do you want to save it ?" ),
			wxYES | wxNO | wxCENTRE | wxICON_QUESTION ) == wxID_YES )
		{
			if ( ! saveDB()) return;
		}
	}
	wxDialog::OnCancel( event );
}


/*****************************************************
**
**   LocationDialog   ---   saveFavouriteContries 
**
******************************************************/
void LocationDialog::saveFavouriteContries()
{
	wxString s = wxT( "" );
	list<wxString>::iterator citer;

	config->favouriteCountries = wxT( "" );
	for( citer = fav_countries.begin(); citer != fav_countries.end(); citer++ )
	{
		if ( config->favouriteCountries != wxT( "" )) config->favouriteCountries.Append( wxT( ";" ));
		config->favouriteCountries.Append( (*citer) );
	}
}

/**************************************************************
***
**
***
***************************************************************/
Location *LocationDialog::getSelectedLocation()
{
	if ( selectedEntry == 0 ) return 0;
	selectedLocation.setLocation( selectedEntry->name.c_str(), selectedEntry->longitude, selectedEntry->latitude, selectedEntry->timeZone, 0 );
	return &selectedLocation;
}

/*****************************************************
**
**   LocationDialog   ---   OnAddCountry 
**
******************************************************/
void LocationDialog::OnAddCountry( wxCommandEvent &event )
{
	wxString s[200];
	list<wxString>::iterator citer;
	list<wxString>::iterator fiter;
	int i = 0;
	bool found;

	for( citer = countries.begin(); citer != countries.end(); citer++ )
	{
		found = false;
		for( fiter = fav_countries.begin(); fiter != fav_countries.end(); fiter++ )
		{
			if ( (*citer) == (*fiter) )
			{
				found = true;
				break;
			}
		}
		if ( found == false )
		{
			s[i] = (*citer);
			i++;
		}
		if ( i >= 199 ) break;
	}
	wxSingleChoiceDialog dialog( this, _( "Select the country you want to add to the list." ), _( "Add Country" ), i, s );
	if ( dialog.ShowModal() == wxID_OK )
	{
		i = dialog.GetSelection();
		fav_countries.push_back( s[i].c_str() );
		fav_countries.sort( string_sort() );
		updateCountryChoice();
	}
}

/*****************************************************
**
**   LocationDialog   ---   OnDelCountry 
**
******************************************************/
void LocationDialog::OnDelCountry( wxCommandEvent &event )
{
	wxString s[200];
	int i = 0;
	list<wxString>::iterator citer;

	for( citer = fav_countries.begin(); citer != fav_countries.end(); citer++ )
	{
		s[i] = (*citer);
		i++;
		if ( i >= 199 ) break;
	}
	wxSingleChoiceDialog dialog( this, _( "Select the country you want to delete." ), _( "Delete Country" ), i, s );
	if ( dialog.ShowModal() == wxID_OK )
	{
		i = dialog.GetSelection();
		//for( j = 0; j < fav_countries.GetCount(); j++ )
		for( citer = fav_countries.begin(); citer != fav_countries.end(); citer++ )
		{
			if (( *citer) == s[i].c_str() )
			{
				fav_countries.erase( citer );
				updateCountryChoice();
				break;
			}
		}
	}
}

/*****************************************************
**
**   LocationDialog   ---   set_properties 
**
******************************************************/
void LocationDialog::set_properties()
{
    // begin wxGlade: LocationDialog::set_properties
    SetTitle(_("Location Database"));
    label_filtermatch->SetMinSize(wxSize(100, 16));
    filter_edit->SetMinSize(wxSize(250, 25));
    label_filtercountry->SetMinSize(wxSize(100, 16));
    country_choice->SetMinSize(wxSize(180, 25));
    country_choice->SetSelection(0);
    match_label->SetMinSize(wxSize(200, 16));
    label_showname->SetMinSize(wxSize(100, 16));
    label_name->SetMinSize(wxSize(150, 16));
    label_showcountry->SetMinSize(wxSize(100, 16));
    label_country->SetMinSize(wxSize(150, 16));
    label_showstate->SetMinSize(wxSize(100, 16));
    label_state->SetMinSize(wxSize(150, 16));
    label_showlong->SetMinSize(wxSize(100, 16));
    label_lon->SetMinSize(wxSize(150, 16));
    label_showlat->SetMinSize(wxSize(100, 16));
    label_lat->SetMinSize(wxSize(150, 16));
    // end wxGlade
	locgrid->CreateGrid(100, 6);
	locgrid->SetRowLabelSize(30);
	locgrid->SetColLabelSize(26);
	locgrid->EnableEditing(false);
	locgrid->SetColLabelValue(0, _("Name"));
	locgrid->SetColSize(0, 140);
	locgrid->SetColLabelValue(1, _("Country"));
	locgrid->SetColSize(1, 120);
	locgrid->SetColLabelValue(2, _("State"));
	locgrid->SetColSize(2, 120);
	locgrid->SetColLabelValue(3, _("Longitude"));
	locgrid->SetColSize(3, 80);
	locgrid->SetColLabelValue(4, _("Latitude"));
	locgrid->SetColSize(4, 80);
	locgrid->SetColLabelValue(5, _("TZ"));

	locgrid->SetColSize(5, 40);
	locgrid->SetMinSize(wxSize(400, 250));
	locgrid->SetDefaultCellFont( *FontProvider().getDefaultFont() );
	locgrid->SetLabelFont( *FontProvider().getHeaderFont() );
	//locgrid->SetExtraStyle( wxWS_EX_BLOCK_EVENTS );
}


/*****************************************************
**
**   LocationDialog   ---   do_layout 
**
******************************************************/
void LocationDialog::do_layout()
{
    // begin wxGlade: LocationDialog::do_layout
    wxFlexGridSizer* sizer_main = new wxFlexGridSizer(7, 1, 0, 0);
    wxFlexGridSizer* sizer_button = new wxFlexGridSizer(1, 5, 0, 0);
    wxFlexGridSizer* sizer_data = new wxFlexGridSizer(1, 2, 0, 0);
    wxStaticBoxSizer* sizer_entry = new wxStaticBoxSizer(sizer_entry_staticbox, wxVERTICAL);
    wxStaticBoxSizer* sizer_datadetails = new wxStaticBoxSizer(sizer_datadetails_staticbox, wxVERTICAL);
    wxBoxSizer* sizer_showlat = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_showlong = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_showstate = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_showcountry = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_showname = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_entrymatch = new wxBoxSizer(wxHORIZONTAL);
    wxFlexGridSizer* sizer_top = new wxFlexGridSizer(1, 3, 0, 0);
    wxStaticBoxSizer* sizer_country_edit = new wxStaticBoxSizer(sizer_country_edit_staticbox, wxVERTICAL);
    wxStaticBoxSizer* sizer_actions = new wxStaticBoxSizer(sizer_actions_staticbox, wxVERTICAL);
    wxStaticBoxSizer* sizer_filter = new wxStaticBoxSizer(sizer_filter_staticbox, wxVERTICAL);
    wxBoxSizer* sizer_country = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_match = new wxBoxSizer(wxHORIZONTAL);
    sizer_match->Add(label_filtermatch, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_match->Add(filter_edit, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_filter->Add(sizer_match, 1, wxALL|wxEXPAND, 3);
    sizer_country->Add(label_filtercountry, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_country->Add(country_choice, 0, wxADJUST_MINSIZE, 0);
    sizer_filter->Add(sizer_country, 1, wxALL|wxEXPAND, 3);
    sizer_top->Add(sizer_filter, 1, wxALL|wxEXPAND, 5);
    sizer_actions->Add(applyfilterbutton, 0, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL, 5);
    sizer_actions->Add(resetbutton, 0, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL, 5);
    sizer_top->Add(sizer_actions, 1, wxALL|wxEXPAND, 3);
    sizer_country_edit->Add(button_addcountry, 0, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL, 5);
    sizer_country_edit->Add(button_delcountry, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);
    sizer_top->Add(sizer_country_edit, 1, wxALL|wxEXPAND, 3);
    sizer_top->AddGrowableCol(0);
    sizer_main->Add(sizer_top, 1, wxALL|wxALIGN_CENTER_HORIZONTAL, 6);
    sizer_main->Add(locgrid, 1, wxEXPAND, 0);
    sizer_entrymatch->Add(match_label, 0, wxLEFT, 60);
    sizer_main->Add(sizer_entrymatch, 1, wxEXPAND, 5);
    sizer_main->Add(static_line_match, 0, wxTOP|wxBOTTOM|wxEXPAND, 5);
    sizer_showname->Add(label_showname, 0, wxADJUST_MINSIZE, 0);
    sizer_showname->Add(label_name, 0, wxADJUST_MINSIZE, 0);
    sizer_datadetails->Add(sizer_showname, 1, wxALL|wxEXPAND, 3);
    sizer_showcountry->Add(label_showcountry, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_showcountry->Add(label_country, 0, wxADJUST_MINSIZE, 0);
    sizer_datadetails->Add(sizer_showcountry, 1, wxALL|wxEXPAND, 3);
    sizer_showstate->Add(label_showstate, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_showstate->Add(label_state, 0, wxADJUST_MINSIZE, 0);
    sizer_datadetails->Add(sizer_showstate, 1, wxALL|wxEXPAND, 3);
    sizer_showlong->Add(label_showlong, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_showlong->Add(label_lon, 0, wxADJUST_MINSIZE, 0);
    sizer_datadetails->Add(sizer_showlong, 1, wxALL|wxEXPAND, 3);
    sizer_showlat->Add(label_showlat, 0, wxALIGN_CENTER_VERTICAL|wxADJUST_MINSIZE, 0);
    sizer_showlat->Add(label_lat, 0, wxADJUST_MINSIZE, 0);
    sizer_datadetails->Add(sizer_showlat, 1, wxALL|wxEXPAND, 3);
    sizer_data->Add(sizer_datadetails, 1, wxALL|wxEXPAND, 3);
    sizer_entry->Add(addbutton, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    sizer_entry->Add(button_mod, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    sizer_entry->Add(delbutton, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    sizer_data->Add(sizer_entry, 1, wxALL|wxEXPAND, 3);
    sizer_main->Add(sizer_data, 1, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    sizer_main->Add(static_line_button, 0, wxALL|wxEXPAND, 5);
    sizer_button->Add(okbutton, 0, wxALL, 3);
    sizer_button->Add(cacelbutton, 0, wxALL, 3);
    sizer_button->Add(button_save, 0, wxALL, 3);
    sizer_main->Add(sizer_button, 1, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    SetAutoLayout(true);
    SetSizer(sizer_main);
    sizer_main->Fit(this);
    sizer_main->SetSizeHints(this);
    sizer_main->AddGrowableRow(1);
    sizer_main->AddGrowableCol(0);
    Layout();
    // end wxGlade
}

