/*

	This is the source code of

   	M A I T R E Y A
    ===============

	Open source platform for Vedic and western astrology

  File           Ashtakavarga.cpp
  Release        4.1
  Author         Martin Pettau
  Copyright (C)  2003-2006 by the author

  Released under the Artistic License as published by the
  Free Software Foundation, read the file 'COPYING' for more information.

*/

#ifdef __GNUG__
	#pragma implementation "Ashtakavarga.h"
#endif

#include "Ashtakavarga.h"

#include<wx/log.h>
#include<math.h>

#include "constants.h"
#include "func.h"
#include "Conf.h"
#include "Lang.h"
#include "Session.h"
#include "Varga.h"
#include "Writer.h"

extern Config *config;

/*****************************************************
**
**   AshtakavargaExpert   ---   update 
**
******************************************************/
void AshtakavargaExpert::update()
{
	int i, j;

	for( i = 0; i < 8; i++ )
		for( j = ARIES; j <= PISCES; j++ )
			rekha[i][j] = trikona[i][j] = ekadhi[i][j] = 0;

	for( i = ARIES; i <= PISCES; i++ ) planetNumber[i] = 0;

	calcRekha();
	calcTrikonaShodana();
	calcEkadhipatyaShodana();
	calcSarva();
	calcPinda();
}

/*****************************************************
**
**   AshtakavargaExpert   ---   getPinda 
**
******************************************************/
int AshtakavargaExpert::getPinda( const int &type, const int &rasi )
{
	switch( type )
	{
		case GRAHAPINDA:
			return grahaPinda[rasi];
		break;
		case RASIPINDA:
			return rasiPinda[rasi];
		break;
		case YOGAPINDA:
			return yogaPinda[rasi];
		break;
		default:
			wxLogError( wxT( "Error  AshtakavargaExpert::getPinda, type is %d" ), type );
			exit(1);
		break;
	}
	return 0;
}

/*****************************************************
**
**   AshtakavargaExpert   ---   getItem 
**
******************************************************/
int AshtakavargaExpert::getItem( const int &type, const int &p, const int &rasi )
{
	assert(( ISUN <= p && p <= ISATURN ) || p == IASCENDANT );
	int planet = ( p <= ISATURN ? p : 7 );
	switch( type )
	{
		case REKHA:
			return rekha[planet][rasi];
		break;
		case TRIKONA:
			return trikona[planet][rasi];
		break;
		case EKADHI:
			return ekadhi[planet][rasi];
		break;
		default:
			wxLogError( wxT( "Error  AshtakavargaExpert::getItem, type is %d" ), type );
			exit(1);
		break;
	}
	return 0;
}

/*****************************************************
**
**   AshtakavargaExpert   ---   getSarva 
**
******************************************************/
int AshtakavargaExpert::getSarva( const int &type, const int &rasi )
{
	assert( rasi >= ARIES && rasi <= PISCES );
	switch( type )
	{
		case REKHA:
			return sarvaRekha[rasi];
		break;
		case TRIKONA:
			return sarvaTrikona[rasi];
		break;
		case EKADHI:
			return sarvaEkadhi[rasi];
		break;
		default:
			wxLogError( wxT( "Error  AshtakavargaExpert::getSarva, type is %d" ), type );
			exit(1);
		break;
	}
	return 0;
}

/*****************************************************
**
**   AshtakavargaExpert   ---   getPlanetSarva 
**
******************************************************/
int AshtakavargaExpert::getPlanetSarva( const int &type, const int &p )
{
	assert(( ISUN <= p && p <= ISATURN ) || p == IASCENDANT );
	int planet = ( p <= ISATURN ? p : 7 );
	switch( type )
	{
		case REKHA:
			return psarvaRekha[planet];
		break;
		case TRIKONA:
			return psarvaTrikona[planet];
		break;
		case EKADHI:
			return psarvaEkadhi[planet];
		break;
		default:
			wxLogError( wxT( "Error  AshtakavargaExpert::getPlanetSarva, type is %d" ), type );
			exit(1);
		break;
	}
	return 0;
}

/*****************************************************
**
**   AshtakavargaExpert   ---   dump 
**
******************************************************/
void AshtakavargaExpert::dump( wxString &o )
{
  VargaExpert expert;
  int d = view->getDivision();
  int i, j;
  wxString s;

  o << wxT( "AshtakaVarga: " ) << expert.getVargaName( d ) << wxT( " (D" ) << expert.getVargaDivision(d)
		<< wxT( ")" ) << Endl;
  o << wxT( "                 REKHA                                 TRIKONA SHODANA" ) << Endl;
  o << wxT( "     " );
  for( i = 0; i < 12; i++ ) o << Lang::get()->getSignName( i ) << wxT( " " );
  o << wxT( "     " );
  for( i = 0; i < 12; i++ ) o << Lang::get()->getSignName( i ) << wxT( " " );
  o << Endl;
  o << wxT( "----------------------------------------      -----------------------------------" ) << Endl;

  for( i = ISUN;  i <= IASCENDANT; i++ )
  {
    if ( i == IRAHU || i == IKETU ) continue;
    o <<  Lang::get()->getObjectName( i, TMEDIUM ) << wxT( "  " );
    for( j = ARIES; j <= PISCES; j++ )
    {
      s.Printf( wxT( "%02d "), getItem( REKHA, i, j ));
      o << s;
    }
   
    o << wxT( "     " );
    for( j = ARIES; j <= PISCES; j++ )
    {
      s.Printf( wxT( "%02d " ), getItem( TRIKONA, i, j ) );   
      o << s;
    }
    o << Endl;
  }
  o << wxT( "----------------------------------------      -----------------------------------" ) << Endl;
  o << wxT( "     " );
  for( j = ARIES; j <= PISCES; j++ )
  {
    s.Printf( wxT( "%02d " ), getSarva( REKHA, j ));
    o << s;
  }
  o << wxT( "     " );  
  for( j =ARIES; j <= PISCES; j++ )
  {
    s.Printf( wxT( "%02d " ), getSarva( TRIKONA, j ));
    o << s;
  }
  o << Endl;
  o << Endl;

  o << wxT( "         EKADHIPATYA SHODANA                        PINDA" ) << Endl;
  o << wxT( "     " );
  for( i = ARIES; i <= PISCES; i++ ) o << Lang::get()->getSignName( i ) << wxT( " " );
  o << wxT( "    Rasi   Graha  Yoga"  ) << Endl;
  o << wxT( "----------------------------------------    --------------------" ) << Endl;
  for( i = ISUN; i <= IASCENDANT; i++ )
  {
    if ( i == IRAHU || i == IKETU ) continue;
    o <<  Lang::get()->getObjectName( i, TMEDIUM ) << wxT( "  " );
    for( j = ARIES; j <= PISCES; j++ )
    {
      s.Printf( wxT( "%02d " ), getItem( EKADHI, i, j ));
      o << s;
    }
    o << wxT( "    " );
    s.Printf( wxT( "%03d    " ), getPinda( RASIPINDA, i ));
    o << s;
    s.Printf( wxT( "%03d    " ), getPinda( GRAHAPINDA, i ));
    o << s;
    s.Printf( wxT( "%03d" ), getPinda( YOGAPINDA, i ));
    o << s << Endl;
  }
  o << wxT( "----------------------------------------    --------------------" ) << Endl;
  o << wxT( "     " );
  for( j = ARIES; j <= PISCES; j++ )
  {
    s.Printf( wxT( "%02d " ), getSarva( EKADHI, j ));
    o << s;
  }
  o << Endl;
}

/*****************************************************
**
**   AshtakavargaExpert   ---   write
**
******************************************************/
void AshtakavargaExpert::write( Writer *writer, const int varga )
{
	VargaExpert expert;
	wxString s;
	int i;
	int d = view->getDivision();

	writer->writeHeader1( _( "Ashtakavarga" ));
	s << _T( "Varga" ) << wxT( ": " ) << expert.getVargaName(d) << wxT( " D" ) << expert.getVargaDivision(d);
	writer->writeLine( s );

  writer->writeHeader2( _( "Rekha" ));
  writeSingleAv( writer, REKHA );
  writer->writeHeader2( _( "Trikona Shodana" ));
  writeSingleAv( writer, TRIKONA );
  writer->writeHeader2( _( "Ekadhipatya Shodana" ));
  writeSingleAv( writer, EKADHI );

	Table table( 4, 8 );
  writer->writeHeader2( _( "Pinda" ));
  table.setHeader( 0,  _( "Planet" ));
  table.setHeader( 1,  _( "Rasi" ));
  table.setHeader( 2,  _( "Graha" ));
  table.setHeader( 3,  _( "Yoga" ));

	int line = 1;
  for( i = ISUN;  i < IASCENDANT; i++ )
  {
    if ( i == IRAHU || i == IKETU ) continue;
    table.setEntry( 0, line, Lang::get()->getObjectName( i, TMEDIUM ));
    s.Printf( wxT( "%03d" ), getPinda( RASIPINDA, i ));
		table.setEntry( 1, line, s );
    s.Printf( wxT( "%03d" ), getPinda( GRAHAPINDA, i ));
		table.setEntry( 2, line, s );
    s.Printf( wxT( "%03d" ), getPinda( YOGAPINDA, i ));
		table.setEntry( 3, line, s );
		line++;
  }
	writer->writeTable( table ); 
}

/*****************************************************
**
**   AshtakavargaExpert   ---   writeSingleAv
**
******************************************************/
void AshtakavargaExpert::writeSingleAv( Writer *writer, const int&w )
{
	int i, j, line = 1;
	wxString s1;
	Table table( 13, 10 );
	table.setHeader( 0, wxEmptyString);
	for( i = ARIES; i <= PISCES; i++ )
	{
		table.setHeader( i+1, Lang::get()->getSignName( i, TMEDIUM));
	}

  for( i = ISUN;  i <= IASCENDANT; i++ )
  {
    if ( i == IRAHU || i == IKETU ) continue;
    table.setEntry( 0, line, Lang::get()->getObjectName( i, TMEDIUM ));
    for( j = ARIES; j <= PISCES; j++ )
    {
      s1.Printf( wxT( "%02d "), getItem( w, i, j ));
			table.setEntry( j+1, line, s1 );
    }
	line++;
  } 
  table.setEntry( 0, 9, _( "Sum" ));
  for( j = ARIES; j <= PISCES; j++ )
  { 
    s1.Printf( wxT( "%02d " ), getSarva( w, j ));
		table.setEntry( j+1, 9, s1 );
  } 

	writer->writeTable( table ); 
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcRekha 
**
******************************************************/
void AshtakavargaExpert::calcRekha()
{
	wxChar REKHA_MAP[8][8][12] =
	{
		{ // Sun
			{ 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1 },
			{ 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0 },
			{ 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 1 }
		},
		{ // Moon
			{ 0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 0 },
			{ 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0 },
			{ 0, 1, 1, 0, 1, 1, 0, 0, 1, 1, 1, 0 },
			{ 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0 },
			{ 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 0 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0 }
		},
		{ // Mars
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0 },
			{ 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1 },
			{ 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1 },
			{ 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0 }
		},
		{ // Mercury
			{ 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1 },
			{ 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1 },
			{ 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1 },
			{ 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0 }
		},
		{ // Jupiter
			{ 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0 },
			{ 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0 },
			{ 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0 },
			{ 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0 },
			{ 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1, 0 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1 },
			{ 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0 }
		},
		{  // Venus
			{ 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1 },
			{ 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1 },
			{ 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0 },
			{ 0, 0, 0, 0, 1, 0, 0, 1, 1, 1, 1, 0 },
			{ 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0 },
			{ 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0 },
			{ 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0 }
		},
		{  // Saturn
			{ 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1 },
			{ 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1 },
			{ 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1 },
			{ 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1 },
			{ 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0 },
			{ 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 0 }
		},
		{  // Ascendant
			{ 0, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 1 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 1 },
			{ 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0 },
			{ 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0 },
			{ 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0 },
			{ 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 0 },
			{ 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0 }
		}
	};
	int i, j, k, house, p1, p2;

	// Varaha Mihira
	if ( config->iAshtakavargaMode )
	{
		REKHA_MAP[1][1][8]  = 0; // Moon  malefic in  9 from Moon
		REKHA_MAP[1][2][8]  = 1; // Moon  benefic in  9 from Mars
		REKHA_MAP[1][4][1]  = 0; // Moon  malefic in  2 from Jupiter
		REKHA_MAP[1][4][11] = 1; // Moon  benefic in 12 from Jupiter
		REKHA_MAP[5][2][3]  = 0; // Venus malefic in  4 from Mars
		REKHA_MAP[5][2][4]  = 1; // Venus benefic in  5 from Mars
	}
	for ( i = 0; i < 8; i++ )
	//for ( i = 0; i < 1; i++ )
	{
		i < 7 ? p1 = i : p1 = IASCENDANT;
		for ( j = 0; j < 8; j++ )
		//for ( j = 1; j < 2; j++ )
		{
			j < 7 ? p2 = j : p2 = IASCENDANT;
			for ( k = 0; k < 12; k++ )
			{
				house = red12( view->getRasi(p2) + k );
				if ( REKHA_MAP[i][j][k] ) rekha[i][house]++;
			}
		}
	}
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcSarva 
**
******************************************************/
void AshtakavargaExpert::calcSarva()
{
	int i, j;
	for( i = 0; i < 12; i++ ) sarvaRekha[i] = sarvaTrikona[i] = sarvaEkadhi[i] = 0;
	for( i = 0; i < 8; i++ ) psarvaRekha[i] = psarvaTrikona[i] = psarvaEkadhi[i] = 0;

	for( i = 0; i < 12; i++ )
	{
		for( j = 0; j < 8; j++ )
		{
			sarvaRekha[i] += rekha[j][i];
			sarvaTrikona[i] += trikona[j][i];
			sarvaEkadhi[i] += ekadhi[j][i];

			psarvaRekha[j] += rekha[j][i];
			psarvaTrikona[j] += trikona[j][i];
			psarvaEkadhi[j] += ekadhi[j][i];
		}
	}
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcPinda 
**
******************************************************/
void AshtakavargaExpert::calcPinda()
{
	int rasi, planet, p;
	//const int k_rasimana[12] = { 7, 10, 8, 4, 10, 6, 7, 8, 9, 5, 11, 12 };
	const int k_rasimana[12] = { 7, 10, 8, 4, 10, 5, 7, 8, 9, 5, 11, 12 };
	const int k_grahamana[7] = { 5, 5, 8, 5, 10, 7, 5 };
	
	for( int i = 0; i < 12; i++ )
		rasiPinda[i] = grahaPinda[i] = yogaPinda[i] = 0;

	for( planet = 0; planet <= 8; planet++ )
	{
		for( rasi = 0; rasi < 12; rasi++ )
		{
			rasiPinda[planet] += ekadhi[planet][rasi] * k_rasimana[rasi];
		}
		for( p = 0; p < 7; p++ )
			{ grahaPinda[planet] += ekadhi[planet][view->getRasi(p)] * k_grahamana[p]; }

		yogaPinda[planet] = grahaPinda[planet] + rasiPinda[planet];
	}
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcTrikonaShodana 
**
******************************************************/
void AshtakavargaExpert::calcTrikonaShodana()
{
	wxChar minrekha;
	for( int i = 0; i < 8; i++ )
	{
		for( int j = 0; j < 4; j++ )
		{
			minrekha = Min( rekha[i][j], rekha[i][j+4] );
			minrekha = Min( rekha[i][j+8], minrekha );

			for( int k = 0; k < 3; k++ ) trikona[i][j+4*k] = rekha[i][j+4*k] - minrekha;
		}
	}
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcEkadhipatyaPair 
**
******************************************************/
void AshtakavargaExpert::calcEkadhipatyaPair( const int &rasi1, const int &rasi2 )
{
	int p;

	for( p = 0; p < 12; p++ )
	{
		/*********************************************************
		*** (0)
		** one rasi has got a number in Trikona Shodana
		** and the other ist bereft of
		** any number -> nothing to be done
		***
		*********************************************************/
		if ( ! trikona[p][rasi1] || ! trikona[p][rasi2])
		{
		}

		/*********************************************************
		*** (1)
		** both rasis without a planet and Trikona Shodana different
		** -> both are given the smaller number
		***
		*********************************************************/
		else if(( ! planetNumber[rasi1] ) && ( ! planetNumber[rasi2] ))
		{
			ekadhi[p][rasi1] = ekadhi[p][rasi2] = Min( trikona[p][rasi1], trikona[p][rasi2] );
		}

		/*********************************************************
		*** (2)
		** both rasis with planets -> no shodana
		***
		*********************************************************/
		else if(( planetNumber[rasi1] ) && ( planetNumber[rasi2] ))
		{
		}

		/*********************************************************
		*** (3)
		** one rasi is with a planet and a smaller trikona number
		** an the other is without a planet and a bigger trikona number
		** -> deduct the smaller number from the bigger number
		**  and the number of the rasi with planet should be kept unchanged
		***
		*********************************************************/
		else if(( planetNumber[rasi1] ) && ( ! planetNumber[rasi2] )
			&& ( trikona[p][rasi1] < trikona[p][rasi2]))
		{
			ekadhi[p][rasi2] -= trikona[p][rasi1];
		}

		else if(( planetNumber[rasi2] ) && ( ! planetNumber[rasi1] )
			&& ( trikona[p][rasi2] < trikona[p][rasi1]))
		{
			ekadhi[p][rasi1] -= trikona[p][rasi2];
		}

		/*********************************************************
		*** (4)
		** if the rasi with the planet has a bigger number than that
		** of the rasi without planet
		** -> shodana with the number of the rasi without planet
		** and the number of the rasi with planet schould be kept unchaged
		***
		*********************************************************/
		else if(( planetNumber[rasi1] ) && ( ! planetNumber[rasi2] )
			&& ( trikona[p][rasi1] > trikona[p][rasi2]))
		{
			ekadhi[p][rasi2] = 0;
		}
		else if(( planetNumber[rasi2] ) && ( ! planetNumber[rasi1] )
			&& ( trikona[p][rasi2] > trikona[p][rasi1]))
		{
			ekadhi[p][rasi1] = 0;
		}

		/*********************************************************
		*** (5)
		**  if both the rasis are without a planet an possess the
		**  same numbers
		**  -> shodana with both the numbers
		***
		*********************************************************/
		else if(( ! planetNumber[rasi1] ) && ( ! planetNumber[rasi2] )
			&& ( trikona[p][rasi1] == trikona[p][rasi2]))
		{
			ekadhi[p][rasi1] = ekadhi[p][rasi2] = 0;
		}

		/*********************************************************
		*** (6)
		** if one rasi is with an planet and the other is without
		** any planet
		** -> the number of the latter should be reduced to zero
		***
		*********************************************************/
		else if(( planetNumber[rasi1] ) && ( ! planetNumber[rasi2] ))
		{
			ekadhi[p][rasi2] = 0;
		}
		else if(( planetNumber[rasi2] ) && ( ! planetNumber[rasi1] ))
		{
			ekadhi[p][rasi1] = 0;
		}
	}
}

/*****************************************************
**
**   AshtakavargaExpert   ---   calcEkadhipatyaShodana 
**
******************************************************/
void AshtakavargaExpert::calcEkadhipatyaShodana()
{
	int i, j;
	
	// calculate number of planets in rasi
	for( i = ISUN; i <= ISATURN; i++ ) planetNumber[view->getRasi(i)]++;

	for( i = 0; i < 8; i++ ) 
		for( j = 0; j < 12; j++ ) 
			ekadhi[i][j] = trikona[i][j];

	calcEkadhipatyaPair( ARIES, SCORPIO );
	calcEkadhipatyaPair( TAURUS, LIBRA );
	calcEkadhipatyaPair( GEMINI, VIRGO );
	calcEkadhipatyaPair( SAGITTARIUS, PISCES );
	calcEkadhipatyaPair( CAPRICORNUS, AQUARIUS );
}


